"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateTemplate = exports.validateInstance = void 0;
var _ajv = _interopRequireDefault(require("ajv"));
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const ajv = new _ajv.default();
const staticAsset = {
  type: 'object',
  properties: {
    path: {
      type: 'string'
    },
    annotation: {
      type: 'string',
      nullable: true
    },
    data: {
      type: 'string',
      nullable: true
    }
  },
  required: ['path'],
  additionalProperties: false
};
const templateSchema = {
  type: 'object',
  properties: {
    name: {
      type: 'string'
    },
    version: {
      type: 'string'
    },
    displayName: {
      type: 'string',
      nullable: true
    },
    license: {
      type: 'string'
    },
    type: {
      type: 'string'
    },
    labels: {
      type: 'array',
      items: {
        type: 'string'
      },
      nullable: true
    },
    tags: {
      type: 'array',
      items: {
        type: 'string'
      },
      nullable: true
    },
    author: {
      type: 'string',
      nullable: true
    },
    description: {
      type: 'string',
      nullable: true
    },
    sourceUrl: {
      type: 'string',
      nullable: true
    },
    statics: {
      type: 'object',
      properties: {
        logo: {
          ...staticAsset,
          nullable: true
        },
        gallery: {
          type: 'array',
          items: staticAsset,
          nullable: true
        },
        darkModeLogo: {
          ...staticAsset,
          nullable: true
        },
        darkModeGallery: {
          type: 'array',
          items: staticAsset,
          nullable: true
        }
      },
      additionalProperties: false,
      nullable: true
    },
    components: {
      type: 'array',
      items: {
        type: 'object',
        properties: {
          name: {
            type: 'string'
          },
          version: {
            type: 'string'
          },
          data: {
            type: 'string',
            nullable: true
          }
        },
        required: ['name', 'version']
      }
    },
    assets: {
      type: 'object',
      properties: {
        savedObjects: {
          type: 'object',
          properties: {
            name: {
              type: 'string'
            },
            version: {
              type: 'string'
            },
            data: {
              type: 'string',
              nullable: true
            }
          },
          required: ['name', 'version'],
          nullable: true,
          additionalProperties: false
        },
        queries: {
          type: 'array',
          items: {
            type: 'object',
            properties: {
              name: {
                type: 'string'
              },
              version: {
                type: 'string'
              },
              language: {
                type: 'string'
              },
              data: {
                type: 'string',
                nullable: true
              }
            },
            required: ['name', 'version', 'language']
          },
          nullable: true
        }
      },
      additionalProperties: false
    },
    sampleData: {
      type: 'object',
      properties: {
        path: {
          type: 'string'
        },
        data: {
          type: 'string',
          nullable: true
        }
      },
      required: ['path'],
      additionalProperties: false,
      nullable: true
    }
  },
  required: ['name', 'version', 'license', 'type', 'components', 'assets'],
  additionalProperties: false
};
const instanceSchema = {
  type: 'object',
  properties: {
    name: {
      type: 'string'
    },
    templateName: {
      type: 'string'
    },
    dataSource: {
      type: 'string'
    },
    creationDate: {
      type: 'string'
    },
    assets: {
      type: 'array',
      items: {
        type: 'object',
        properties: {
          assetType: {
            type: 'string'
          },
          assetId: {
            type: 'string'
          },
          isDefaultAsset: {
            type: 'boolean'
          },
          description: {
            type: 'string'
          }
        },
        required: ['assetType', 'assetId', 'isDefaultAsset', 'description']
      }
    }
  },
  required: ['name', 'templateName', 'dataSource', 'creationDate', 'assets']
};
const templateValidator = ajv.compile(templateSchema);
const instanceValidator = ajv.compile(instanceSchema);

/**
 * Validates an integration template against a predefined schema using the AJV library.
 * Since AJV validators use side effects for errors,
 * this is a more conventional wrapper that simplifies calling.
 *
 * @param data The data to be validated as an IntegrationTemplate.
 * @return A Result indicating whether the validation was successful or not.
 *         If validation succeeds, returns an object with 'ok' set to true and the validated data.
 *         If validation fails, returns an object with 'ok' set to false and an Error object describing the validation error.
 */
const validateTemplate = data => {
  if (!templateValidator(data)) {
    return {
      ok: false,
      error: new Error(ajv.errorsText(templateValidator.errors))
    };
  }
  // We assume an invariant that the type of an integration is connected with its component.
  if (data.components.findIndex(x => x.name === data.type) < 0) {
    return {
      ok: false,
      error: new Error(`The integration type '${data.type}' must be included as a component`)
    };
  }
  return {
    ok: true,
    value: data
  };
};

/**
 * Validates an integration instance against a predefined schema using the AJV library.
 *
 * @param data The data to be validated as an IntegrationInstance.
 * @return A Result indicating whether the validation was successful or not.
 *         If validation succeeds, returns an object with 'ok' set to true and the validated data.
 *         If validation fails, returns an object with 'ok' set to false and an Error object describing the validation error.
 */
exports.validateTemplate = validateTemplate;
const validateInstance = data => {
  if (!instanceValidator(data)) {
    return {
      ok: false,
      error: new Error(ajv.errorsText(instanceValidator.errors))
    };
  }
  return {
    ok: true,
    value: data
  };
};
exports.validateInstance = validateInstance;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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