"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ObservabilityPlugin = void 0;
var _operators = require("rxjs/operators");
var _opensearch_observability_plugin = require("./adaptors/opensearch_observability_plugin");
var _ppl_plugin = require("./adaptors/ppl_plugin");
var _index = require("./routes/index");
var _observability_saved_object = require("./saved_objects/observability_saved_object");
var _ppl_parser = require("./parsers/ppl_parser");
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
class ObservabilityPlugin {
  constructor(initializerContext) {
    this.initializerContext = initializerContext;
    _defineProperty(this, "logger", void 0);
    this.logger = initializerContext.logger.get();
  }
  async setup(core, deps) {
    const {
      assistantDashboards
    } = deps;
    this.logger.debug('Observability: Setup');
    const router = core.http.createRouter();
    const config = await this.initializerContext.config.create().pipe((0, _operators.first)()).toPromise();
    const openSearchObservabilityClient = core.opensearch.legacy.createClient('opensearch_observability', {
      plugins: [_ppl_plugin.PPLPlugin, _opensearch_observability_plugin.OpenSearchObservabilityPlugin]
    });

    // @ts-ignore
    core.http.registerRouteHandlerContext('observability_plugin', (_context, _request) => {
      return {
        logger: this.logger,
        observabilityClient: openSearchObservabilityClient
      };
    });
    const obsPanelType = {
      name: 'observability-panel',
      hidden: false,
      namespaceType: 'single',
      mappings: {
        dynamic: false,
        properties: {
          title: {
            type: 'text'
          },
          description: {
            type: 'text'
          }
        }
      },
      management: {
        importableAndExportable: true,
        getInAppUrl() {
          return {
            path: `/app/management/observability/settings`,
            uiCapabilitiesPath: 'advancedSettings.show'
          };
        },
        getTitle(obj) {
          return `Observability Settings [${obj.id}]`;
        }
      },
      migrations: {
        '3.0.0': doc => ({
          ...doc,
          description: ''
        }),
        '3.0.1': doc => ({
          ...doc,
          description: 'Some Description Text'
        }),
        '3.0.2': doc => ({
          ...doc,
          dateCreated: parseInt(doc.dateCreated || '0', 10)
        })
      }
    };
    const integrationInstanceType = {
      name: 'integration-instance',
      hidden: false,
      namespaceType: 'single',
      management: {
        importableAndExportable: true,
        getInAppUrl(obj) {
          return {
            path: `/app/integrations#/installed/${obj.id}`,
            uiCapabilitiesPath: 'advancedSettings.show'
          };
        },
        getTitle(obj) {
          return obj.attributes.name;
        }
      },
      mappings: {
        dynamic: false,
        properties: {
          name: {
            type: 'text'
          },
          templateName: {
            type: 'text'
          },
          dataSource: {
            type: 'text'
          },
          creationDate: {
            type: 'date'
          },
          assets: {
            type: 'nested'
          }
        }
      }
    };
    const integrationTemplateType = {
      name: 'integration-template',
      hidden: false,
      namespaceType: 'single',
      management: {
        importableAndExportable: true,
        getInAppUrl(obj) {
          return {
            path: `/app/integrations#/available/${obj.attributes.name}`,
            uiCapabilitiesPath: 'advancedSettings.show'
          };
        },
        getTitle(obj) {
          var _obj$attributes$displ;
          return (_obj$attributes$displ = obj.attributes.displayName) !== null && _obj$attributes$displ !== void 0 ? _obj$attributes$displ : obj.attributes.name;
        }
      },
      mappings: {
        dynamic: false,
        properties: {
          name: {
            type: 'text'
          },
          version: {
            type: 'text'
          },
          displayName: {
            type: 'text'
          },
          license: {
            type: 'text'
          },
          type: {
            type: 'text'
          },
          labels: {
            type: 'text'
          },
          author: {
            type: 'text'
          },
          description: {
            type: 'text'
          },
          sourceUrl: {
            type: 'text'
          },
          statics: {
            type: 'nested'
          },
          components: {
            type: 'nested'
          },
          assets: {
            type: 'nested'
          },
          sampleData: {
            type: 'nested'
          }
        }
      }
    };
    core.savedObjects.registerType(obsPanelType);
    core.savedObjects.registerType(integrationInstanceType);
    core.savedObjects.registerType(integrationTemplateType);

    // Register server side APIs
    (0, _index.setupRoutes)({
      router,
      client: openSearchObservabilityClient,
      config
    });
    core.savedObjects.registerType(_observability_saved_object.visualizationSavedObject);
    core.savedObjects.registerType(_observability_saved_object.searchSavedObject);
    core.capabilities.registerProvider(() => ({
      observability: {
        show: true
      }
    }));
    assistantDashboards === null || assistantDashboards === void 0 || assistantDashboards.registerMessageParser(_ppl_parser.PPLParsers);
    return {};
  }
  start(_core) {
    this.logger.debug('Observability: Started');
    return {};
  }
  stop() {}
}
exports.ObservabilityPlugin = ObservabilityPlugin;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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