"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.handleWithCallback = void 0;
exports.registerIntegrationsRoute = registerIntegrationsRoute;
var _configSchema = require("@osd/config-schema");
var mime = _interopRequireWildcard(require("mime"));
var _sanitizeFilename = _interopRequireDefault(require("sanitize-filename"));
var _shared = require("../../../common/constants/shared");
var _integrations_manager = require("../../adaptors/integrations/integrations_manager");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && Object.prototype.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

/**
 * Handle an `OpenSearchDashboardsRequest` using the provided `callback` function.
 * This is a convenience method that handles common error handling and response formatting.
 * The callback must accept a `IntegrationsAdaptor` as its first argument.
 *
 * If the callback throws an error,
 * the `OpenSearchDashboardsResponse` will be formatted using the error's `statusCode` and `message` properties.
 * Otherwise, the callback's return value will be formatted in a JSON object under the `data` field.
 *
 * @param {IntegrationsAdaptor} adaptor The adaptor instance to use for making requests.
 * @param {OpenSearchDashboardsResponseFactory} response The factory to use for creating responses.
 * @callback callback A callback that will invoke a request on a provided adaptor.
 * @returns {Promise<OpenSearchDashboardsResponse>} An `OpenSearchDashboardsResponse` with the return data from the callback.
 */
const handleWithCallback = async (adaptor, response, callback) => {
  try {
    const data = await callback(adaptor);
    return response.ok({
      body: {
        data
      }
    });
  } catch (err) {
    console.error(`handleWithCallback: callback failed with error "${err.message}"`);
    return response.custom({
      statusCode: err.statusCode || 500,
      body: err.message
    });
  }
};
exports.handleWithCallback = handleWithCallback;
const getAdaptor = (context, _request) => {
  return new _integrations_manager.IntegrationsManager(context.core.savedObjects.client);
};
function registerIntegrationsRoute(router) {
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/repository`,
    validate: false
  }, async (context, request, response) => {
    const adaptor = getAdaptor(context, request);
    return handleWithCallback(adaptor, response, async a => a.getIntegrationTemplates());
  });
  router.post({
    path: `${_shared.INTEGRATIONS_BASE}/store/{templateName}`,
    validate: {
      params: _configSchema.schema.object({
        templateName: _configSchema.schema.string()
      }),
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        dataSource: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = getAdaptor(context, request);
    return handleWithCallback(adaptor, response, async a => {
      return a.loadIntegrationInstance(request.params.templateName, request.body.name, request.body.dataSource);
    });
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/repository/{name}`,
    validate: {
      params: _configSchema.schema.object({
        name: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = getAdaptor(context, request);
    return handleWithCallback(adaptor, response, async a => (await a.getIntegrationTemplates({
      name: request.params.name
    })).hits[0]);
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/repository/{id}/static/{path}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string(),
        path: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = getAdaptor(context, request);
    try {
      const requestPath = (0, _sanitizeFilename.default)(request.params.path);
      const result = await adaptor.getStatic(request.params.id, requestPath);
      return response.ok({
        headers: {
          'Content-Type': mime.getType(request.params.path)
        },
        body: result
      });
    } catch (err) {
      return response.custom({
        statusCode: err.statusCode ? err.statusCode : 500,
        body: err.message
      });
    }
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/repository/{id}/schema`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = getAdaptor(context, request);
    return handleWithCallback(adaptor, response, async a => a.getSchemas(request.params.id));
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/repository/{id}/assets`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = getAdaptor(context, request);
    return handleWithCallback(adaptor, response, async a => a.getAssets(request.params.id));
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/repository/{id}/data`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = getAdaptor(context, request);
    return handleWithCallback(adaptor, response, async a => a.getSampleData(request.params.id));
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/store`,
    validate: false
  }, async (context, request, response) => {
    const adaptor = getAdaptor(context, request);
    return handleWithCallback(adaptor, response, async a => a.getIntegrationInstances({}));
  });
  router.delete({
    path: `${_shared.INTEGRATIONS_BASE}/store/{id}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = getAdaptor(context, request);
    return handleWithCallback(adaptor, response, async a => a.deleteIntegrationInstance(request.params.id));
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/store/{id}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = getAdaptor(context, request);
    return handleWithCallback(adaptor, response, async a => a.getIntegrationInstance({
      id: request.params.id
    }));
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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