"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerQueryAssistRoutes = registerQueryAssistRoutes;
var _configSchema = require("@osd/config-schema");
var _errors = require("../../../../../src/core/server/opensearch/client/errors");
var _query_assist = require("../../../common/constants/query_assist");
var _generate_field_context = require("../../common/helpers/query_assist/generate_field_context");
var _agents = require("./utils/agents");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function registerQueryAssistRoutes(router, config) {
  const {
    ppl_agent_name: pplAgentName
  } = config.query_assist;
  const {
    response_summary_agent_name: responseSummaryAgentName,
    error_summary_agent_name: errorSummaryAgentName
  } = config.summarize;

  /**
   * Returns whether the PPL agent is configured.
   */
  router.get({
    path: _query_assist.QUERY_ASSIST_API.CONFIGURED,
    validate: {}
  }, async (context, request, response) => {
    const client = context.core.opensearch.client.asCurrentUser;
    try {
      // if the call does not throw any error, then the agent is properly configured
      await (0, _agents.searchAgentIdByName)(client, pplAgentName);
      return response.ok({
        body: {
          configured: true
        }
      });
    } catch (error) {
      return response.ok({
        body: {
          configured: false,
          error: error.message
        }
      });
    }
  });
  router.post({
    path: _query_assist.QUERY_ASSIST_API.GENERATE_PPL,
    validate: {
      body: _configSchema.schema.object({
        index: _configSchema.schema.string(),
        question: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    if (!pplAgentName) return response.custom({
      statusCode: 400,
      body: 'PPL agent name not found in opensearch_dashboards.yml. Expected observability.query_assist.ppl_agent_name'
    });
    const client = context.core.opensearch.client.asCurrentUser;
    try {
      const pplRequest = await (0, _agents.requestWithRetryAgentSearch)({
        client,
        agentName: pplAgentName,
        body: {
          parameters: {
            index: request.body.index,
            question: request.body.question
          }
        }
      });
      if (!pplRequest.body.inference_results[0].output[0].result) throw new Error('Generated PPL query not found.');
      const result = JSON.parse(pplRequest.body.inference_results[0].output[0].result);
      const ppl = result.ppl.replace(/[\r\n]/g, ' ').trim().replace(/ISNOTNULL/g, 'isnotnull') // https://github.com/opensearch-project/sql/issues/2431
      .replace(/`/g, '') // https://github.com/opensearch-project/dashboards-observability/issues/509, https://github.com/opensearch-project/dashboards-observability/issues/557
      .replace(/\bSPAN\(/g, 'span('); // https://github.com/opensearch-project/dashboards-observability/issues/759
      return response.ok({
        body: ppl
      });
    } catch (error) {
      var _error$body$error;
      // parse PPL query from error response if exists
      // TODO remove after https://github.com/opensearch-project/skills/issues/138
      if ((0, _errors.isResponseError)(error) && (_error$body$error = error.body.error) !== null && _error$body$error !== void 0 && _error$body$error.reason) {
        const pplMatch = error.body.error.reason.match(/execute ppl:(.+), get error:/);
        if (pplMatch[1]) return response.ok({
          body: pplMatch[1]
        });
      }
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: _query_assist.QUERY_ASSIST_API.SUMMARIZE,
    validate: {
      body: _configSchema.schema.object({
        index: _configSchema.schema.string(),
        question: _configSchema.schema.string(),
        query: _configSchema.schema.maybe(_configSchema.schema.string()),
        response: _configSchema.schema.string(),
        isError: _configSchema.schema.boolean()
      })
    }
  }, async (context, request, response) => {
    if (!responseSummaryAgentName || !errorSummaryAgentName) return response.custom({
      statusCode: 400,
      body: 'Summary agent name not found in opensearch_dashboards.yml. Expected observability.query_assist.response_summary_agent_name and observability.query_assist.error_summary_agent_name'
    });
    const client = context.core.opensearch.client.asCurrentUser;
    const {
      index,
      question,
      query,
      response: _response,
      isError
    } = request.body;
    const queryResponse = JSON.stringify(_response);
    let summaryRequest;
    try {
      var _summaryRequest$body$;
      if (!isError) {
        summaryRequest = await (0, _agents.requestWithRetryAgentSearch)({
          client,
          agentName: responseSummaryAgentName,
          body: {
            parameters: {
              index,
              question,
              query,
              response: queryResponse
            }
          }
        });
      } else {
        const [mappings, sampleDoc] = await Promise.all([client.indices.getMapping({
          index
        }), client.search({
          index,
          size: 1
        })]);
        const fields = (0, _generate_field_context.generateFieldContext)(mappings, sampleDoc);
        summaryRequest = await (0, _agents.requestWithRetryAgentSearch)({
          client,
          agentName: errorSummaryAgentName,
          body: {
            parameters: {
              index,
              question,
              query,
              response: queryResponse,
              fields
            }
          }
        });
      }
      const summary = summaryRequest.body.inference_results[0].output[0].result;
      if (!summary) throw new Error('Generated summary not found.');
      const suggestedQuestions = Array.from((((_summaryRequest$body$ = summaryRequest.body.inference_results[0].output[1]) === null || _summaryRequest$body$ === void 0 ? void 0 : _summaryRequest$body$.result) || '').matchAll(/<question>((.|[\r\n])+?)<\/question>/g)).map(m => m[1]);
      return response.ok({
        body: {
          summary,
          suggestedQuestions
        }
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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