"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.searchAgentIdByName = exports.requestWithRetryAgentSearch = exports.agentIdMap = void 0;
var _errors = require("../../../../../../src/core/server/opensearch/client/errors");
var _query_assist = require("../../../../common/constants/query_assist");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const AGENT_REQUEST_OPTIONS = {
  /**
   * It is time-consuming for LLM to generate final answer
   * Give it a large timeout window
   */
  requestTimeout: 5 * 60 * 1000,
  /**
   * Do not retry
   */
  maxRetries: 0
};
const agentIdMap = exports.agentIdMap = {};
const searchAgentIdByName = async (opensearchClient, name) => {
  try {
    const response = await opensearchClient.transport.request({
      method: 'GET',
      path: `${_query_assist.ML_COMMONS_API_PREFIX}/agents/_search`,
      body: {
        query: {
          term: {
            'name.keyword': name
          }
        },
        sort: {
          created_time: 'desc'
        }
      }
    });
    if (!response || typeof response.body.hits.total === 'number' && response.body.hits.total === 0 || response.body.hits.total.value === 0) {
      throw new Error('cannot find any agent by name: ' + name);
    }
    const id = response.body.hits.hits[0]._id;
    return id;
  } catch (error) {
    var _error$meta;
    const errorMessage = JSON.stringify((_error$meta = error.meta) === null || _error$meta === void 0 ? void 0 : _error$meta.body) || error;
    throw new Error(`search agent '${name}' failed, reason: ` + errorMessage);
  }
};
exports.searchAgentIdByName = searchAgentIdByName;
const requestWithRetryAgentSearch = async options => {
  const {
    client,
    agentName,
    shouldRetryAgentSearch = true,
    body
  } = options;
  let retry = shouldRetryAgentSearch;
  if (!agentIdMap[agentName]) {
    agentIdMap[agentName] = await searchAgentIdByName(client, agentName);
    retry = false;
  }
  return client.transport.request({
    method: 'POST',
    path: `${_query_assist.ML_COMMONS_API_PREFIX}/agents/${agentIdMap[agentName]}/_execute`,
    body
  }, AGENT_REQUEST_OPTIONS).catch(async error => {
    if (retry && (0, _errors.isResponseError)(error) && error.statusCode === 404) {
      agentIdMap[agentName] = await searchAgentIdByName(client, agentName);
      return requestWithRetryAgentSearch({
        ...options,
        shouldRetryAgentSearch: false
      });
    }
    return Promise.reject(error);
  });
};
exports.requestWithRetryAgentSearch = requestWithRetryAgentSearch;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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