import { EncodedSourceMap } from '@jridgewell/trace-mapping';
import { TemplateNode } from 'svelte/types/compiler/interfaces';
import { IExportedNames } from 'svelte2tsx';
import ts from 'typescript';
import { Position, Range, TextDocumentContentChangeEvent } from 'vscode-languageserver';
import { Document, DocumentMapper, IdentityMapper, TagInformation, FilePosition } from '../../lib/documents';
import { SvelteNode } from './svelte-ast-utils';
/**
 * An error which occurred while trying to parse/preprocess the svelte file contents.
 */
export interface ParserError {
    message: string;
    range: Range;
    code: number;
}
/**
 * Initial version of snapshots.
 */
export declare const INITIAL_VERSION = 0;
/**
 * A document snapshot suitable for the ts language service and the plugin.
 * Can be a real ts/js file or a virtual ts/js file which is generated from a Svelte file.
 */
export interface DocumentSnapshot extends ts.IScriptSnapshot, DocumentMapper {
    version: number;
    filePath: string;
    scriptKind: ts.ScriptKind;
    scriptInfo: TagInformation | null;
    positionAt(offset: number): Position;
    offsetAt(position: Position): number;
    /**
     * Convenience function for getText(0, getLength())
     */
    getFullText(): string;
}
/**
 * Options that apply to svelte files.
 */
export interface SvelteSnapshotOptions {
    transformOnTemplateError: boolean;
    typingsNamespace: string;
}
export declare namespace DocumentSnapshot {
    /**
     * Returns a svelte snapshot from a svelte document.
     * @param document the svelte document
     * @param options options that apply to the svelte document
     */
    function fromDocument(document: Document, options: SvelteSnapshotOptions): SvelteDocumentSnapshot;
    /**
     * Returns a svelte or ts/js snapshot from a file path, depending on the file contents.
     * @param filePath path to the js/ts/svelte file
     * @param createDocument function that is used to create a document in case it's a Svelte file
     * @param options options that apply in case it's a svelte file
     */
    function fromFilePath(filePath: string, createDocument: (filePath: string, text: string) => Document, options: SvelteSnapshotOptions, tsSystem: ts.System): SvelteDocumentSnapshot | JSOrTSDocumentSnapshot;
    /**
     * Returns a ts/js snapshot from a file path.
     * @param filePath path to the js/ts file
     * @param options options that apply in case it's a svelte file
     */
    function fromNonSvelteFilePath(filePath: string, tsSystem: ts.System): JSOrTSDocumentSnapshot;
    /**
     * Returns a svelte snapshot from a file path.
     * @param filePath path to the svelte file
     * @param createDocument function that is used to create a document
     * @param options options that apply in case it's a svelte file
     */
    function fromSvelteFilePath(filePath: string, createDocument: (filePath: string, text: string) => Document, options: SvelteSnapshotOptions): SvelteDocumentSnapshot;
}
/**
 * A svelte document snapshot suitable for the TS language service and the plugin.
 * It contains the generated code (Svelte->TS/JS) so the TS language service can understand it.
 */
export declare class SvelteDocumentSnapshot implements DocumentSnapshot {
    readonly parent: Document;
    readonly parserError: ParserError | null;
    readonly scriptKind: ts.ScriptKind;
    private readonly text;
    private readonly nrPrependedLines;
    private readonly exportedNames;
    private readonly tsxMap?;
    private readonly htmlAst?;
    private mapper?;
    private lineOffsets?;
    private url;
    version: number;
    constructor(parent: Document, parserError: ParserError | null, scriptKind: ts.ScriptKind, text: string, nrPrependedLines: number, exportedNames: IExportedNames, tsxMap?: EncodedSourceMap | undefined, htmlAst?: TemplateNode | undefined);
    get filePath(): string;
    get scriptInfo(): TagInformation | null;
    get moduleScriptInfo(): TagInformation | null;
    getOriginalText(range?: Range): string;
    getText(start: number, end: number): string;
    getLength(): number;
    getFullText(): string;
    getChangeRange(): undefined;
    positionAt(offset: number): Position;
    offsetAt(position: Position): number;
    getLineContainingOffset(offset: number): string;
    hasProp(name: string): boolean;
    svelteNodeAt(positionOrOffset: number | Position): SvelteNode | null;
    getOriginalPosition(pos: Position): Position;
    getGeneratedPosition(pos: Position): Position;
    isInGenerated(pos: Position): boolean;
    getURL(): string;
    private getLineOffsets;
    private getMapper;
    private initMapper;
}
/**
 * A js/ts document snapshot suitable for the ts language service and the plugin.
 * Since no mapping has to be done here, it also implements the mapper interface.
 * If it's a SvelteKit file (e.g. +page.ts), types will be auto-added if not explicitly typed.
 */
export declare class JSOrTSDocumentSnapshot extends IdentityMapper implements DocumentSnapshot {
    version: number;
    readonly filePath: string;
    private text;
    scriptKind: ts.ScriptKind;
    scriptInfo: null;
    originalText: string;
    kitFile: boolean;
    private lineOffsets?;
    private internalLineOffsets?;
    private addedCode;
    private paramsPath;
    private serverHooksPath;
    private clientHooksPath;
    constructor(version: number, filePath: string, text: string);
    getText(start: number, end: number): string;
    getLength(): number;
    getFullText(): string;
    getChangeRange(): undefined;
    positionAt(offset: number): Position;
    offsetAt(position: Position): number;
    getGeneratedPosition(originalPosition: Position): Position;
    getOriginalPosition(generatedPosition: Position): Position;
    update(changes: TextDocumentContentChangeEvent[]): void;
    protected getLineOffsets(): number[];
    private originalOffsetAt;
    private originalPositionAt;
    private getOriginalLineOffsets;
    private adjustText;
    private createSource;
}
export declare class DtsDocumentSnapshot extends JSOrTSDocumentSnapshot implements DocumentMapper {
    private tsSys;
    private traceMap;
    private mapperInitialized;
    constructor(version: number, filePath: string, text: string, tsSys: ts.System);
    getOriginalFilePosition(generatedPosition: Position): FilePosition;
    private initMapper;
    private initMapperByRawSourceMap;
    private logFailedToResolveSourceMap;
}
