#!/usr/bin/python
from __future__ import absolute_import, division, print_function

# Copyright: (c) 2022 Fortinet
# GNU General Public License v3.0+ (see LICENSE or https://www.gnu.org/licenses/gpl-3.0.txt)
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

__metaclass__ = type

ANSIBLE_METADATA = {
    "status": ["preview"],
    "supported_by": "community",
    "metadata_version": "1.1",
}

DOCUMENTATION = """
---
module: fortios_wireless_controller_wids_profile
short_description: Configure wireless intrusion detection system (WIDS) profiles in Fortinet's FortiOS and FortiGate.
description:
    - This module is able to configure a FortiGate or FortiOS (FOS) device by allowing the
      user to set and modify wireless_controller feature and wids_profile category.
      Examples include all parameters and values need to be adjusted to datasources before usage.
      Tested with FOS v6.0.0
version_added: "2.0.0"
author:
    - Link Zheng (@chillancezen)
    - Jie Xue (@JieX19)
    - Hongbin Lu (@fgtdev-hblu)
    - Frank Shen (@frankshen01)
    - Miguel Angel Munoz (@mamunozgonzalez)
    - Nicolas Thomas (@thomnico)
notes:
    - Legacy fortiosapi has been deprecated, httpapi is the preferred way to run playbooks

requirements:
    - ansible>=2.14
options:
    access_token:
        description:
            - Token-based authentication.
              Generated from GUI of Fortigate.
        type: str
        required: false
    enable_log:
        description:
            - Enable/Disable logging for task.
        type: bool
        required: false
        default: false
    vdom:
        description:
            - Virtual domain, among those defined previously. A vdom is a
              virtual instance of the FortiGate that can be configured and
              used as a different unit.
        type: str
        default: root
    member_path:
        type: str
        description:
            - Member attribute path to operate on.
            - Delimited by a slash character if there are more than one attribute.
            - Parameter marked with member_path is legitimate for doing member operation.
    member_state:
        type: str
        description:
            - Add or delete a member under specified attribute path.
            - When member_state is specified, the state option is ignored.
        choices:
            - 'present'
            - 'absent'

    state:
        description:
            - Indicates whether to create or remove the object.
        type: str
        required: true
        choices:
            - 'present'
            - 'absent'
    wireless_controller_wids_profile:
        description:
            - Configure wireless intrusion detection system (WIDS) profiles.
        default: null
        type: dict
        suboptions:
            ap_auto_suppress:
                description:
                    - Enable/disable on-wire rogue AP auto-suppression .
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            ap_bgscan_disable_day:
                description:
                    - Optionally turn off scanning for one or more days of the week. Separate the days with a space. By default, no days are set.
                type: str
                choices:
                    - 'sunday'
                    - 'monday'
                    - 'tuesday'
                    - 'wednesday'
                    - 'thursday'
                    - 'friday'
                    - 'saturday'
            ap_bgscan_disable_end:
                description:
                    - 'End time, using a 24-hour clock in the format of hh:mm, for disabling background scanning .'
                type: str
            ap_bgscan_disable_schedules:
                description:
                    - Firewall schedules for turning off FortiAP radio background scan. Background scan will be disabled when at least one of the schedules is
                       valid. Separate multiple schedule names with a space.
                type: list
                elements: dict
                suboptions:
                    name:
                        description:
                            - Schedule name. Source firewall.schedule.group.name firewall.schedule.recurring.name firewall.schedule.onetime.name.
                        required: true
                        type: str
            ap_bgscan_disable_start:
                description:
                    - 'Start time, using a 24-hour clock in the format of hh:mm, for disabling background scanning .'
                type: str
            ap_bgscan_duration:
                description:
                    - Listen time on scanning a channel (10 - 1000 msec).
                type: int
            ap_bgscan_idle:
                description:
                    - Wait time for channel inactivity before scanning this channel (0 - 1000 msec).
                type: int
            ap_bgscan_intv:
                description:
                    - Period between successive channel scans (1 - 600 sec).
                type: int
            ap_bgscan_period:
                description:
                    - Period between background scans (10 - 3600 sec).
                type: int
            ap_bgscan_report_intv:
                description:
                    - Period between background scan reports (15 - 600 sec).
                type: int
            ap_fgscan_report_intv:
                description:
                    - Period between foreground scan reports (15 - 600 sec).
                type: int
            ap_scan:
                description:
                    - Enable/disable rogue AP detection.
                type: str
                choices:
                    - 'disable'
                    - 'enable'
            ap_scan_channel_list_2G_5G:
                description:
                    - Selected ap scan channel list for 2.4G and 5G bands.
                type: list
                elements: dict
                suboptions:
                    chan:
                        description:
                            - Channel number.
                        required: true
                        type: str
            ap_scan_channel_list_6G:
                description:
                    - Selected ap scan channel list for 6G band.
                type: list
                elements: dict
                suboptions:
                    chan:
                        description:
                            - Channel 6g number.
                        required: true
                        type: str
            ap_scan_passive:
                description:
                    - Enable/disable passive scanning. Enable means do not send probe request on any channels .
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            ap_scan_threshold:
                description:
                    - Minimum signal level/threshold in dBm required for the AP to report detected rogue AP (-95 to -20).
                type: str
            asleap_attack:
                description:
                    - Enable/disable asleap attack detection .
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            assoc_flood_thresh:
                description:
                    - The threshold value for association frame flooding.
                type: int
            assoc_flood_time:
                description:
                    - Number of seconds after which a station is considered not connected.
                type: int
            assoc_frame_flood:
                description:
                    - Enable/disable association frame flooding detection .
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            auth_flood_thresh:
                description:
                    - The threshold value for authentication frame flooding.
                type: int
            auth_flood_time:
                description:
                    - Number of seconds after which a station is considered not connected.
                type: int
            auth_frame_flood:
                description:
                    - Enable/disable authentication frame flooding detection .
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            comment:
                description:
                    - Comment.
                type: str
            deauth_broadcast:
                description:
                    - Enable/disable broadcasting de-authentication detection .
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            deauth_unknown_src_thresh:
                description:
                    - 'Threshold value per second to deauth unknown src for DoS attack (0: no limit).'
                type: int
            eapol_fail_flood:
                description:
                    - Enable/disable EAPOL-Failure flooding (to AP) detection .
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            eapol_fail_intv:
                description:
                    - The detection interval for EAPOL-Failure flooding (1 - 3600 sec).
                type: int
            eapol_fail_thresh:
                description:
                    - The threshold value for EAPOL-Failure flooding in specified interval.
                type: int
            eapol_logoff_flood:
                description:
                    - Enable/disable EAPOL-Logoff flooding (to AP) detection .
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            eapol_logoff_intv:
                description:
                    - The detection interval for EAPOL-Logoff flooding (1 - 3600 sec).
                type: int
            eapol_logoff_thresh:
                description:
                    - The threshold value for EAPOL-Logoff flooding in specified interval.
                type: int
            eapol_pre_fail_flood:
                description:
                    - Enable/disable premature EAPOL-Failure flooding (to STA) detection .
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            eapol_pre_fail_intv:
                description:
                    - The detection interval for premature EAPOL-Failure flooding (1 - 3600 sec).
                type: int
            eapol_pre_fail_thresh:
                description:
                    - The threshold value for premature EAPOL-Failure flooding in specified interval.
                type: int
            eapol_pre_succ_flood:
                description:
                    - Enable/disable premature EAPOL-Success flooding (to STA) detection .
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            eapol_pre_succ_intv:
                description:
                    - The detection interval for premature EAPOL-Success flooding (1 - 3600 sec).
                type: int
            eapol_pre_succ_thresh:
                description:
                    - The threshold value for premature EAPOL-Success flooding in specified interval.
                type: int
            eapol_start_flood:
                description:
                    - Enable/disable EAPOL-Start flooding (to AP) detection .
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            eapol_start_intv:
                description:
                    - The detection interval for EAPOL-Start flooding (1 - 3600 sec).
                type: int
            eapol_start_thresh:
                description:
                    - The threshold value for EAPOL-Start flooding in specified interval.
                type: int
            eapol_succ_flood:
                description:
                    - Enable/disable EAPOL-Success flooding (to AP) detection .
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            eapol_succ_intv:
                description:
                    - The detection interval for EAPOL-Success flooding (1 - 3600 sec).
                type: int
            eapol_succ_thresh:
                description:
                    - The threshold value for EAPOL-Success flooding in specified interval.
                type: int
            invalid_mac_oui:
                description:
                    - Enable/disable invalid MAC OUI detection.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            long_duration_attack:
                description:
                    - Enable/disable long duration attack detection based on user configured threshold .
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            long_duration_thresh:
                description:
                    - Threshold value for long duration attack detection (1000 - 32767 usec).
                type: int
            name:
                description:
                    - WIDS profile name.
                required: true
                type: str
            null_ssid_probe_resp:
                description:
                    - Enable/disable null SSID probe response detection .
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            sensor_mode:
                description:
                    - Scan nearby WiFi stations .
                type: str
                choices:
                    - 'disable'
                    - 'foreign'
                    - 'both'
            spoofed_deauth:
                description:
                    - Enable/disable spoofed de-authentication attack detection .
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            weak_wep_iv:
                description:
                    - Enable/disable weak WEP IV (Initialization Vector) detection .
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            wireless_bridge:
                description:
                    - Enable/disable wireless bridge detection .
                type: str
                choices:
                    - 'enable'
                    - 'disable'
"""

EXAMPLES = """
- name: Configure wireless intrusion detection system (WIDS) profiles.
  fortinet.fortios.fortios_wireless_controller_wids_profile:
      vdom: "{{ vdom }}"
      state: "present"
      access_token: "<your_own_value>"
      wireless_controller_wids_profile:
          ap_auto_suppress: "enable"
          ap_bgscan_disable_day: "sunday"
          ap_bgscan_disable_end: "<your_own_value>"
          ap_bgscan_disable_schedules:
              -
                  name: "default_name_7 (source firewall.schedule.group.name firewall.schedule.recurring.name firewall.schedule.onetime.name)"
          ap_bgscan_disable_start: "<your_own_value>"
          ap_bgscan_duration: "30"
          ap_bgscan_idle: "20"
          ap_bgscan_intv: "3"
          ap_bgscan_period: "600"
          ap_bgscan_report_intv: "30"
          ap_fgscan_report_intv: "15"
          ap_scan: "disable"
          ap_scan_channel_list_2G_5G:
              -
                  chan: "<your_own_value>"
          ap_scan_channel_list_6G:
              -
                  chan: "<your_own_value>"
          ap_scan_passive: "enable"
          ap_scan_threshold: "<your_own_value>"
          asleap_attack: "enable"
          assoc_flood_thresh: "30"
          assoc_flood_time: "10"
          assoc_frame_flood: "enable"
          auth_flood_thresh: "30"
          auth_flood_time: "10"
          auth_frame_flood: "enable"
          comment: "Comment."
          deauth_broadcast: "enable"
          deauth_unknown_src_thresh: "10"
          eapol_fail_flood: "enable"
          eapol_fail_intv: "1"
          eapol_fail_thresh: "10"
          eapol_logoff_flood: "enable"
          eapol_logoff_intv: "1"
          eapol_logoff_thresh: "10"
          eapol_pre_fail_flood: "enable"
          eapol_pre_fail_intv: "1"
          eapol_pre_fail_thresh: "10"
          eapol_pre_succ_flood: "enable"
          eapol_pre_succ_intv: "1"
          eapol_pre_succ_thresh: "10"
          eapol_start_flood: "enable"
          eapol_start_intv: "1"
          eapol_start_thresh: "10"
          eapol_succ_flood: "enable"
          eapol_succ_intv: "1"
          eapol_succ_thresh: "10"
          invalid_mac_oui: "enable"
          long_duration_attack: "enable"
          long_duration_thresh: "8200"
          name: "default_name_53"
          null_ssid_probe_resp: "enable"
          sensor_mode: "disable"
          spoofed_deauth: "enable"
          weak_wep_iv: "enable"
          wireless_bridge: "enable"
"""

RETURN = """
build:
  description: Build number of the fortigate image
  returned: always
  type: str
  sample: '1547'
http_method:
  description: Last method used to provision the content into FortiGate
  returned: always
  type: str
  sample: 'PUT'
http_status:
  description: Last result given by FortiGate on last operation applied
  returned: always
  type: str
  sample: "200"
mkey:
  description: Master key (id) used in the last call to FortiGate
  returned: success
  type: str
  sample: "id"
name:
  description: Name of the table used to fulfill the request
  returned: always
  type: str
  sample: "urlfilter"
path:
  description: Path of the table used to fulfill the request
  returned: always
  type: str
  sample: "webfilter"
revision:
  description: Internal revision number
  returned: always
  type: str
  sample: "17.0.2.10658"
serial:
  description: Serial number of the unit
  returned: always
  type: str
  sample: "FGVMEVYYQT3AB5352"
status:
  description: Indication of the operation's result
  returned: always
  type: str
  sample: "success"
vdom:
  description: Virtual domain used
  returned: always
  type: str
  sample: "root"
version:
  description: Version of the FortiGate
  returned: always
  type: str
  sample: "v5.6.3"
"""
from ansible.module_utils.basic import AnsibleModule
from ansible.module_utils.connection import Connection
from ansible_collections.fortinet.fortios.plugins.module_utils.fortios.fortios import (
    FortiOSHandler,
)
from ansible_collections.fortinet.fortios.plugins.module_utils.fortios.fortios import (
    check_legacy_fortiosapi,
)
from ansible_collections.fortinet.fortios.plugins.module_utils.fortios.fortios import (
    schema_to_module_spec,
)
from ansible_collections.fortinet.fortios.plugins.module_utils.fortios.fortios import (
    check_schema_versioning,
)
from ansible_collections.fortinet.fortios.plugins.module_utils.fortimanager.common import (
    FAIL_SOCKET_MSG,
)
from ansible_collections.fortinet.fortios.plugins.module_utils.fortios.data_post_processor import (
    remove_invalid_fields,
)
from ansible_collections.fortinet.fortios.plugins.module_utils.fortios.comparison import (
    is_same_comparison,
)
from ansible_collections.fortinet.fortios.plugins.module_utils.fortios.comparison import (
    serialize,
)
from ansible_collections.fortinet.fortios.plugins.module_utils.fortios.comparison import (
    find_current_values,
)


def filter_wireless_controller_wids_profile_data(json):
    option_list = [
        "ap_auto_suppress",
        "ap_bgscan_disable_day",
        "ap_bgscan_disable_end",
        "ap_bgscan_disable_schedules",
        "ap_bgscan_disable_start",
        "ap_bgscan_duration",
        "ap_bgscan_idle",
        "ap_bgscan_intv",
        "ap_bgscan_period",
        "ap_bgscan_report_intv",
        "ap_fgscan_report_intv",
        "ap_scan",
        "ap_scan_channel_list_2G_5G",
        "ap_scan_channel_list_6G",
        "ap_scan_passive",
        "ap_scan_threshold",
        "asleap_attack",
        "assoc_flood_thresh",
        "assoc_flood_time",
        "assoc_frame_flood",
        "auth_flood_thresh",
        "auth_flood_time",
        "auth_frame_flood",
        "comment",
        "deauth_broadcast",
        "deauth_unknown_src_thresh",
        "eapol_fail_flood",
        "eapol_fail_intv",
        "eapol_fail_thresh",
        "eapol_logoff_flood",
        "eapol_logoff_intv",
        "eapol_logoff_thresh",
        "eapol_pre_fail_flood",
        "eapol_pre_fail_intv",
        "eapol_pre_fail_thresh",
        "eapol_pre_succ_flood",
        "eapol_pre_succ_intv",
        "eapol_pre_succ_thresh",
        "eapol_start_flood",
        "eapol_start_intv",
        "eapol_start_thresh",
        "eapol_succ_flood",
        "eapol_succ_intv",
        "eapol_succ_thresh",
        "invalid_mac_oui",
        "long_duration_attack",
        "long_duration_thresh",
        "name",
        "null_ssid_probe_resp",
        "sensor_mode",
        "spoofed_deauth",
        "weak_wep_iv",
        "wireless_bridge",
    ]

    json = remove_invalid_fields(json)
    dictionary = {}

    for attribute in option_list:
        if attribute in json and json[attribute] is not None:
            dictionary[attribute] = json[attribute]

    return dictionary


def underscore_to_hyphen(data):
    if isinstance(data, list):
        for i, elem in enumerate(data):
            data[i] = underscore_to_hyphen(elem)
    elif isinstance(data, dict):
        new_data = {}
        for k, v in data.items():
            new_data[k.replace("_", "-")] = underscore_to_hyphen(v)
        data = new_data

    return data


def wireless_controller_wids_profile(data, fos, check_mode=False):
    vdom = data["vdom"]

    state = data["state"]

    wireless_controller_wids_profile_data = data["wireless_controller_wids_profile"]
    filtered_data = underscore_to_hyphen(
        filter_wireless_controller_wids_profile_data(
            wireless_controller_wids_profile_data
        )
    )

    # check_mode starts from here
    if check_mode:
        diff = {
            "before": "",
            "after": filtered_data,
        }
        mkey = fos.get_mkey(
            "wireless-controller", "wids-profile", filtered_data, vdom=vdom
        )
        current_data = fos.get(
            "wireless-controller", "wids-profile", vdom=vdom, mkey=mkey
        )
        is_existed = (
            current_data
            and current_data.get("http_status") == 200
            and isinstance(current_data.get("results"), list)
            and len(current_data["results"]) > 0
        )

        # 2. if it exists and the state is 'present' then compare current settings with desired
        if state == "present" or state is True:
            if mkey is None:
                return False, True, filtered_data, diff

            # if mkey exists then compare each other
            # record exits and they're matched or not
            if is_existed:
                is_same = is_same_comparison(
                    serialize(current_data["results"][0]), serialize(filtered_data)
                )

                current_values = find_current_values(
                    current_data["results"][0], filtered_data
                )

                return (
                    False,
                    not is_same,
                    filtered_data,
                    {"before": current_values, "after": filtered_data},
                )

            # record does not exist
            return False, True, filtered_data, diff

        if state == "absent":
            if mkey is None:
                return (
                    False,
                    False,
                    filtered_data,
                    {"before": current_data["results"][0], "after": ""},
                )

            if is_existed:
                return (
                    False,
                    True,
                    filtered_data,
                    {"before": current_data["results"][0], "after": ""},
                )
            return False, False, filtered_data, {}

        return True, False, {"reason: ": "Must provide state parameter"}, {}

    if state == "present" or state is True:
        return fos.set(
            "wireless-controller", "wids-profile", data=filtered_data, vdom=vdom
        )

    elif state == "absent":
        return fos.delete(
            "wireless-controller", "wids-profile", mkey=filtered_data["name"], vdom=vdom
        )
    else:
        fos._module.fail_json(msg="state must be present or absent!")


def is_successful_status(resp):
    return (
        "status" in resp
        and resp["status"] == "success"
        or "http_status" in resp
        and resp["http_status"] == 200
        or "http_method" in resp
        and resp["http_method"] == "DELETE"
        and resp["http_status"] == 404
    )


def fortios_wireless_controller(data, fos, check_mode):
    fos.do_member_operation("wireless-controller", "wids-profile")
    if data["wireless_controller_wids_profile"]:
        resp = wireless_controller_wids_profile(data, fos, check_mode)
    else:
        fos._module.fail_json(
            msg="missing task body: %s" % ("wireless_controller_wids_profile")
        )
    if isinstance(resp, tuple) and len(resp) == 4:
        return resp
    return (
        not is_successful_status(resp),
        is_successful_status(resp)
        and (resp["revision_changed"] if "revision_changed" in resp else True),
        resp,
        {},
    )


versioned_schema = {
    "type": "list",
    "elements": "dict",
    "children": {
        "name": {"v_range": [["v6.0.0", ""]], "type": "string", "required": True},
        "comment": {"v_range": [["v6.0.0", ""]], "type": "string"},
        "sensor_mode": {
            "v_range": [["v6.0.0", ""]],
            "type": "string",
            "options": [{"value": "disable"}, {"value": "foreign"}, {"value": "both"}],
        },
        "ap_scan": {
            "v_range": [["v6.0.0", ""]],
            "type": "string",
            "options": [{"value": "disable"}, {"value": "enable"}],
        },
        "ap_scan_channel_list_2G_5G": {
            "type": "list",
            "elements": "dict",
            "children": {
                "chan": {
                    "v_range": [["v7.4.1", ""]],
                    "type": "string",
                    "required": True,
                }
            },
            "v_range": [["v7.4.1", ""]],
        },
        "ap_scan_channel_list_6G": {
            "type": "list",
            "elements": "dict",
            "children": {
                "chan": {
                    "v_range": [["v7.4.1", ""]],
                    "type": "string",
                    "required": True,
                }
            },
            "v_range": [["v7.4.1", ""]],
        },
        "ap_bgscan_period": {"v_range": [["v6.0.0", ""]], "type": "integer"},
        "ap_bgscan_intv": {"v_range": [["v6.0.0", ""]], "type": "integer"},
        "ap_bgscan_duration": {"v_range": [["v6.0.0", ""]], "type": "integer"},
        "ap_bgscan_idle": {"v_range": [["v6.0.0", ""]], "type": "integer"},
        "ap_bgscan_report_intv": {"v_range": [["v6.0.0", ""]], "type": "integer"},
        "ap_bgscan_disable_schedules": {
            "type": "list",
            "elements": "dict",
            "children": {
                "name": {
                    "v_range": [["v6.2.0", ""]],
                    "type": "string",
                    "required": True,
                }
            },
            "v_range": [["v6.2.0", ""]],
        },
        "ap_fgscan_report_intv": {"v_range": [["v6.0.0", ""]], "type": "integer"},
        "ap_scan_passive": {
            "v_range": [["v6.0.0", ""]],
            "type": "string",
            "options": [{"value": "enable"}, {"value": "disable"}],
        },
        "ap_scan_threshold": {"v_range": [["v6.2.0", ""]], "type": "string"},
        "ap_auto_suppress": {
            "v_range": [["v6.0.0", ""]],
            "type": "string",
            "options": [{"value": "enable"}, {"value": "disable"}],
        },
        "wireless_bridge": {
            "v_range": [["v6.0.0", ""]],
            "type": "string",
            "options": [{"value": "enable"}, {"value": "disable"}],
        },
        "deauth_broadcast": {
            "v_range": [["v6.0.0", ""]],
            "type": "string",
            "options": [{"value": "enable"}, {"value": "disable"}],
        },
        "null_ssid_probe_resp": {
            "v_range": [["v6.0.0", ""]],
            "type": "string",
            "options": [{"value": "enable"}, {"value": "disable"}],
        },
        "long_duration_attack": {
            "v_range": [["v6.0.0", ""]],
            "type": "string",
            "options": [{"value": "enable"}, {"value": "disable"}],
        },
        "long_duration_thresh": {"v_range": [["v6.0.0", ""]], "type": "integer"},
        "invalid_mac_oui": {
            "v_range": [["v6.0.0", ""]],
            "type": "string",
            "options": [{"value": "enable"}, {"value": "disable"}],
        },
        "weak_wep_iv": {
            "v_range": [["v6.0.0", ""]],
            "type": "string",
            "options": [{"value": "enable"}, {"value": "disable"}],
        },
        "auth_frame_flood": {
            "v_range": [["v6.0.0", ""]],
            "type": "string",
            "options": [{"value": "enable"}, {"value": "disable"}],
        },
        "auth_flood_time": {"v_range": [["v6.0.0", ""]], "type": "integer"},
        "auth_flood_thresh": {"v_range": [["v6.0.0", ""]], "type": "integer"},
        "assoc_frame_flood": {
            "v_range": [["v6.0.0", ""]],
            "type": "string",
            "options": [{"value": "enable"}, {"value": "disable"}],
        },
        "assoc_flood_time": {"v_range": [["v6.0.0", ""]], "type": "integer"},
        "assoc_flood_thresh": {"v_range": [["v6.0.0", ""]], "type": "integer"},
        "spoofed_deauth": {
            "v_range": [["v6.0.0", ""]],
            "type": "string",
            "options": [{"value": "enable"}, {"value": "disable"}],
        },
        "asleap_attack": {
            "v_range": [["v6.0.0", ""]],
            "type": "string",
            "options": [{"value": "enable"}, {"value": "disable"}],
        },
        "eapol_start_flood": {
            "v_range": [["v6.0.0", ""]],
            "type": "string",
            "options": [{"value": "enable"}, {"value": "disable"}],
        },
        "eapol_start_thresh": {"v_range": [["v6.0.0", ""]], "type": "integer"},
        "eapol_start_intv": {"v_range": [["v6.0.0", ""]], "type": "integer"},
        "eapol_logoff_flood": {
            "v_range": [["v6.0.0", ""]],
            "type": "string",
            "options": [{"value": "enable"}, {"value": "disable"}],
        },
        "eapol_logoff_thresh": {"v_range": [["v6.0.0", ""]], "type": "integer"},
        "eapol_logoff_intv": {"v_range": [["v6.0.0", ""]], "type": "integer"},
        "eapol_succ_flood": {
            "v_range": [["v6.0.0", ""]],
            "type": "string",
            "options": [{"value": "enable"}, {"value": "disable"}],
        },
        "eapol_succ_thresh": {"v_range": [["v6.0.0", ""]], "type": "integer"},
        "eapol_succ_intv": {"v_range": [["v6.0.0", ""]], "type": "integer"},
        "eapol_fail_flood": {
            "v_range": [["v6.0.0", ""]],
            "type": "string",
            "options": [{"value": "enable"}, {"value": "disable"}],
        },
        "eapol_fail_thresh": {"v_range": [["v6.0.0", ""]], "type": "integer"},
        "eapol_fail_intv": {"v_range": [["v6.0.0", ""]], "type": "integer"},
        "eapol_pre_succ_flood": {
            "v_range": [["v6.0.0", ""]],
            "type": "string",
            "options": [{"value": "enable"}, {"value": "disable"}],
        },
        "eapol_pre_succ_thresh": {"v_range": [["v6.0.0", ""]], "type": "integer"},
        "eapol_pre_succ_intv": {"v_range": [["v6.0.0", ""]], "type": "integer"},
        "eapol_pre_fail_flood": {
            "v_range": [["v6.0.0", ""]],
            "type": "string",
            "options": [{"value": "enable"}, {"value": "disable"}],
        },
        "eapol_pre_fail_thresh": {"v_range": [["v6.0.0", ""]], "type": "integer"},
        "eapol_pre_fail_intv": {"v_range": [["v6.0.0", ""]], "type": "integer"},
        "deauth_unknown_src_thresh": {"v_range": [["v6.0.0", ""]], "type": "integer"},
        "ap_bgscan_disable_day": {
            "v_range": [["v6.0.0", "v6.0.11"]],
            "type": "string",
            "options": [
                {"value": "sunday"},
                {"value": "monday"},
                {"value": "tuesday"},
                {"value": "wednesday"},
                {"value": "thursday"},
                {"value": "friday"},
                {"value": "saturday"},
            ],
        },
        "ap_bgscan_disable_start": {
            "v_range": [["v6.0.0", "v6.0.11"]],
            "type": "string",
        },
        "ap_bgscan_disable_end": {"v_range": [["v6.0.0", "v6.0.11"]], "type": "string"},
    },
    "v_range": [["v6.0.0", ""]],
}


def main():
    module_spec = schema_to_module_spec(versioned_schema)
    mkeyname = "name"
    fields = {
        "access_token": {"required": False, "type": "str", "no_log": True},
        "enable_log": {"required": False, "type": "bool", "default": False},
        "vdom": {"required": False, "type": "str", "default": "root"},
        "member_path": {"required": False, "type": "str"},
        "member_state": {
            "type": "str",
            "required": False,
            "choices": ["present", "absent"],
        },
        "state": {"required": True, "type": "str", "choices": ["present", "absent"]},
        "wireless_controller_wids_profile": {
            "required": False,
            "type": "dict",
            "default": None,
            "options": {},
        },
    }
    for attribute_name in module_spec["options"]:
        fields["wireless_controller_wids_profile"]["options"][
            attribute_name
        ] = module_spec["options"][attribute_name]
        if mkeyname and mkeyname == attribute_name:
            fields["wireless_controller_wids_profile"]["options"][attribute_name][
                "required"
            ] = True

    module = AnsibleModule(argument_spec=fields, supports_check_mode=True)
    check_legacy_fortiosapi(module)

    is_error = False
    has_changed = False
    result = None
    diff = None

    versions_check_result = None
    if module._socket_path:
        connection = Connection(module._socket_path)
        if "access_token" in module.params:
            connection.set_option("access_token", module.params["access_token"])

        if "enable_log" in module.params:
            connection.set_option("enable_log", module.params["enable_log"])
        else:
            connection.set_option("enable_log", False)
        fos = FortiOSHandler(connection, module, mkeyname)
        versions_check_result = check_schema_versioning(
            fos, versioned_schema, "wireless_controller_wids_profile"
        )

        is_error, has_changed, result, diff = fortios_wireless_controller(
            module.params, fos, module.check_mode
        )

    else:
        module.fail_json(**FAIL_SOCKET_MSG)

    if versions_check_result and versions_check_result["matched"] is False:
        module.warn(
            "Ansible has detected version mismatch between FortOS system and your playbook, see more details by specifying option -vvv"
        )

    if not is_error:
        if versions_check_result and versions_check_result["matched"] is False:
            module.exit_json(
                changed=has_changed,
                version_check_warning=versions_check_result,
                meta=result,
                diff=diff,
            )
        else:
            module.exit_json(changed=has_changed, meta=result, diff=diff)
    else:
        if versions_check_result and versions_check_result["matched"] is False:
            module.fail_json(
                msg="Error in repo",
                version_check_warning=versions_check_result,
                meta=result,
            )
        else:
            module.fail_json(msg="Error in repo", meta=result)


if __name__ == "__main__":
    main()
