"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
exports.registerOpenSearchRoutes = registerOpenSearchRoutes;
var _lodash = require("lodash");
var _adHelpers = require("./utils/adHelpers");
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 *
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */
function registerOpenSearchRoutes(apiRouter, opensearchService) {
  apiRouter.get('/_indices', opensearchService.getIndices);
  apiRouter.get('/_aliases', opensearchService.getAliases);
  apiRouter.get('/_mappings', opensearchService.getMapping);
  apiRouter.post('/_search', opensearchService.executeSearch);
  apiRouter.put('/create_index', opensearchService.createIndex);
  apiRouter.post('/bulk', opensearchService.bulk);
  apiRouter.post('/delete_index', opensearchService.deleteIndex);
}
class OpenSearchService {
  constructor(client) {
    _defineProperty(this, "client", void 0);
    _defineProperty(this, "executeSearch", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          index,
          query,
          size = 0,
          sort = undefined,
          collapse = undefined,
          aggs = undefined,
          rawQuery = undefined
        } = request.body;
        const requestBody = rawQuery ? rawQuery : {
          query: query,
          ...(sort !== undefined && {
            sort: sort
          }),
          ...(collapse !== undefined && {
            collapse: collapse
          }),
          ...(aggs !== undefined && {
            aggs: aggs
          })
        };
        const params = {
          index,
          size,
          body: requestBody
        };
        const results = await this.client.asScoped(request).callAsCurrentUser('search', params);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: results
          }
        });
      } catch (err) {
        console.error('Anomaly detector - Unable to execute search', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getIndices", async (context, request, opensearchDashboardsResponse) => {
      const {
        index
      } = request.query;
      try {
        const response = await this.client.asScoped(request).callAsCurrentUser('cat.indices', {
          index,
          format: 'json',
          h: 'health,index'
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              indices: response
            }
          }
        });
      } catch (err) {
        // In case no matching indices is found it throws an error.
        if (err.statusCode === 404 && (0, _lodash.get)(err, 'body.error.type', '') === 'index_not_found_exception') {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                indices: []
              }
            }
          });
        }
        console.log('Anomaly detector - Unable to get indices', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getAliases", async (context, request, opensearchDashboardsResponse) => {
      const {
        alias
      } = request.query;
      try {
        const response = await this.client.asScoped(request).callAsCurrentUser('cat.aliases', {
          alias,
          format: 'json',
          h: 'alias,index'
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              aliases: response
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get aliases', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "createIndex", async (context, request, opensearchDashboardsResponse) => {
      //@ts-ignore
      const index = request.body.index;
      //@ts-ignore
      const body = request.body.body;
      try {
        await this.client.asScoped(request).callAsCurrentUser('indices.create', {
          index: index,
          body: body
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to create index', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
      try {
        const response = await this.client.asScoped(request).callAsCurrentUser('cat.indices', {
          index,
          format: 'json',
          h: 'health,index'
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              indices: response
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get indices', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "bulk", async (context, request, opensearchDashboardsResponse) => {
      const body = request.body;
      try {
        const response = await this.client.asScoped(request).callAsCurrentUser('bulk', {
          body: body
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              response
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to perform bulk action', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "deleteIndex", async (context, request, opensearchDashboardsResponse) => {
      const index = request.query;
      try {
        await this.client.asScoped(request).callAsCurrentUser('indices.delete', {
          index: index
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to perform delete index action', err);
        // Ignore the error if it's an index_not_found_exception
        if (!(0, _adHelpers.isIndexNotFoundError)(err)) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: false,
              error: (0, _adHelpers.getErrorMessage)(err)
            }
          });
        }
      }
      try {
        const response = await this.client.asScoped(request).callAsCurrentUser('cat.indices', {
          index,
          format: 'json',
          h: 'health,index'
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              indices: response
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get indices', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getMapping", async (context, request, opensearchDashboardsResponse) => {
      const {
        index
      } = request.query;
      try {
        const response = await this.client.asScoped(request).callAsCurrentUser('indices.getMapping', {
          index
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              mappings: response
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get mappings', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    this.client = client;
  }
}
exports.default = OpenSearchService;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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