"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.loadSampleData = void 0;
var _moment = _interopRequireDefault(require("moment"));
var _readline = _interopRequireDefault(require("readline"));
var _fs = _interopRequireDefault(require("fs"));
var _zlib = require("zlib");
var _lodash = require("lodash");
var _helpers = require("../../utils/helpers");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 *
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

//@ts-ignore

const BULK_INSERT_SIZE = 500;
const loadSampleData = (filePath, indexName, client, request) => {
  return new Promise((resolve, reject) => {
    let count = 0;
    let docs = [];
    let isPaused = false;
    let offset = 0;
    const startTime = (0, _moment.default)(new Date().getTime()).subtract({
      days: 7
    });

    // Create the read stream for the file. Set a smaller buffer size here to prevent it from
    // getting too large, to prevent inserting too many docs at once into the index.
    const readStream = _fs.default.createReadStream(filePath, {
      highWaterMark: 1024 * 4
    });
    const lineStream = _readline.default.createInterface({
      input: readStream.pipe((0, _zlib.createUnzip)())
    });

    // This is only ran when the end of lineStream closes normally. It is used to
    // bulk insert the final batch of lines that are < BULK_INSERT_SIZE
    const onClose = async () => {
      if (docs.length > 0) {
        try {
          await bulkInsert(docs);
        } catch (err) {
          reject(err);
          return;
        }
      }
      resolve(count);
    };
    lineStream.on('close', onClose);
    lineStream.on('pause', async () => {
      isPaused = true;
    });
    lineStream.on('resume', async () => {
      isPaused = false;
    });
    lineStream.on('line', async doc => {
      // for the initial doc, get the timestamp to properly set an offset
      if (count === 0) {
        const initialTime = (0, _moment.default)(JSON.parse(doc).timestamp);
        offset = startTime.diff(initialTime);
      }
      count++;
      docs.push(doc);

      // If not currently paused: pause the stream to prevent concurrent bulk inserts
      // on the cluster which could cause performance issues.
      // Also, empty the current docs[] before performing the bulk insert to prevent
      // buffered docs from being dropped.
      if (docs.length >= BULK_INSERT_SIZE && !isPaused) {
        lineStream.pause();

        // save the docs to insert, and empty out current docs list
        const docsToInsert = docs.slice();
        docs = [];
        try {
          await bulkInsert(docsToInsert);
          lineStream.resume();
        } catch (err) {
          lineStream.removeListener('close', onClose);
          lineStream.close();
          reject(err);
        }
      }
    });
    const bulkInsert = async docs => {
      try {
        const bulkBody = prepareBody(docs, offset);
        const resp = await client.asScoped(request).callAsCurrentUser('bulk', {
          body: bulkBody
        });
        if (resp.errors) {
          const errorItems = resp.items;
          const firstErrorReason = (0, _lodash.isEmpty)(errorItems) ? 'Error while bulk inserting. Please try again.' : (0, _helpers.prettifyErrorMessage)(errorItems[0].index.error.reason);
          return Promise.reject(new Error(firstErrorReason));
        }
      } catch (err) {
        console.log('Error while bulk inserting. ', err);
        return Promise.reject(new Error('Error while bulk inserting. Please try again.'));
      }
    };
    const prepareBody = (docs, offset) => {
      const bulkBody = [];
      let docIdCount = count - docs.length;
      docs.forEach(doc => {
        bulkBody.push(getDocDetails(docIdCount));
        bulkBody.push(updateTimestamp(doc, offset));
        docIdCount++;
      });
      return bulkBody;
    };
    const updateTimestamp = (doc, offset) => {
      let docAsJSON = JSON.parse(doc);
      const updatedTimestamp = docAsJSON.timestamp + offset;
      docAsJSON.timestamp = updatedTimestamp;
      return docAsJSON;
    };
    const getDocDetails = docIdCount => {
      return `{ "index": { "_index": "${indexName}", "_id": ${docIdCount} } }`;
    };
  });
};
exports.loadSampleData = loadSampleData;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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