"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.backendToUiReportsList = exports.backendToUiReportDefinitionsList = exports.backendToUiReportDefinition = exports.backendToUiReport = void 0;
var _model = require("../../../model");
var _backendModel = require("../../../model/backendModel");
var _constants = require("../constants");
var _moment = _interopRequireDefault(require("moment"));
var _url = require("url");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const backendToUiReport = (backendReportInstance, serverBasePath) => {
  const {
    inContextDownloadUrlPath,
    beginTimeMs,
    endTimeMs,
    tenant,
    status,
    lastUpdatedTimeMs: reportLastUpdatedTimeMs,
    createdTimeMs: reportCreatedTimeMs,
    reportDefinitionDetails: backendReportDefinitionDetails
  } = backendReportInstance;
  const {
    reportDefinition: {
      source: {
        type: sourceType,
        id: sourceId
      },
      delivery
    }
  } = backendReportDefinitionDetails;
  const baseUrl = getBaseUrl(sourceType, sourceId);
  let report = {
    // inContextDownloadUrlPath may not exist for report instance created from scheduled job
    query_url: inContextDownloadUrlPath || getUiQueryUrl(baseUrl, beginTimeMs, endTimeMs, tenant),
    time_from: beginTimeMs,
    time_to: endTimeMs,
    last_updated: reportLastUpdatedTimeMs,
    time_created: reportCreatedTimeMs,
    state: getUiReportState(status, delivery),
    report_definition: backendToUiReportDefinition(backendReportDefinitionDetails, serverBasePath)
  };

  // Add severbasePath back to query_url
  report.query_url = serverBasePath + report.query_url;
  return report;
};
exports.backendToUiReport = backendToUiReport;
const backendToUiReportsList = (backendReportsList, serverBasePath) => {
  const res = backendReportsList.map(backendReport => {
    return {
      _id: backendReport.id,
      _source: backendToUiReport(backendReport, serverBasePath)
    };
  });
  return res;
};
exports.backendToUiReportsList = backendToUiReportsList;
const backendToUiReportDefinition = (backendReportDefinitionDetails, serverBasePath) => {
  const {
    lastUpdatedTimeMs,
    createdTimeMs,
    reportDefinition: {
      name,
      isEnabled,
      source: {
        type: sourceType,
        description,
        id: sourceId,
        origin
      },
      format: {
        fileFormat,
        duration,
        header,
        footer,
        limit
      },
      trigger: {
        triggerType,
        schedule
      },
      delivery
    }
  } = backendReportDefinitionDetails;
  const baseUrl = getBaseUrl(sourceType, sourceId);
  const reportSource = getUiReportSource(sourceType);
  let uiReportDefinition = {
    report_params: {
      report_name: name,
      report_source: reportSource,
      description: description,
      core_params: reportSource === _constants.REPORT_TYPE.savedSearch ? getDataReportCoreParams(limit, sourceId, fileFormat, duration, baseUrl, origin) : getVisualReportCoreParams(fileFormat, header, footer, duration, baseUrl, origin)
    },
    trigger: getUiTriggerParams(triggerType, schedule, createdTimeMs, isEnabled),
    delivery: getUiDeliveryParams(delivery),
    //TODO:
    time_created: createdTimeMs,
    last_updated: lastUpdatedTimeMs,
    status: getUiReportDefinitionStatus(isEnabled)
  };
  // validate to assign default values to some fields for UI model
  uiReportDefinition = _model.reportDefinitionSchema.validate(uiReportDefinition);
  uiReportDefinition.report_params.core_params.base_url = serverBasePath + uiReportDefinition.report_params.core_params.base_url;
  return uiReportDefinition;
};
exports.backendToUiReportDefinition = backendToUiReportDefinition;
const backendToUiReportDefinitionsList = (backendReportDefinitionDetailsList, serverBasePath) => {
  const res = backendReportDefinitionDetailsList.map(backendReportDefinitionDetails => {
    return {
      _id: backendReportDefinitionDetails.id,
      _source: {
        // TODO: this property can be removed, but need UI changes as well
        report_definition: backendToUiReportDefinition(backendReportDefinitionDetails, serverBasePath)
      }
    };
  });
  return res;
};
exports.backendToUiReportDefinitionsList = backendToUiReportDefinitionsList;
const getVisualReportCoreParams = (fileFormat, header = '', footer = '', duration, baseUrl, origin) => {
  let res = {
    base_url: baseUrl,
    report_format: getUiReportFormat(fileFormat),
    header: header,
    footer: footer,
    time_duration: duration,
    origin: origin
  };
  return res;
};

// queryUrl = baseUrl + time range
const getUiQueryUrl = (baseUrl, beginTimeMs, endTimeMs, tenant) => {
  const timeFrom = (0, _moment.default)(beginTimeMs).toISOString();
  const timeTo = (0, _moment.default)(endTimeMs).toISOString();
  let queryUrl = `${baseUrl}?_g=(time:(from:'${timeFrom}',to:'${timeTo}'))`;
  if (tenant !== undefined) {
    if (tenant === '') {
      tenant = 'global';
    } else if (tenant === '__user__') {
      tenant = 'private';
    }
    queryUrl = addTenantToURL(queryUrl, tenant);
  }
  return queryUrl;
};
const getBaseUrl = (sourceType, sourceId) => {
  //TODO: AES domain has different prefix, need figure out a general solution
  const baseUrl = `${_backendModel.URL_PREFIX_DICT[sourceType]}${sourceId}`;
  return baseUrl;
};
const getDataReportCoreParams = (limit = _constants.DEFAULT_MAX_SIZE, sourceId, fileFormat, duration, baseUrl, origin) => {
  let res = {
    base_url: baseUrl,
    report_format: getUiReportFormat(fileFormat),
    limit: limit,
    time_duration: duration,
    saved_search_id: sourceId,
    origin: origin
  };
  return res;
};
const getUiScheduleParams = (schedule, createdTimeMs, isEnabled) => {
  let res = {
    trigger_params: {
      enabled_time: createdTimeMs,
      enabled: isEnabled,
      schedule_type: schedule && 'cron' in schedule ? _constants.SCHEDULE_TYPE.cron : _constants.SCHEDULE_TYPE.recurring,
      schedule: schedule
    }
  };
  return res;
};
const getUiTriggerType = backendField => {
  let res;
  for (let [ui, backendFieldList] of Object.entries(_backendModel.TRIGGER_TYPE_DICT)) {
    for (let item of backendFieldList) {
      if (item === backendField) {
        res = ui;
      }
    }
  }
  return res;
};
const getUiReportFormat = backendField => {
  let res;
  for (let [ui, backend] of Object.entries(_backendModel.REPORT_FORMAT_DICT)) {
    if (backend === backendField) {
      res = ui;
    }
  }
  return res;
};
const getUiReportState = (status, delivery) => {
  let res;
  for (let [ui, backend] of Object.entries(_backendModel.REPORT_STATE_DICT)) {
    if (backend === status) {
      // distinguish "shared" and "created"
      if (status === _backendModel.BACKEND_REPORT_STATE.success && delivery) {
        res = _constants.REPORT_STATE.shared;
      } else {
        res = ui;
      }
    } else if (status === _backendModel.BACKEND_REPORT_STATE.scheduled) {
      // corner case
      res = _constants.REPORT_STATE.pending;
    }
  }
  return res;
};
const getUiReportSource = type => {
  let res;
  for (let [ui, backend] of Object.entries(_backendModel.REPORT_SOURCE_DICT)) {
    if (backend === type) {
      res = ui;
    }
  }
  return res;
};
const getUiReportDefinitionStatus = isEnabled => {
  return isEnabled ? _constants.REPORT_DEFINITION_STATUS.active : _constants.REPORT_DEFINITION_STATUS.disabled;
};
const getUiTriggerParams = (triggerType, schedule, createdTimeMs, isEnabled) => {
  let res = {
    trigger_type: getUiTriggerType(triggerType),
    ...(getUiTriggerType(triggerType) === _constants.TRIGGER_TYPE.schedule && getUiScheduleParams(schedule, createdTimeMs, isEnabled))
  };
  return res;
};

// Delivery
const getUiDeliveryParams = delivery => {
  const opensearchDashboardsUserDeliveryParams = {
    configIds: [],
    title: '',
    textDescription: '',
    htmlDescription: ''
  };
  let params;
  if (delivery) {
    const {
      ...rest
    } = delivery;
    params = {
      ...rest
    };
  } else {
    params = opensearchDashboardsUserDeliveryParams;
  }
  return params;
};

// helper function to add tenant info to url(if tenant is available)
const addTenantToURL = (url, userRequestedTenant) => {
  // build fake url from relative url
  const fakeUrl = `http://opensearch.com${url}`;
  const tenantKey = 'security_tenant';
  const tenantKeyAndValue = tenantKey + '=' + encodeURIComponent(userRequestedTenant);
  const {
    pathname,
    search
  } = (0, _url.parse)(fakeUrl);
  const queryDelimiter = !search ? '?' : '&';

  // The url parser returns null if the search is empty. Change that to an empty
  // string so that we can use it to build the values later
  if (search && search.toLowerCase().indexOf(tenantKey) > -1) {
    // If we for some reason already have a tenant in the URL we skip any updates
    return url;
  }

  // A helper for finding the part in the string that we want to extend/replace
  const valueToReplace = pathname + (search || '');
  const replaceWith = valueToReplace + queryDelimiter + tenantKeyAndValue;
  return url.replace(valueToReplace, replaceWith);
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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