"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
var _lodash = _interopRequireDefault(require("lodash"));
var _constants = require("../../utils/constants");
var _helpers = require("./utils/helpers");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
class MonitorService {
  constructor(esDriver) {
    _defineProperty(this, "createMonitor", async (context, req, res) => {
      try {
        const params = {
          body: req.body
        };
        const {
          callAsCurrentUser
        } = await this.esDriver.asScoped(req);
        const createResponse = await callAsCurrentUser('alerting.createMonitor', params);
        return res.ok({
          body: {
            ok: true,
            resp: createResponse
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - createMonitor:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "createWorkflow", async (context, req, res) => {
      try {
        const params = {
          body: req.body
        };
        const {
          callAsCurrentUser
        } = await this.esDriver.asScoped(req);
        const createResponse = await callAsCurrentUser('alerting.createWorkflow', params);
        return res.ok({
          body: {
            ok: true,
            resp: createResponse
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - createWorkflow:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "deleteMonitor", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          monitorId: id
        };
        const {
          callAsCurrentUser
        } = await this.esDriver.asScoped(req);
        const response = await callAsCurrentUser('alerting.deleteMonitor', params);
        return res.ok({
          body: {
            ok: response.result === 'deleted' || response.result === undefined
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - deleteMonitor:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "deleteWorkflow", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          workflowId: id
        };
        const {
          callAsCurrentUser
        } = await this.esDriver.asScoped(req);
        const response = await callAsCurrentUser('alerting.deleteWorkflow', params);
        return res.ok({
          body: {
            ok: response.result === 'deleted' || response.result === undefined
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - deleteWorkflow:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "getMonitor", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          monitorId: id
        };
        const {
          callAsCurrentUser
        } = await this.esDriver.asScoped(req);
        const getResponse = await callAsCurrentUser('alerting.getMonitor', params);
        let monitor = _lodash.default.get(getResponse, 'monitor', null);
        const version = _lodash.default.get(getResponse, '_version', null);
        const ifSeqNo = _lodash.default.get(getResponse, '_seq_no', null);
        const ifPrimaryTerm = _lodash.default.get(getResponse, '_primary_term', null);
        const associated_workflows = _lodash.default.get(getResponse, 'associated_workflows', null);
        if (monitor) {
          const {
            callAsCurrentUser
          } = this.esDriver.asScoped(req);
          const aggsParams = {
            index: _constants.INDEX.ALL_ALERTS,
            body: {
              size: 0,
              query: {
                bool: {
                  must: {
                    term: {
                      monitor_id: id
                    }
                  }
                }
              },
              aggs: {
                active_count: {
                  terms: {
                    field: 'state'
                  }
                },
                '24_hour_count': {
                  date_range: {
                    field: 'start_time',
                    ranges: [{
                      from: 'now-24h/h'
                    }]
                  }
                }
              }
            }
          };
          const searchResponse = await callAsCurrentUser('alerting.getMonitors', aggsParams);
          const dayCount = _lodash.default.get(searchResponse, 'aggregations.24_hour_count.buckets.0.doc_count', 0);
          const activeBuckets = _lodash.default.get(searchResponse, 'aggregations.active_count.buckets', []);
          const activeCount = activeBuckets.reduce((acc, curr) => curr.key === 'ACTIVE' ? curr.doc_count : acc, 0);
          if (associated_workflows) {
            monitor = {
              ...monitor,
              associated_workflows,
              associatedCompositeMonitorCnt: associated_workflows.length
            };
          }
          monitor = {
            ...monitor,
            item_type: monitor.workflow_type || monitor.monitor_type,
            id,
            version
          };
          return res.ok({
            body: {
              ok: true,
              resp: monitor,
              activeCount,
              dayCount,
              version,
              ifSeqNo,
              ifPrimaryTerm
            }
          });
        } else {
          return res.ok({
            body: {
              ok: false
            }
          });
        }
      } catch (err) {
        console.error('Alerting - MonitorService - getMonitor:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "getWorkflow", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          monitorId: id
        };
        const {
          callAsCurrentUser
        } = await this.esDriver.asScoped(req);
        const getResponse = await callAsCurrentUser('alerting.getWorkflow', params);
        let workflow = _lodash.default.get(getResponse, 'workflow', null);
        const version = _lodash.default.get(getResponse, '_version', null);
        const ifSeqNo = _lodash.default.get(getResponse, '_seq_no', null);
        const ifPrimaryTerm = _lodash.default.get(getResponse, '_primary_term', null);
        workflow.monitor_type = workflow.workflow_type;
        workflow = {
          ...workflow,
          item_type: workflow.workflow_type,
          id,
          version
        };
        return res.ok({
          body: {
            ok: true,
            resp: workflow,
            activeCount: 0,
            dayCount: 0,
            version,
            ifSeqNo,
            ifPrimaryTerm
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - getWorkflow:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "updateMonitor", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          monitorId: id,
          body: req.body,
          refresh: 'wait_for'
        };
        const {
          type
        } = req.body;

        // TODO DRAFT: Are we sure we need to include ifSeqNo and ifPrimaryTerm from the UI side when updating monitors?
        const {
          ifSeqNo,
          ifPrimaryTerm
        } = req.query;
        if (ifSeqNo && ifPrimaryTerm) {
          params.if_seq_no = ifSeqNo;
          params.if_primary_term = ifPrimaryTerm;
        }
        const {
          callAsCurrentUser
        } = await this.esDriver.asScoped(req);
        const updateResponse = await callAsCurrentUser(`alerting.${type === 'workflow' ? 'updateWorkflow' : 'updateMonitor'}`, params);
        const {
          _version,
          _id
        } = updateResponse;
        return res.ok({
          body: {
            ok: true,
            version: _version,
            id: _id
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - updateMonitor:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "getMonitors", async (context, req, res) => {
      try {
        const {
          from,
          size,
          search,
          sortDirection,
          sortField,
          state,
          monitorIds
        } = req.query;
        let must = {
          match_all: {}
        };
        if (search.trim()) {
          // This is an expensive wildcard query to match monitor names such as: "This is a long monitor name"
          // search query => "long monit"
          // This is acceptable because we will never allow more than 1,000 monitors
          must = {
            query_string: {
              default_field: 'monitor.name',
              default_operator: 'AND',
              query: `*${search.trim().split(' ').join('* *')}*`
            }
          };
        }
        const should = [];
        const mustList = [must];
        if (monitorIds !== undefined) {
          mustList.push({
            terms: {
              _id: Array.isArray(monitorIds) ? monitorIds : [monitorIds]
            }
          });
        } else if (monitorIds === 'empty') {
          mustList.push({
            terms: {
              _id: []
            }
          });
        }
        if (state !== 'all') {
          const enabled = state === 'enabled';
          should.push({
            term: {
              'monitor.enabled': enabled
            }
          });
          should.push({
            term: {
              'workflow.enabled': enabled
            }
          });
        }
        const monitorSorts = {
          name: 'monitor.name.keyword'
        };
        const monitorSortPageData = {
          size: 1000
        };
        if (monitorSorts[sortField]) {
          monitorSortPageData.sort = [{
            [monitorSorts[sortField]]: sortDirection
          }];
          monitorSortPageData.size = _lodash.default.defaultTo(size, 1000);
          monitorSortPageData.from = _lodash.default.defaultTo(from, 0);
        }
        const params = {
          body: {
            seq_no_primary_term: true,
            version: true,
            ...monitorSortPageData,
            query: {
              bool: {
                should,
                must: mustList
              }
            },
            aggregations: {
              associated_composite_monitors: {
                nested: {
                  path: 'workflow.inputs.composite_input.sequence.delegates'
                },
                aggs: {
                  monitor_ids: {
                    terms: {
                      field: 'workflow.inputs.composite_input.sequence.delegates.monitor_id'
                    }
                  }
                }
              }
            }
          }
        };
        const {
          callAsCurrentUser: alertingCallAsCurrentUser
        } = await this.esDriver.asScoped(req);
        const getResponse = await alertingCallAsCurrentUser('alerting.getMonitors', params);
        const totalMonitors = _lodash.default.get(getResponse, 'hits.total.value', 0);
        const monitorKeyValueTuples = _lodash.default.get(getResponse, 'hits.hits', []).map(result => {
          const {
            _id: id,
            _version: version,
            _seq_no: ifSeqNo,
            _primary_term: ifPrimaryTerm,
            _source
          } = result;
          const monitor = _source.monitor ? _source.monitor : _source;
          monitor['item_type'] = monitor.workflow_type || monitor.monitor_type;
          const {
            name,
            enabled,
            item_type
          } = monitor;
          return [id, {
            id,
            version,
            ifSeqNo,
            ifPrimaryTerm,
            name,
            enabled,
            item_type,
            monitor
          }];
        }, {});
        const monitorMap = new Map(monitorKeyValueTuples);
        const associatedCompositeMonitorCountMap = {};
        _lodash.default.get(getResponse, 'aggregations.associated_composite_monitors.monitor_ids.buckets', []).forEach(({
          key,
          doc_count
        }) => {
          associatedCompositeMonitorCountMap[key] = doc_count;
        });
        const monitorIdsOutput = [...monitorMap.keys()];
        const aggsOrderData = {};
        const aggsSorts = {
          active: 'active',
          acknowledged: 'acknowledged',
          errors: 'errors',
          ignored: 'ignored',
          lastNotificationTime: 'last_notification_time'
        };
        if (aggsSorts[sortField]) {
          aggsOrderData.order = {
            [aggsSorts[sortField]]: sortDirection
          };
        }
        const aggsParams = {
          index: _constants.INDEX.ALL_ALERTS,
          body: {
            size: 0,
            query: {
              terms: {
                monitor_id: monitorIdsOutput
              }
            },
            aggregations: {
              uniq_monitor_ids: {
                terms: {
                  field: 'monitor_id',
                  ...aggsOrderData,
                  size: from + size
                },
                aggregations: {
                  active: {
                    filter: {
                      term: {
                        state: 'ACTIVE'
                      }
                    }
                  },
                  acknowledged: {
                    filter: {
                      term: {
                        state: 'ACKNOWLEDGED'
                      }
                    }
                  },
                  errors: {
                    filter: {
                      term: {
                        state: 'ERROR'
                      }
                    }
                  },
                  ignored: {
                    filter: {
                      bool: {
                        filter: {
                          term: {
                            state: 'COMPLETED'
                          }
                        },
                        must_not: {
                          exists: {
                            field: 'acknowledged_time'
                          }
                        }
                      }
                    }
                  },
                  last_notification_time: {
                    max: {
                      field: 'last_notification_time'
                    }
                  },
                  latest_alert: {
                    top_hits: {
                      size: 1,
                      sort: [{
                        start_time: {
                          order: 'desc'
                        }
                      }],
                      _source: {
                        includes: ['last_notification_time', 'trigger_name']
                      }
                    }
                  }
                }
              }
            }
          }
        };
        const {
          callAsCurrentUser
        } = this.esDriver.asScoped(req);
        const esAggsResponse = await callAsCurrentUser('alerting.getMonitors', aggsParams);
        const buckets = _lodash.default.get(esAggsResponse, 'aggregations.uniq_monitor_ids.buckets', []).map(bucket => {
          const {
            key: id,
            last_notification_time: {
              value: lastNotificationTime
            },
            ignored: {
              doc_count: ignored
            },
            acknowledged: {
              doc_count: acknowledged
            },
            active: {
              doc_count: active
            },
            errors: {
              doc_count: errors
            },
            latest_alert: {
              hits: {
                hits: [{
                  _source: {
                    trigger_name: latestAlert
                  }
                }]
              }
            }
          } = bucket;
          const monitor = monitorMap.get(id);
          monitorMap.delete(id);
          return {
            ...monitor,
            id,
            lastNotificationTime,
            ignored,
            latestAlert,
            acknowledged,
            active,
            errors,
            currentTime: Date.now(),
            associatedCompositeMonitorCnt: associatedCompositeMonitorCountMap[id] || 0
          };
        });
        const unusedMonitors = [...monitorMap.values()].map(monitor => ({
          ...monitor,
          lastNotificationTime: null,
          ignored: 0,
          active: 0,
          acknowledged: 0,
          errors: 0,
          latestAlert: '--',
          currentTime: Date.now(),
          associatedCompositeMonitorCnt: associatedCompositeMonitorCountMap[monitor.id] || 0
        }));
        let results = _lodash.default.orderBy(buckets.concat(unusedMonitors), [sortField], [sortDirection]);
        // If we sorted on monitor name then we already applied from/size to the first query to limit what we're aggregating over
        // Therefore we do not need to apply from/size to this result set
        // If we sorted on aggregations, then this is our in memory pagination
        if (!monitorSorts[sortField]) {
          results = results.slice(from, from + size);
        }
        return res.ok({
          body: {
            ok: true,
            monitors: results,
            totalMonitors
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - getMonitors', err);
        if ((0, _helpers.isIndexNotFoundError)(err)) {
          return res.ok({
            body: {
              ok: false,
              resp: {
                totalMonitors: 0,
                monitors: []
              }
            }
          });
        }
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "acknowledgeAlerts", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          monitorId: id,
          body: req.body
        };
        const {
          callAsCurrentUser
        } = this.esDriver.asScoped(req);
        const acknowledgeResponse = await callAsCurrentUser('alerting.acknowledgeAlerts', params);
        return res.ok({
          body: {
            ok: !acknowledgeResponse.failed.length,
            resp: acknowledgeResponse
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - acknowledgeAlerts:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "acknowledgeChainedAlerts", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          workflowId: id,
          body: req.body
        };
        const {
          callAsCurrentUser
        } = this.esDriver.asScoped(req);
        const acknowledgeResponse = await callAsCurrentUser('alerting.acknowledgeChainedAlerts', params);
        return res.ok({
          body: {
            ok: !acknowledgeResponse.failed.length,
            resp: acknowledgeResponse
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - acknowledgeChainedAlerts:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "executeMonitor", async (context, req, res) => {
      try {
        const {
          dryrun = 'true'
        } = req.query;
        const params = {
          body: req.body,
          dryrun
        };
        const {
          callAsCurrentUser
        } = await this.esDriver.asScoped(req);
        const executeResponse = await callAsCurrentUser('alerting.executeMonitor', params);
        return res.ok({
          body: {
            ok: true,
            resp: executeResponse
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - executeMonitor:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    //TODO: This is temporarily a pass through call which needs to be deprecated
    _defineProperty(this, "searchMonitors", async (context, req, res) => {
      try {
        const {
          query,
          index,
          size
        } = req.body;
        const params = {
          index,
          size,
          body: query
        };
        const {
          callAsCurrentUser
        } = await this.esDriver.asScoped(req);
        const results = await callAsCurrentUser('alerting.getMonitors', params);
        return res.ok({
          body: {
            ok: true,
            resp: results
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - searchMonitor:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    this.esDriver = esDriver;
  }
}
exports.default = MonitorService;
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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