"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.JwtAuthentication = exports.JWT_DEFAULT_EXTRA_STORAGE_OPTIONS = void 0;
var _authentication_type = require("../authentication_type");
var _routes = require("./routes");
var _cookie_splitter = require("../../../session/cookie_splitter");
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
const JWT_DEFAULT_EXTRA_STORAGE_OPTIONS = exports.JWT_DEFAULT_EXTRA_STORAGE_OPTIONS = {
  cookiePrefix: 'security_authentication_jwt',
  additionalCookies: 5
};
class JwtAuthentication extends _authentication_type.AuthenticationType {
  constructor(config, sessionStorageFactory, router, esClient, coreSetup, logger) {
    var _this$config$jwt;
    super(config, sessionStorageFactory, router, esClient, coreSetup, logger);
    _defineProperty(this, "type", 'jwt');
    _defineProperty(this, "authHeaderName", void 0);
    this.authHeaderName = ((_this$config$jwt = this.config.jwt) === null || _this$config$jwt === void 0 ? void 0 : _this$config$jwt.header.toLowerCase()) || 'authorization';
  }
  async init() {
    this.createExtraStorage();
    const routes = new _routes.JwtAuthRoutes(this.router, this.sessionStorageFactory, this.config);
    routes.setupRoutes();
  }
  createExtraStorage() {
    // @ts-ignore
    const hapiServer = this.sessionStorageFactory.asScoped({}).server;
    const {
      cookiePrefix,
      additionalCookies
    } = this.getExtraAuthStorageOptions();
    const extraCookieSettings = {
      isSecure: this.config.cookie.secure,
      isSameSite: this.config.cookie.isSameSite,
      password: this.config.cookie.password,
      domain: this.config.cookie.domain,
      path: this.coreSetup.http.basePath.serverBasePath || '/',
      clearInvalid: false,
      isHttpOnly: true,
      ignoreErrors: true,
      encoding: 'iron' // Same as hapi auth cookie
    };

    for (let i = 1; i <= additionalCookies; i++) {
      hapiServer.states.add(cookiePrefix + i, extraCookieSettings);
    }
  }
  getExtraAuthStorageOptions() {
    var _this$config$jwt2, _this$config$jwt3;
    const extraAuthStorageOptions = {
      cookiePrefix: ((_this$config$jwt2 = this.config.jwt) === null || _this$config$jwt2 === void 0 ? void 0 : _this$config$jwt2.extra_storage.cookie_prefix) || JWT_DEFAULT_EXTRA_STORAGE_OPTIONS.cookiePrefix,
      additionalCookies: ((_this$config$jwt3 = this.config.jwt) === null || _this$config$jwt3 === void 0 ? void 0 : _this$config$jwt3.extra_storage.additional_cookies) || JWT_DEFAULT_EXTRA_STORAGE_OPTIONS.additionalCookies,
      logger: this.logger
    };
    return extraAuthStorageOptions;
  }
  getTokenFromUrlParam(request) {
    var _this$config$jwt4;
    const urlParamName = (_this$config$jwt4 = this.config.jwt) === null || _this$config$jwt4 === void 0 ? void 0 : _this$config$jwt4.url_param;
    if (urlParamName) {
      const token = request.url.searchParams.get(urlParamName);
      return token || undefined;
    }
    return undefined;
  }
  getBearerToken(request) {
    const token = this.getTokenFromUrlParam(request);
    if (token) {
      return `Bearer ${token}`;
    }

    // no token in url parameter, try to get token from header
    return request.headers[this.authHeaderName] || undefined;
  }
  requestIncludesAuthInfo(request) {
    var _this$config$jwt5;
    if (request.headers[this.authHeaderName]) {
      return true;
    }
    const urlParamName = (_this$config$jwt5 = this.config.jwt) === null || _this$config$jwt5 === void 0 ? void 0 : _this$config$jwt5.url_param;
    if (urlParamName && request.url.searchParams.get(urlParamName)) {
      return true;
    }
    return false;
  }
  async getAdditionalAuthHeader(request) {
    const header = {};
    const token = this.getTokenFromUrlParam(request);
    if (token) {
      header[this.authHeaderName] = `Bearer ${token}`;
    }
    return header;
  }
  getCookie(request, authInfo) {
    (0, _cookie_splitter.setExtraAuthStorage)(request, this.getBearerToken(request) || '', this.getExtraAuthStorageOptions());
    return {
      username: authInfo.user_name,
      credentials: {
        authHeaderValueExtra: true
      },
      authType: this.type,
      expiryTime: Date.now() + this.config.session.ttl
    };
  }
  async isValidCookie(cookie, request) {
    var _cookie$credentials;
    const hasAuthHeaderValue = ((_cookie$credentials = cookie.credentials) === null || _cookie$credentials === void 0 ? void 0 : _cookie$credentials.authHeaderValue) || this.getExtraAuthStorageValue(request, cookie);
    return cookie.authType === this.type && cookie.username && cookie.expiryTime && hasAuthHeaderValue;
  }
  handleUnauthedRequest(request, response, toolkit) {
    return response.unauthorized();
  }
  getExtraAuthStorageValue(request, cookie) {
    var _cookie$credentials2;
    let extraValue = '';
    if (!((_cookie$credentials2 = cookie.credentials) !== null && _cookie$credentials2 !== void 0 && _cookie$credentials2.authHeaderValueExtra)) {
      return extraValue;
    }
    try {
      extraValue = (0, _cookie_splitter.getExtraAuthStorageValue)(request, this.getExtraAuthStorageOptions());
    } catch (error) {
      this.logger.info(error);
    }
    return extraValue;
  }
  buildAuthHeaderFromCookie(cookie, request) {
    var _cookie$credentials3;
    const header = {};
    if (cookie.credentials.authHeaderValueExtra) {
      try {
        const extraAuthStorageValue = this.getExtraAuthStorageValue(request, cookie);
        header.authorization = extraAuthStorageValue;
        return header;
      } catch (error) {
        this.logger.error(error);
      }
    }
    const authHeaderValue = (_cookie$credentials3 = cookie.credentials) === null || _cookie$credentials3 === void 0 ? void 0 : _cookie$credentials3.authHeaderValue;
    if (authHeaderValue) {
      header[this.authHeaderName] = authHeaderValue;
    }
    return header;
  }
}
exports.JwtAuthentication = JwtAuthentication;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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