"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OpenIdAuthentication = void 0;
var fs = _interopRequireWildcard(require("fs"));
var _wreck = _interopRequireDefault(require("@hapi/wreck"));
var _http = _interopRequireDefault(require("http"));
var _https = _interopRequireDefault(require("https"));
var _security_cookie = require("../../../session/security_cookie");
var _routes = require("./routes");
var _authentication_type = require("../authentication_type");
var _helper = require("./helper");
var _object_properties_defined = require("../../../utils/object_properties_defined");
var _common = require("../../../../common");
var _cookie_splitter = require("../../../session/cookie_splitter");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && Object.prototype.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
class OpenIdAuthentication extends _authentication_type.AuthenticationType {
  constructor(config, sessionStorageFactory, router, esClient, core, logger) {
    var _this$config$openid, _this$config$openid2;
    super(config, sessionStorageFactory, router, esClient, core, logger);
    _defineProperty(this, "type", _common.AuthType.OPEN_ID);
    _defineProperty(this, "openIdAuthConfig", void 0);
    _defineProperty(this, "authHeaderName", void 0);
    _defineProperty(this, "openIdConnectUrl", void 0);
    _defineProperty(this, "wreckClient", void 0);
    _defineProperty(this, "wreckHttpsOption", {});
    _defineProperty(this, "redirectOIDCCapture", (request, toolkit) => {
      const nextUrl = this.generateNextUrl(request);
      const clearOldVersionCookie = (0, _security_cookie.clearOldVersionCookieValue)(this.config);
      return toolkit.redirected({
        location: `${this.coreSetup.http.basePath.serverBasePath}/auth/openid/captureUrlFragment?nextUrl=${nextUrl}`,
        'set-cookie': clearOldVersionCookie
      });
    });
    this.wreckClient = this.createWreckClient();
    this.openIdAuthConfig = {};
    this.authHeaderName = ((_this$config$openid = this.config.openid) === null || _this$config$openid === void 0 ? void 0 : _this$config$openid.header) || '';
    this.openIdAuthConfig.authHeaderName = this.authHeaderName;
    this.openIdConnectUrl = ((_this$config$openid2 = this.config.openid) === null || _this$config$openid2 === void 0 ? void 0 : _this$config$openid2.connect_url) || '';
    let scope = this.config.openid.scope;
    if (scope.indexOf('openid') < 0) {
      scope = `openid ${scope}`;
    }
    this.openIdAuthConfig.scope = scope;
  }
  async init() {
    try {
      const response = await this.wreckClient.get(this.openIdConnectUrl);
      const payload = JSON.parse(response.payload);
      this.openIdAuthConfig.authorizationEndpoint = payload.authorization_endpoint;
      this.openIdAuthConfig.tokenEndpoint = payload.token_endpoint;
      this.openIdAuthConfig.endSessionEndpoint = payload.end_session_endpoint || undefined;
      this.createExtraStorage();
      const routes = new _routes.OpenIdAuthRoutes(this.router, this.config, this.sessionStorageFactory, this.openIdAuthConfig, this.securityClient, this.coreSetup, this.wreckClient);
      routes.setupRoutes();
    } catch (error) {
      this.logger.error(error); // TODO: log more info
      throw new Error('Failed when trying to obtain the endpoints from your IdP');
    }
  }
  generateNextUrl(request) {
    const path = this.coreSetup.http.basePath.serverBasePath + (request.url.pathname || '/app/opensearch-dashboards');
    return escape(path);
  }
  createWreckClient() {
    var _this$config$openid3, _this$config$openid4, _this$config$openid5, _this$config$openid6, _this$config$openid7, _this$config$openid9;
    if ((_this$config$openid3 = this.config.openid) !== null && _this$config$openid3 !== void 0 && _this$config$openid3.root_ca) {
      this.wreckHttpsOption.ca = [fs.readFileSync(this.config.openid.root_ca)];
      this.logger.debug(`Using CA Cert: ${this.config.openid.root_ca}`);
    }
    if ((_this$config$openid4 = this.config.openid) !== null && _this$config$openid4 !== void 0 && _this$config$openid4.pfx) {
      // Use PFX or PKCS12 if provided
      this.logger.debug(`Using PFX or PKCS12: ${this.config.openid.pfx}`);
      this.wreckHttpsOption.pfx = [fs.readFileSync(this.config.openid.pfx)];
    } else if ((_this$config$openid5 = this.config.openid) !== null && _this$config$openid5 !== void 0 && _this$config$openid5.certificate && (_this$config$openid6 = this.config.openid) !== null && _this$config$openid6 !== void 0 && _this$config$openid6.private_key) {
      // Use 'certificate' and 'private_key' if provided
      this.logger.debug(`Using Certificate: ${this.config.openid.certificate}`);
      this.logger.debug(`Using Private Key: ${this.config.openid.private_key}`);
      this.wreckHttpsOption.cert = [fs.readFileSync(this.config.openid.certificate)];
      this.wreckHttpsOption.key = [fs.readFileSync(this.config.openid.private_key)];
    } else {
      this.logger.debug(`Client certificates not provided. Mutual TLS will not be used to obtain endpoints.`);
    }
    // Check if passphrase is provided, use it for 'pfx' and 'key'
    if (((_this$config$openid7 = this.config.openid) === null || _this$config$openid7 === void 0 ? void 0 : _this$config$openid7.passphrase) !== '') {
      var _this$config$openid8;
      this.logger.debug(`Passphrase not provided for private key and/or pfx.`);
      this.wreckHttpsOption.passphrase = (_this$config$openid8 = this.config.openid) === null || _this$config$openid8 === void 0 ? void 0 : _this$config$openid8.passphrase;
    }
    if (((_this$config$openid9 = this.config.openid) === null || _this$config$openid9 === void 0 ? void 0 : _this$config$openid9.verify_hostnames) === false) {
      this.logger.debug(`openId auth 'verify_hostnames' option is off.`);
      this.wreckHttpsOption.checkServerIdentity = (host, cert) => {
        return undefined;
      };
    }
    this.logger.info((0, _object_properties_defined.getObjectProperties)(this.wreckHttpsOption, 'WreckHttpsOptions'));
    if (Object.keys(this.wreckHttpsOption).length > 0) {
      return _wreck.default.defaults({
        agents: {
          http: new _http.default.Agent(),
          https: new _https.default.Agent(this.wreckHttpsOption),
          httpsAllowUnauthorized: new _https.default.Agent({
            rejectUnauthorized: false
          })
        }
      });
    } else {
      return _wreck.default;
    }
  }
  getWreckHttpsOptions() {
    return this.wreckHttpsOption;
  }
  createExtraStorage() {
    // @ts-ignore
    const hapiServer = this.sessionStorageFactory.asScoped({}).server;
    const extraCookiePrefix = this.config.openid.extra_storage.cookie_prefix;
    const extraCookieSettings = {
      isSecure: this.config.cookie.secure,
      isSameSite: this.config.cookie.isSameSite,
      password: this.config.cookie.password,
      domain: this.config.cookie.domain,
      path: this.coreSetup.http.basePath.serverBasePath || '/',
      clearInvalid: false,
      isHttpOnly: true,
      ignoreErrors: true,
      encoding: 'iron' // Same as hapi auth cookie
    };

    for (let i = 1; i <= this.config.openid.extra_storage.additional_cookies; i++) {
      hapiServer.states.add(extraCookiePrefix + i, extraCookieSettings);
    }
  }
  getExtraAuthStorageOptions() {
    // If we're here, we will always have the openid configuration
    return {
      cookiePrefix: this.config.openid.extra_storage.cookie_prefix,
      additionalCookies: this.config.openid.extra_storage.additional_cookies,
      logger: this.logger
    };
  }
  requestIncludesAuthInfo(request) {
    return request.headers.authorization ? true : false;
  }
  async getAdditionalAuthHeader(request) {
    return {};
  }
  getCookie(request, authInfo) {
    (0, _cookie_splitter.setExtraAuthStorage)(request, request.headers.authorization, this.getExtraAuthStorageOptions());
    return {
      username: authInfo.user_name,
      credentials: {
        authHeaderValueExtra: true
      },
      authType: this.type,
      expiryTime: Date.now() + this.config.session.ttl
    };
  }

  // TODO: Add token expiration check here
  async isValidCookie(cookie, request) {
    var _cookie$credentials, _cookie$credentials2, _cookie$credentials3;
    if (cookie.authType !== this.type || !cookie.username || !cookie.expiryTime || !((_cookie$credentials = cookie.credentials) !== null && _cookie$credentials !== void 0 && _cookie$credentials.authHeaderValue) && !this.getExtraAuthStorageValue(request, cookie) || !((_cookie$credentials2 = cookie.credentials) !== null && _cookie$credentials2 !== void 0 && _cookie$credentials2.expires_at)) {
      return false;
    }
    if (((_cookie$credentials3 = cookie.credentials) === null || _cookie$credentials3 === void 0 ? void 0 : _cookie$credentials3.expires_at) > Date.now()) {
      return true;
    }

    // need to renew id token
    if (cookie.credentials.refresh_token) {
      try {
        var _this$config$openid10, _this$config$openid11;
        const query = {
          grant_type: 'refresh_token',
          client_id: (_this$config$openid10 = this.config.openid) === null || _this$config$openid10 === void 0 ? void 0 : _this$config$openid10.client_id,
          client_secret: (_this$config$openid11 = this.config.openid) === null || _this$config$openid11 === void 0 ? void 0 : _this$config$openid11.client_secret,
          refresh_token: cookie.credentials.refresh_token
        };
        const refreshTokenResponse = await (0, _helper.callTokenEndpoint)(this.openIdAuthConfig.tokenEndpoint, query, this.wreckClient);

        // if no id_token from refresh token call, maybe the Idp doesn't allow refresh id_token
        if (refreshTokenResponse.idToken) {
          cookie.credentials = {
            authHeaderValueExtra: true,
            refresh_token: refreshTokenResponse.refreshToken,
            expires_at: (0, _helper.getExpirationDate)(refreshTokenResponse) // expiresIn is in second
          };

          (0, _cookie_splitter.setExtraAuthStorage)(request, `Bearer ${refreshTokenResponse.idToken}`, this.getExtraAuthStorageOptions());
          return true;
        } else {
          return false;
        }
      } catch (error) {
        this.logger.error(error);
        return false;
      }
    } else {
      // no refresh token, and current token is expired
      return false;
    }
  }
  handleUnauthedRequest(request, response, toolkit) {
    if (this.isPageRequest(request)) {
      return this.redirectOIDCCapture(request, toolkit);
    } else {
      return response.unauthorized();
    }
  }
  getExtraAuthStorageValue(request, cookie) {
    var _cookie$credentials4;
    let extraValue = '';
    if (!((_cookie$credentials4 = cookie.credentials) !== null && _cookie$credentials4 !== void 0 && _cookie$credentials4.authHeaderValueExtra)) {
      return extraValue;
    }
    try {
      extraValue = (0, _cookie_splitter.getExtraAuthStorageValue)(request, this.getExtraAuthStorageOptions());
    } catch (error) {
      this.logger.info(error);
    }
    return extraValue;
  }
  buildAuthHeaderFromCookie(cookie, request) {
    var _cookie$credentials5;
    const header = {};
    if (cookie.credentials.authHeaderValueExtra) {
      try {
        const extraAuthStorageValue = this.getExtraAuthStorageValue(request, cookie);
        header.authorization = extraAuthStorageValue;
        return header;
      } catch (error) {
        this.logger.error(error);
        // TODO Re-throw?
        // throw error;
      }
    }

    const authHeaderValue = (_cookie$credentials5 = cookie.credentials) === null || _cookie$credentials5 === void 0 ? void 0 : _cookie$credentials5.authHeaderValue;
    if (authHeaderValue) {
      header.authorization = authHeaderValue;
    }
    return header;
  }
}
exports.OpenIdAuthentication = OpenIdAuthentication;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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