"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const chai_1 = require("chai");
const misc_1 = require("../../src/utils/misc");
describe("withInterpreter", () => {
    const tests = [
        {
            scenario: "when activation script is provided",
            executable: "ansible-lint",
            args: "playbook.yml",
            interpreterPath: "",
            activationScript: "/path/to/venv/bin/activate",
            expectedCommand: "bash -c 'source /path/to/venv/bin/activate && ansible-lint playbook.yml'",
            expectedEnv: undefined,
        },
        {
            scenario: "when no activation script is provided",
            executable: "ansible-lint",
            args: "playbook.yml",
            interpreterPath: "",
            activationScript: "",
            expectedCommand: "ansible-lint playbook.yml",
        },
        {
            scenario: "when absolute path of executable is provided",
            executable: "/absolute/path/to/ansible-lint",
            args: "playbook.yml",
            interpreterPath: "",
            activationScript: "",
            expectedCommand: "/absolute/path/to/ansible-lint playbook.yml",
        },
        {
            scenario: "when absolute path of interpreter is provided",
            executable: "/absolute/path/to/ansible-lint",
            args: "playbook.yml",
            interpreterPath: "/path/to/venv/bin/python",
            activationScript: "",
            expectedCommand: "/absolute/path/to/ansible-lint playbook.yml",
            expectedEnv: {
                VIRTUAL_ENV: "/path/to/venv",
                PATH: "/path/to/venv/bin",
            },
        },
    ];
    tests.forEach(({ scenario, executable, args, interpreterPath, activationScript, expectedCommand, expectedEnv, }) => {
        it(`should provide command ${scenario}`, () => {
            const actualCommand = (0, misc_1.withInterpreter)(executable, args, interpreterPath, activationScript);
            (0, chai_1.expect)(actualCommand[0]).to.equal(expectedCommand);
            if (expectedEnv) {
                const expectedKeys = Object.keys(expectedEnv);
                expectedKeys.forEach((key) => {
                    (0, chai_1.expect)(actualCommand[1]).to.haveOwnProperty(key);
                    (0, chai_1.expect)(actualCommand[1][key]).to.include(expectedEnv[key]);
                });
            }
        });
    });
});
//# sourceMappingURL=withInterpreter.test.js.map