"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ObservabilityPlugin = void 0;
var _opensearch_observability_plugin = require("./adaptors/opensearch_observability_plugin");
var _ppl_plugin = require("./adaptors/ppl_plugin");
var _index = require("./routes/index");
var _observability_saved_object = require("./saved_objects/observability_saved_object");
var _ppl_parser = require("./parsers/ppl_parser");
var _migrations = require("./adaptors/integrations/migrations");
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
class ObservabilityPlugin {
  constructor(initializerContext) {
    this.initializerContext = initializerContext;
    _defineProperty(this, "logger", void 0);
    this.logger = initializerContext.logger.get();
  }
  async setup(core, deps) {
    const {
      assistantDashboards
    } = deps;
    this.logger.debug('Observability: Setup');
    const router = core.http.createRouter();
    const openSearchObservabilityClient = core.opensearch.legacy.createClient('opensearch_observability', {
      plugins: [_ppl_plugin.PPLPlugin, _opensearch_observability_plugin.OpenSearchObservabilityPlugin]
    });

    // @ts-ignore
    core.http.registerRouteHandlerContext('observability_plugin', (_context, _request) => {
      return {
        logger: this.logger,
        observabilityClient: openSearchObservabilityClient
      };
    });
    const obsPanelType = {
      name: 'observability-panel',
      hidden: false,
      namespaceType: 'single',
      mappings: {
        dynamic: false,
        properties: {
          title: {
            type: 'text'
          },
          description: {
            type: 'text'
          }
        }
      },
      management: {
        importableAndExportable: true,
        getInAppUrl() {
          return {
            path: `/app/management/observability/settings`,
            uiCapabilitiesPath: 'advancedSettings.show'
          };
        },
        getTitle(obj) {
          return `Observability Settings [${obj.id}]`;
        }
      },
      migrations: {
        '3.0.0': doc => ({
          ...doc,
          description: ''
        }),
        '3.0.1': doc => ({
          ...doc,
          description: 'Some Description Text'
        }),
        '3.0.2': doc => ({
          ...doc,
          dateCreated: parseInt(doc.dateCreated || '0', 10)
        })
      }
    };
    const integrationInstanceType = {
      name: 'integration-instance',
      hidden: false,
      namespaceType: 'single',
      management: {
        importableAndExportable: true,
        getInAppUrl(obj) {
          return {
            path: `/app/integrations#/installed/${obj.id}`,
            uiCapabilitiesPath: 'advancedSettings.show'
          };
        },
        getTitle(obj) {
          return obj.attributes.name;
        }
      },
      mappings: {
        dynamic: false,
        properties: {
          name: {
            type: 'text'
          },
          templateName: {
            type: 'text'
          },
          dataSource: {
            type: 'text'
          },
          creationDate: {
            type: 'date'
          },
          assets: {
            type: 'nested'
          }
        }
      }
    };
    const integrationTemplateType = {
      name: 'integration-template',
      hidden: false,
      namespaceType: 'single',
      management: {
        importableAndExportable: true,
        getInAppUrl(obj) {
          return {
            path: `/app/integrations#/available/${obj.attributes.name}`,
            uiCapabilitiesPath: 'advancedSettings.show'
          };
        },
        getTitle(obj) {
          var _obj$attributes$displ;
          return (_obj$attributes$displ = obj.attributes.displayName) !== null && _obj$attributes$displ !== void 0 ? _obj$attributes$displ : obj.attributes.name;
        }
      },
      mappings: {
        dynamic: false,
        properties: {
          name: {
            type: 'text'
          },
          version: {
            type: 'text'
          },
          displayName: {
            type: 'text'
          },
          license: {
            type: 'text'
          },
          type: {
            type: 'text'
          },
          labels: {
            type: 'text'
          },
          author: {
            type: 'text'
          },
          description: {
            type: 'text'
          },
          sourceUrl: {
            type: 'text'
          },
          statics: {
            type: 'nested'
          },
          components: {
            type: 'nested'
          },
          assets: {
            type: 'nested'
          },
          sampleData: {
            type: 'nested'
          }
        }
      },
      migrations: {
        '3.0.0': _migrations.migrateV1IntegrationToV2Integration
      }
    };
    core.savedObjects.registerType(obsPanelType);
    core.savedObjects.registerType(integrationInstanceType);
    core.savedObjects.registerType(integrationTemplateType);

    // Register server side APIs
    (0, _index.setupRoutes)({
      router,
      client: openSearchObservabilityClient
    });
    core.savedObjects.registerType(_observability_saved_object.visualizationSavedObject);
    core.savedObjects.registerType(_observability_saved_object.searchSavedObject);
    core.capabilities.registerProvider(() => ({
      observability: {
        show: true
      }
    }));
    assistantDashboards === null || assistantDashboards === void 0 || assistantDashboards.registerMessageParser(_ppl_parser.PPLParsers);
    return {};
  }
  start(_core) {
    this.logger.debug('Observability: Started');
    return {};
  }
  stop() {}
}
exports.ObservabilityPlugin = ObservabilityPlugin;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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