"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.handleWithCallback = void 0;
exports.registerIntegrationsRoute = registerIntegrationsRoute;
var _configSchema = require("@osd/config-schema");
var mime = _interopRequireWildcard(require("mime"));
var _sanitizeFilename = _interopRequireDefault(require("sanitize-filename"));
var _shared = require("../../../common/constants/shared");
var _integrations_manager = require("../../adaptors/integrations/integrations_manager");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && Object.prototype.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

/**
 * Handle an `OpenSearchDashboardsRequest` using the provided `callback` function.
 * This is a convenience method that handles common error handling and response formatting.
 * The callback must accept a `IntegrationsAdaptor` as its first argument.
 *
 * If the callback throws an error,
 * the `OpenSearchDashboardsResponse` will be formatted using the error's `statusCode` and `message` properties.
 * Otherwise, the callback's return value will be formatted in a JSON object under the `data` field.
 *
 * @param {IntegrationsAdaptor} adaptor The adaptor instance to use for making requests.
 * @param {OpenSearchDashboardsResponseFactory} response The factory to use for creating responses.
 * @callback callback A callback that will invoke a request on a provided adaptor.
 * @returns {Promise<OpenSearchDashboardsResponse>} An `OpenSearchDashboardsResponse` with the return data from the callback.
 */
const handleWithCallback = async (adaptor, response, callback) => {
  try {
    const data = await callback(adaptor);
    return response.ok({
      body: {
        data
      }
    });
  } catch (err) {
    console.error(`handleWithCallback: callback failed with error "${err.message}"`);
    return response.custom({
      statusCode: err.statusCode || 500,
      body: err.message
    });
  }
};
exports.handleWithCallback = handleWithCallback;
const getAdaptor = (context, _request) => {
  return new _integrations_manager.IntegrationsManager(context.core.savedObjects.client);
};
function registerIntegrationsRoute(router) {
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/repository`,
    validate: false
  }, async (context, request, response) => {
    const adaptor = getAdaptor(context, request);
    return handleWithCallback(adaptor, response, async a => a.getIntegrationTemplates());
  });
  router.post({
    path: `${_shared.INTEGRATIONS_BASE}/store/{templateName}`,
    validate: {
      params: _configSchema.schema.object({
        templateName: _configSchema.schema.string()
      }),
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        dataSource: _configSchema.schema.string(),
        workflows: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string()))
      })
    }
  }, async (context, request, response) => {
    const adaptor = getAdaptor(context, request);
    return handleWithCallback(adaptor, response, async a => {
      return a.loadIntegrationInstance(request.params.templateName, request.body.name, request.body.dataSource, request.body.workflows);
    });
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/repository/{name}`,
    validate: {
      params: _configSchema.schema.object({
        name: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = getAdaptor(context, request);
    return handleWithCallback(adaptor, response, async a => (await a.getIntegrationTemplates({
      name: request.params.name
    })).hits[0]);
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/repository/{id}/static/{path}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string(),
        path: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = getAdaptor(context, request);
    try {
      const requestPath = (0, _sanitizeFilename.default)(request.params.path);
      const result = await adaptor.getStatic(request.params.id, requestPath);
      return response.ok({
        headers: {
          'Content-Type': mime.getType(request.params.path)
        },
        body: result
      });
    } catch (err) {
      return response.custom({
        statusCode: err.statusCode ? err.statusCode : 500,
        body: err.message
      });
    }
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/repository/{id}/schema`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = getAdaptor(context, request);
    return handleWithCallback(adaptor, response, async a => a.getSchemas(request.params.id));
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/repository/{id}/assets`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = getAdaptor(context, request);
    return handleWithCallback(adaptor, response, async a => a.getAssets(request.params.id));
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/repository/{id}/data`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = getAdaptor(context, request);
    return handleWithCallback(adaptor, response, async a => a.getSampleData(request.params.id));
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/store`,
    validate: false
  }, async (context, request, response) => {
    const adaptor = getAdaptor(context, request);
    return handleWithCallback(adaptor, response, async a => a.getIntegrationInstances({}));
  });
  router.delete({
    path: `${_shared.INTEGRATIONS_BASE}/store/{id}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = getAdaptor(context, request);
    return handleWithCallback(adaptor, response, async a => a.deleteIntegrationInstance(request.params.id));
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/store/{id}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = getAdaptor(context, request);
    return handleWithCallback(adaptor, response, async a => a.getIntegrationInstance({
      id: request.params.id
    }));
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJuYW1lcyI6WyJfY29uZmlnU2NoZW1hIiwicmVxdWlyZSIsIm1pbWUiLCJfaW50ZXJvcFJlcXVpcmVXaWxkY2FyZCIsIl9zYW5pdGl6ZUZpbGVuYW1lIiwiX2ludGVyb3BSZXF1aXJlRGVmYXVsdCIsIl9zaGFyZWQiLCJfaW50ZWdyYXRpb25zX21hbmFnZXIiLCJvYmoiLCJfX2VzTW9kdWxlIiwiZGVmYXVsdCIsIl9nZXRSZXF1aXJlV2lsZGNhcmRDYWNoZSIsImUiLCJXZWFrTWFwIiwiciIsInQiLCJoYXMiLCJnZXQiLCJuIiwiX19wcm90b19fIiwiYSIsIk9iamVjdCIsImRlZmluZVByb3BlcnR5IiwiZ2V0T3duUHJvcGVydHlEZXNjcmlwdG9yIiwidSIsInByb3RvdHlwZSIsImhhc093blByb3BlcnR5IiwiY2FsbCIsImkiLCJzZXQiLCJoYW5kbGVXaXRoQ2FsbGJhY2siLCJhZGFwdG9yIiwicmVzcG9uc2UiLCJjYWxsYmFjayIsImRhdGEiLCJvayIsImJvZHkiLCJlcnIiLCJjb25zb2xlIiwiZXJyb3IiLCJtZXNzYWdlIiwiY3VzdG9tIiwic3RhdHVzQ29kZSIsImV4cG9ydHMiLCJnZXRBZGFwdG9yIiwiY29udGV4dCIsIl9yZXF1ZXN0IiwiSW50ZWdyYXRpb25zTWFuYWdlciIsImNvcmUiLCJzYXZlZE9iamVjdHMiLCJjbGllbnQiLCJyZWdpc3RlckludGVncmF0aW9uc1JvdXRlIiwicm91dGVyIiwicGF0aCIsIklOVEVHUkFUSU9OU19CQVNFIiwidmFsaWRhdGUiLCJyZXF1ZXN0IiwiZ2V0SW50ZWdyYXRpb25UZW1wbGF0ZXMiLCJwb3N0IiwicGFyYW1zIiwic2NoZW1hIiwib2JqZWN0IiwidGVtcGxhdGVOYW1lIiwic3RyaW5nIiwibmFtZSIsImRhdGFTb3VyY2UiLCJ3b3JrZmxvd3MiLCJtYXliZSIsImFycmF5T2YiLCJsb2FkSW50ZWdyYXRpb25JbnN0YW5jZSIsImhpdHMiLCJpZCIsInJlcXVlc3RQYXRoIiwic2FuaXRpemUiLCJyZXN1bHQiLCJnZXRTdGF0aWMiLCJoZWFkZXJzIiwiZ2V0VHlwZSIsImdldFNjaGVtYXMiLCJnZXRBc3NldHMiLCJnZXRTYW1wbGVEYXRhIiwiZ2V0SW50ZWdyYXRpb25JbnN0YW5jZXMiLCJkZWxldGUiLCJkZWxldGVJbnRlZ3JhdGlvbkluc3RhbmNlIiwiZ2V0SW50ZWdyYXRpb25JbnN0YW5jZSJdLCJzb3VyY2VzIjpbImludGVncmF0aW9uc19yb3V0ZXIudHMiXSwic291cmNlc0NvbnRlbnQiOlsiLypcbiAqIENvcHlyaWdodCBPcGVuU2VhcmNoIENvbnRyaWJ1dG9yc1xuICogU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IEFwYWNoZS0yLjBcbiAqL1xuXG5pbXBvcnQgeyBzY2hlbWEgfSBmcm9tICdAb3NkL2NvbmZpZy1zY2hlbWEnO1xuaW1wb3J0ICogYXMgbWltZSBmcm9tICdtaW1lJztcbmltcG9ydCBzYW5pdGl6ZSBmcm9tICdzYW5pdGl6ZS1maWxlbmFtZSc7XG5pbXBvcnQgeyBJUm91dGVyLCBSZXF1ZXN0SGFuZGxlckNvbnRleHQgfSBmcm9tICcuLi8uLi8uLi8uLi8uLi9zcmMvY29yZS9zZXJ2ZXInO1xuaW1wb3J0IHsgSU5URUdSQVRJT05TX0JBU0UgfSBmcm9tICcuLi8uLi8uLi9jb21tb24vY29uc3RhbnRzL3NoYXJlZCc7XG5pbXBvcnQgeyBJbnRlZ3JhdGlvbnNBZGFwdG9yIH0gZnJvbSAnLi4vLi4vYWRhcHRvcnMvaW50ZWdyYXRpb25zL2ludGVncmF0aW9uc19hZGFwdG9yJztcbmltcG9ydCB7XG4gIE9wZW5TZWFyY2hEYXNoYm9hcmRzUmVxdWVzdCxcbiAgT3BlblNlYXJjaERhc2hib2FyZHNSZXNwb25zZSxcbiAgT3BlblNlYXJjaERhc2hib2FyZHNSZXNwb25zZUZhY3RvcnksXG59IGZyb20gJy4uLy4uLy4uLy4uLy4uL3NyYy9jb3JlL3NlcnZlci9odHRwL3JvdXRlcic7XG5pbXBvcnQgeyBJbnRlZ3JhdGlvbnNNYW5hZ2VyIH0gZnJvbSAnLi4vLi4vYWRhcHRvcnMvaW50ZWdyYXRpb25zL2ludGVncmF0aW9uc19tYW5hZ2VyJztcblxuLyoqXG4gKiBIYW5kbGUgYW4gYE9wZW5TZWFyY2hEYXNoYm9hcmRzUmVxdWVzdGAgdXNpbmcgdGhlIHByb3ZpZGVkIGBjYWxsYmFja2AgZnVuY3Rpb24uXG4gKiBUaGlzIGlzIGEgY29udmVuaWVuY2UgbWV0aG9kIHRoYXQgaGFuZGxlcyBjb21tb24gZXJyb3IgaGFuZGxpbmcgYW5kIHJlc3BvbnNlIGZvcm1hdHRpbmcuXG4gKiBUaGUgY2FsbGJhY2sgbXVzdCBhY2NlcHQgYSBgSW50ZWdyYXRpb25zQWRhcHRvcmAgYXMgaXRzIGZpcnN0IGFyZ3VtZW50LlxuICpcbiAqIElmIHRoZSBjYWxsYmFjayB0aHJvd3MgYW4gZXJyb3IsXG4gKiB0aGUgYE9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2VgIHdpbGwgYmUgZm9ybWF0dGVkIHVzaW5nIHRoZSBlcnJvcidzIGBzdGF0dXNDb2RlYCBhbmQgYG1lc3NhZ2VgIHByb3BlcnRpZXMuXG4gKiBPdGhlcndpc2UsIHRoZSBjYWxsYmFjaydzIHJldHVybiB2YWx1ZSB3aWxsIGJlIGZvcm1hdHRlZCBpbiBhIEpTT04gb2JqZWN0IHVuZGVyIHRoZSBgZGF0YWAgZmllbGQuXG4gKlxuICogQHBhcmFtIHtJbnRlZ3JhdGlvbnNBZGFwdG9yfSBhZGFwdG9yIFRoZSBhZGFwdG9yIGluc3RhbmNlIHRvIHVzZSBmb3IgbWFraW5nIHJlcXVlc3RzLlxuICogQHBhcmFtIHtPcGVuU2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlRmFjdG9yeX0gcmVzcG9uc2UgVGhlIGZhY3RvcnkgdG8gdXNlIGZvciBjcmVhdGluZyByZXNwb25zZXMuXG4gKiBAY2FsbGJhY2sgY2FsbGJhY2sgQSBjYWxsYmFjayB0aGF0IHdpbGwgaW52b2tlIGEgcmVxdWVzdCBvbiBhIHByb3ZpZGVkIGFkYXB0b3IuXG4gKiBAcmV0dXJucyB7UHJvbWlzZTxPcGVuU2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlPn0gQW4gYE9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2VgIHdpdGggdGhlIHJldHVybiBkYXRhIGZyb20gdGhlIGNhbGxiYWNrLlxuICovXG5leHBvcnQgY29uc3QgaGFuZGxlV2l0aENhbGxiYWNrID0gYXN5bmMgPFQ+KFxuICBhZGFwdG9yOiBJbnRlZ3JhdGlvbnNBZGFwdG9yLFxuICByZXNwb25zZTogT3BlblNlYXJjaERhc2hib2FyZHNSZXNwb25zZUZhY3RvcnksXG4gIGNhbGxiYWNrOiAoYTogSW50ZWdyYXRpb25zQWRhcHRvcikgPT4gUHJvbWlzZTxUPlxuKTogUHJvbWlzZTxPcGVuU2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlPHsgZGF0YTogVCB9IHwgc3RyaW5nPj4gPT4ge1xuICB0cnkge1xuICAgIGNvbnN0IGRhdGEgPSBhd2FpdCBjYWxsYmFjayhhZGFwdG9yKTtcbiAgICByZXR1cm4gcmVzcG9uc2Uub2soe1xuICAgICAgYm9keToge1xuICAgICAgICBkYXRhLFxuICAgICAgfSxcbiAgICB9KSBhcyBPcGVuU2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlPHsgZGF0YTogVCB9PjtcbiAgfSBjYXRjaCAoZXJyKSB7XG4gICAgY29uc29sZS5lcnJvcihgaGFuZGxlV2l0aENhbGxiYWNrOiBjYWxsYmFjayBmYWlsZWQgd2l0aCBlcnJvciBcIiR7ZXJyLm1lc3NhZ2V9XCJgKTtcbiAgICByZXR1cm4gcmVzcG9uc2UuY3VzdG9tKHtcbiAgICAgIHN0YXR1c0NvZGU6IGVyci5zdGF0dXNDb2RlIHx8IDUwMCxcbiAgICAgIGJvZHk6IGVyci5tZXNzYWdlLFxuICAgIH0pO1xuICB9XG59O1xuXG5jb25zdCBnZXRBZGFwdG9yID0gKFxuICBjb250ZXh0OiBSZXF1ZXN0SGFuZGxlckNvbnRleHQsXG4gIF9yZXF1ZXN0OiBPcGVuU2VhcmNoRGFzaGJvYXJkc1JlcXVlc3Rcbik6IEludGVncmF0aW9uc0FkYXB0b3IgPT4ge1xuICByZXR1cm4gbmV3IEludGVncmF0aW9uc01hbmFnZXIoY29udGV4dC5jb3JlLnNhdmVkT2JqZWN0cy5jbGllbnQpO1xufTtcblxuZXhwb3J0IGZ1bmN0aW9uIHJlZ2lzdGVySW50ZWdyYXRpb25zUm91dGUocm91dGVyOiBJUm91dGVyKSB7XG4gIHJvdXRlci5nZXQoXG4gICAge1xuICAgICAgcGF0aDogYCR7SU5URUdSQVRJT05TX0JBU0V9L3JlcG9zaXRvcnlgLFxuICAgICAgdmFsaWRhdGU6IGZhbHNlLFxuICAgIH0sXG4gICAgYXN5bmMgKGNvbnRleHQsIHJlcXVlc3QsIHJlc3BvbnNlKTogUHJvbWlzZTxPcGVuU2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlPiA9PiB7XG4gICAgICBjb25zdCBhZGFwdG9yID0gZ2V0QWRhcHRvcihjb250ZXh0LCByZXF1ZXN0KTtcbiAgICAgIHJldHVybiBoYW5kbGVXaXRoQ2FsbGJhY2soYWRhcHRvciwgcmVzcG9uc2UsIGFzeW5jIChhOiBJbnRlZ3JhdGlvbnNBZGFwdG9yKSA9PlxuICAgICAgICBhLmdldEludGVncmF0aW9uVGVtcGxhdGVzKClcbiAgICAgICk7XG4gICAgfVxuICApO1xuXG4gIHJvdXRlci5wb3N0KFxuICAgIHtcbiAgICAgIHBhdGg6IGAke0lOVEVHUkFUSU9OU19CQVNFfS9zdG9yZS97dGVtcGxhdGVOYW1lfWAsXG4gICAgICB2YWxpZGF0ZToge1xuICAgICAgICBwYXJhbXM6IHNjaGVtYS5vYmplY3Qoe1xuICAgICAgICAgIHRlbXBsYXRlTmFtZTogc2NoZW1hLnN0cmluZygpLFxuICAgICAgICB9KSxcbiAgICAgICAgYm9keTogc2NoZW1hLm9iamVjdCh7XG4gICAgICAgICAgbmFtZTogc2NoZW1hLnN0cmluZygpLFxuICAgICAgICAgIGRhdGFTb3VyY2U6IHNjaGVtYS5zdHJpbmcoKSxcbiAgICAgICAgICB3b3JrZmxvd3M6IHNjaGVtYS5tYXliZShzY2hlbWEuYXJyYXlPZihzY2hlbWEuc3RyaW5nKCkpKSxcbiAgICAgICAgfSksXG4gICAgICB9LFxuICAgIH0sXG4gICAgYXN5bmMgKGNvbnRleHQsIHJlcXVlc3QsIHJlc3BvbnNlKTogUHJvbWlzZTxPcGVuU2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlPiA9PiB7XG4gICAgICBjb25zdCBhZGFwdG9yID0gZ2V0QWRhcHRvcihjb250ZXh0LCByZXF1ZXN0KTtcbiAgICAgIHJldHVybiBoYW5kbGVXaXRoQ2FsbGJhY2soYWRhcHRvciwgcmVzcG9uc2UsIGFzeW5jIChhOiBJbnRlZ3JhdGlvbnNBZGFwdG9yKSA9PiB7XG4gICAgICAgIHJldHVybiBhLmxvYWRJbnRlZ3JhdGlvbkluc3RhbmNlKFxuICAgICAgICAgIHJlcXVlc3QucGFyYW1zLnRlbXBsYXRlTmFtZSxcbiAgICAgICAgICByZXF1ZXN0LmJvZHkubmFtZSxcbiAgICAgICAgICByZXF1ZXN0LmJvZHkuZGF0YVNvdXJjZSxcbiAgICAgICAgICByZXF1ZXN0LmJvZHkud29ya2Zsb3dzXG4gICAgICAgICk7XG4gICAgICB9KTtcbiAgICB9XG4gICk7XG5cbiAgcm91dGVyLmdldChcbiAgICB7XG4gICAgICBwYXRoOiBgJHtJTlRFR1JBVElPTlNfQkFTRX0vcmVwb3NpdG9yeS97bmFtZX1gLFxuICAgICAgdmFsaWRhdGU6IHtcbiAgICAgICAgcGFyYW1zOiBzY2hlbWEub2JqZWN0KHtcbiAgICAgICAgICBuYW1lOiBzY2hlbWEuc3RyaW5nKCksXG4gICAgICAgIH0pLFxuICAgICAgfSxcbiAgICB9LFxuICAgIGFzeW5jIChjb250ZXh0LCByZXF1ZXN0LCByZXNwb25zZSk6IFByb21pc2U8T3BlblNlYXJjaERhc2hib2FyZHNSZXNwb25zZT4gPT4ge1xuICAgICAgY29uc3QgYWRhcHRvciA9IGdldEFkYXB0b3IoY29udGV4dCwgcmVxdWVzdCk7XG4gICAgICByZXR1cm4gaGFuZGxlV2l0aENhbGxiYWNrKFxuICAgICAgICBhZGFwdG9yLFxuICAgICAgICByZXNwb25zZSxcbiAgICAgICAgYXN5bmMgKGE6IEludGVncmF0aW9uc0FkYXB0b3IpID0+XG4gICAgICAgICAgKFxuICAgICAgICAgICAgYXdhaXQgYS5nZXRJbnRlZ3JhdGlvblRlbXBsYXRlcyh7XG4gICAgICAgICAgICAgIG5hbWU6IHJlcXVlc3QucGFyYW1zLm5hbWUsXG4gICAgICAgICAgICB9KVxuICAgICAgICAgICkuaGl0c1swXVxuICAgICAgKTtcbiAgICB9XG4gICk7XG5cbiAgcm91dGVyLmdldChcbiAgICB7XG4gICAgICBwYXRoOiBgJHtJTlRFR1JBVElPTlNfQkFTRX0vcmVwb3NpdG9yeS97aWR9L3N0YXRpYy97cGF0aH1gLFxuICAgICAgdmFsaWRhdGU6IHtcbiAgICAgICAgcGFyYW1zOiBzY2hlbWEub2JqZWN0KHtcbiAgICAgICAgICBpZDogc2NoZW1hLnN0cmluZygpLFxuICAgICAgICAgIHBhdGg6IHNjaGVtYS5zdHJpbmcoKSxcbiAgICAgICAgfSksXG4gICAgICB9LFxuICAgIH0sXG4gICAgYXN5bmMgKGNvbnRleHQsIHJlcXVlc3QsIHJlc3BvbnNlKTogUHJvbWlzZTxPcGVuU2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlPiA9PiB7XG4gICAgICBjb25zdCBhZGFwdG9yID0gZ2V0QWRhcHRvcihjb250ZXh0LCByZXF1ZXN0KTtcbiAgICAgIHRyeSB7XG4gICAgICAgIGNvbnN0IHJlcXVlc3RQYXRoID0gc2FuaXRpemUocmVxdWVzdC5wYXJhbXMucGF0aCk7XG4gICAgICAgIGNvbnN0IHJlc3VsdCA9IGF3YWl0IGFkYXB0b3IuZ2V0U3RhdGljKHJlcXVlc3QucGFyYW1zLmlkLCByZXF1ZXN0UGF0aCk7XG4gICAgICAgIHJldHVybiByZXNwb25zZS5vayh7XG4gICAgICAgICAgaGVhZGVyczoge1xuICAgICAgICAgICAgJ0NvbnRlbnQtVHlwZSc6IG1pbWUuZ2V0VHlwZShyZXF1ZXN0LnBhcmFtcy5wYXRoKSxcbiAgICAgICAgICB9LFxuICAgICAgICAgIGJvZHk6IHJlc3VsdCxcbiAgICAgICAgfSk7XG4gICAgICB9IGNhdGNoIChlcnIpIHtcbiAgICAgICAgcmV0dXJuIHJlc3BvbnNlLmN1c3RvbSh7XG4gICAgICAgICAgc3RhdHVzQ29kZTogZXJyLnN0YXR1c0NvZGUgPyBlcnIuc3RhdHVzQ29kZSA6IDUwMCxcbiAgICAgICAgICBib2R5OiBlcnIubWVzc2FnZSxcbiAgICAgICAgfSk7XG4gICAgICB9XG4gICAgfVxuICApO1xuXG4gIHJvdXRlci5nZXQoXG4gICAge1xuICAgICAgcGF0aDogYCR7SU5URUdSQVRJT05TX0JBU0V9L3JlcG9zaXRvcnkve2lkfS9zY2hlbWFgLFxuICAgICAgdmFsaWRhdGU6IHtcbiAgICAgICAgcGFyYW1zOiBzY2hlbWEub2JqZWN0KHtcbiAgICAgICAgICBpZDogc2NoZW1hLnN0cmluZygpLFxuICAgICAgICB9KSxcbiAgICAgIH0sXG4gICAgfSxcbiAgICBhc3luYyAoY29udGV4dCwgcmVxdWVzdCwgcmVzcG9uc2UpOiBQcm9taXNlPE9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2U+ID0+IHtcbiAgICAgIGNvbnN0IGFkYXB0b3IgPSBnZXRBZGFwdG9yKGNvbnRleHQsIHJlcXVlc3QpO1xuICAgICAgcmV0dXJuIGhhbmRsZVdpdGhDYWxsYmFjayhhZGFwdG9yLCByZXNwb25zZSwgYXN5bmMgKGE6IEludGVncmF0aW9uc0FkYXB0b3IpID0+XG4gICAgICAgIGEuZ2V0U2NoZW1hcyhyZXF1ZXN0LnBhcmFtcy5pZClcbiAgICAgICk7XG4gICAgfVxuICApO1xuXG4gIHJvdXRlci5nZXQoXG4gICAge1xuICAgICAgcGF0aDogYCR7SU5URUdSQVRJT05TX0JBU0V9L3JlcG9zaXRvcnkve2lkfS9hc3NldHNgLFxuICAgICAgdmFsaWRhdGU6IHtcbiAgICAgICAgcGFyYW1zOiBzY2hlbWEub2JqZWN0KHtcbiAgICAgICAgICBpZDogc2NoZW1hLnN0cmluZygpLFxuICAgICAgICB9KSxcbiAgICAgIH0sXG4gICAgfSxcbiAgICBhc3luYyAoY29udGV4dCwgcmVxdWVzdCwgcmVzcG9uc2UpOiBQcm9taXNlPE9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2U+ID0+IHtcbiAgICAgIGNvbnN0IGFkYXB0b3IgPSBnZXRBZGFwdG9yKGNvbnRleHQsIHJlcXVlc3QpO1xuICAgICAgcmV0dXJuIGhhbmRsZVdpdGhDYWxsYmFjayhhZGFwdG9yLCByZXNwb25zZSwgYXN5bmMgKGE6IEludGVncmF0aW9uc0FkYXB0b3IpID0+XG4gICAgICAgIGEuZ2V0QXNzZXRzKHJlcXVlc3QucGFyYW1zLmlkKVxuICAgICAgKTtcbiAgICB9XG4gICk7XG5cbiAgcm91dGVyLmdldChcbiAgICB7XG4gICAgICBwYXRoOiBgJHtJTlRFR1JBVElPTlNfQkFTRX0vcmVwb3NpdG9yeS97aWR9L2RhdGFgLFxuICAgICAgdmFsaWRhdGU6IHtcbiAgICAgICAgcGFyYW1zOiBzY2hlbWEub2JqZWN0KHtcbiAgICAgICAgICBpZDogc2NoZW1hLnN0cmluZygpLFxuICAgICAgICB9KSxcbiAgICAgIH0sXG4gICAgfSxcbiAgICBhc3luYyAoY29udGV4dCwgcmVxdWVzdCwgcmVzcG9uc2UpOiBQcm9taXNlPE9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2U+ID0+IHtcbiAgICAgIGNvbnN0IGFkYXB0b3IgPSBnZXRBZGFwdG9yKGNvbnRleHQsIHJlcXVlc3QpO1xuICAgICAgcmV0dXJuIGhhbmRsZVdpdGhDYWxsYmFjayhhZGFwdG9yLCByZXNwb25zZSwgYXN5bmMgKGE6IEludGVncmF0aW9uc0FkYXB0b3IpID0+XG4gICAgICAgIGEuZ2V0U2FtcGxlRGF0YShyZXF1ZXN0LnBhcmFtcy5pZClcbiAgICAgICk7XG4gICAgfVxuICApO1xuXG4gIHJvdXRlci5nZXQoXG4gICAge1xuICAgICAgcGF0aDogYCR7SU5URUdSQVRJT05TX0JBU0V9L3N0b3JlYCxcbiAgICAgIHZhbGlkYXRlOiBmYWxzZSxcbiAgICB9LFxuICAgIGFzeW5jIChjb250ZXh0LCByZXF1ZXN0LCByZXNwb25zZSk6IFByb21pc2U8T3BlblNlYXJjaERhc2hib2FyZHNSZXNwb25zZT4gPT4ge1xuICAgICAgY29uc3QgYWRhcHRvciA9IGdldEFkYXB0b3IoY29udGV4dCwgcmVxdWVzdCk7XG4gICAgICByZXR1cm4gaGFuZGxlV2l0aENhbGxiYWNrKGFkYXB0b3IsIHJlc3BvbnNlLCBhc3luYyAoYTogSW50ZWdyYXRpb25zQWRhcHRvcikgPT5cbiAgICAgICAgYS5nZXRJbnRlZ3JhdGlvbkluc3RhbmNlcyh7fSlcbiAgICAgICk7XG4gICAgfVxuICApO1xuXG4gIHJvdXRlci5kZWxldGUoXG4gICAge1xuICAgICAgcGF0aDogYCR7SU5URUdSQVRJT05TX0JBU0V9L3N0b3JlL3tpZH1gLFxuICAgICAgdmFsaWRhdGU6IHtcbiAgICAgICAgcGFyYW1zOiBzY2hlbWEub2JqZWN0KHtcbiAgICAgICAgICBpZDogc2NoZW1hLnN0cmluZygpLFxuICAgICAgICB9KSxcbiAgICAgIH0sXG4gICAgfSxcbiAgICBhc3luYyAoY29udGV4dCwgcmVxdWVzdCwgcmVzcG9uc2UpOiBQcm9taXNlPE9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2U+ID0+IHtcbiAgICAgIGNvbnN0IGFkYXB0b3IgPSBnZXRBZGFwdG9yKGNvbnRleHQsIHJlcXVlc3QpO1xuICAgICAgcmV0dXJuIGhhbmRsZVdpdGhDYWxsYmFjayhhZGFwdG9yLCByZXNwb25zZSwgYXN5bmMgKGE6IEludGVncmF0aW9uc0FkYXB0b3IpID0+XG4gICAgICAgIGEuZGVsZXRlSW50ZWdyYXRpb25JbnN0YW5jZShyZXF1ZXN0LnBhcmFtcy5pZClcbiAgICAgICk7XG4gICAgfVxuICApO1xuXG4gIHJvdXRlci5nZXQoXG4gICAge1xuICAgICAgcGF0aDogYCR7SU5URUdSQVRJT05TX0JBU0V9L3N0b3JlL3tpZH1gLFxuICAgICAgdmFsaWRhdGU6IHtcbiAgICAgICAgcGFyYW1zOiBzY2hlbWEub2JqZWN0KHtcbiAgICAgICAgICBpZDogc2NoZW1hLnN0cmluZygpLFxuICAgICAgICB9KSxcbiAgICAgIH0sXG4gICAgfSxcbiAgICBhc3luYyAoY29udGV4dCwgcmVxdWVzdCwgcmVzcG9uc2UpOiBQcm9taXNlPE9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2U+ID0+IHtcbiAgICAgIGNvbnN0IGFkYXB0b3IgPSBnZXRBZGFwdG9yKGNvbnRleHQsIHJlcXVlc3QpO1xuICAgICAgcmV0dXJuIGhhbmRsZVdpdGhDYWxsYmFjayhhZGFwdG9yLCByZXNwb25zZSwgYXN5bmMgKGE6IEludGVncmF0aW9uc0FkYXB0b3IpID0+XG4gICAgICAgIGEuZ2V0SW50ZWdyYXRpb25JbnN0YW5jZSh7XG4gICAgICAgICAgaWQ6IHJlcXVlc3QucGFyYW1zLmlkLFxuICAgICAgICB9KVxuICAgICAgKTtcbiAgICB9XG4gICk7XG59XG4iXSwibWFwcGluZ3MiOiI7Ozs7Ozs7QUFLQSxJQUFBQSxhQUFBLEdBQUFDLE9BQUE7QUFDQSxJQUFBQyxJQUFBLEdBQUFDLHVCQUFBLENBQUFGLE9BQUE7QUFDQSxJQUFBRyxpQkFBQSxHQUFBQyxzQkFBQSxDQUFBSixPQUFBO0FBRUEsSUFBQUssT0FBQSxHQUFBTCxPQUFBO0FBT0EsSUFBQU0scUJBQUEsR0FBQU4sT0FBQTtBQUF1RixTQUFBSSx1QkFBQUcsR0FBQSxXQUFBQSxHQUFBLElBQUFBLEdBQUEsQ0FBQUMsVUFBQSxHQUFBRCxHQUFBLEtBQUFFLE9BQUEsRUFBQUYsR0FBQTtBQUFBLFNBQUFHLHlCQUFBQyxDQUFBLDZCQUFBQyxPQUFBLG1CQUFBQyxDQUFBLE9BQUFELE9BQUEsSUFBQUUsQ0FBQSxPQUFBRixPQUFBLFlBQUFGLHdCQUFBLFlBQUFBLENBQUFDLENBQUEsV0FBQUEsQ0FBQSxHQUFBRyxDQUFBLEdBQUFELENBQUEsS0FBQUYsQ0FBQTtBQUFBLFNBQUFULHdCQUFBUyxDQUFBLEVBQUFFLENBQUEsU0FBQUEsQ0FBQSxJQUFBRixDQUFBLElBQUFBLENBQUEsQ0FBQUgsVUFBQSxTQUFBRyxDQUFBLGVBQUFBLENBQUEsdUJBQUFBLENBQUEseUJBQUFBLENBQUEsV0FBQUYsT0FBQSxFQUFBRSxDQUFBLFFBQUFHLENBQUEsR0FBQUosd0JBQUEsQ0FBQUcsQ0FBQSxPQUFBQyxDQUFBLElBQUFBLENBQUEsQ0FBQUMsR0FBQSxDQUFBSixDQUFBLFVBQUFHLENBQUEsQ0FBQUUsR0FBQSxDQUFBTCxDQUFBLE9BQUFNLENBQUEsS0FBQUMsU0FBQSxVQUFBQyxDQUFBLEdBQUFDLE1BQUEsQ0FBQUMsY0FBQSxJQUFBRCxNQUFBLENBQUFFLHdCQUFBLFdBQUFDLENBQUEsSUFBQVosQ0FBQSxvQkFBQVksQ0FBQSxJQUFBSCxNQUFBLENBQUFJLFNBQUEsQ0FBQUMsY0FBQSxDQUFBQyxJQUFBLENBQUFmLENBQUEsRUFBQVksQ0FBQSxTQUFBSSxDQUFBLEdBQUFSLENBQUEsR0FBQUMsTUFBQSxDQUFBRSx3QkFBQSxDQUFBWCxDQUFBLEVBQUFZLENBQUEsVUFBQUksQ0FBQSxLQUFBQSxDQUFBLENBQUFYLEdBQUEsSUFBQVcsQ0FBQSxDQUFBQyxHQUFBLElBQUFSLE1BQUEsQ0FBQUMsY0FBQSxDQUFBSixDQUFBLEVBQUFNLENBQUEsRUFBQUksQ0FBQSxJQUFBVixDQUFBLENBQUFNLENBQUEsSUFBQVosQ0FBQSxDQUFBWSxDQUFBLFlBQUFOLENBQUEsQ0FBQVIsT0FBQSxHQUFBRSxDQUFBLEVBQUFHLENBQUEsSUFBQUEsQ0FBQSxDQUFBYyxHQUFBLENBQUFqQixDQUFBLEVBQUFNLENBQUEsR0FBQUEsQ0FBQTtBQWhCdkY7QUFDQTtBQUNBO0FBQ0E7O0FBZUE7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNPLE1BQU1ZLGtCQUFrQixHQUFHLE1BQUFBLENBQ2hDQyxPQUE0QixFQUM1QkMsUUFBNkMsRUFDN0NDLFFBQWdELEtBQ2dCO0VBQ2hFLElBQUk7SUFDRixNQUFNQyxJQUFJLEdBQUcsTUFBTUQsUUFBUSxDQUFDRixPQUFPLENBQUM7SUFDcEMsT0FBT0MsUUFBUSxDQUFDRyxFQUFFLENBQUM7TUFDakJDLElBQUksRUFBRTtRQUNKRjtNQUNGO0lBQ0YsQ0FBQyxDQUFDO0VBQ0osQ0FBQyxDQUFDLE9BQU9HLEdBQUcsRUFBRTtJQUNaQyxPQUFPLENBQUNDLEtBQUssQ0FBRSxtREFBa0RGLEdBQUcsQ0FBQ0csT0FBUSxHQUFFLENBQUM7SUFDaEYsT0FBT1IsUUFBUSxDQUFDUyxNQUFNLENBQUM7TUFDckJDLFVBQVUsRUFBRUwsR0FBRyxDQUFDSyxVQUFVLElBQUksR0FBRztNQUNqQ04sSUFBSSxFQUFFQyxHQUFHLENBQUNHO0lBQ1osQ0FBQyxDQUFDO0VBQ0o7QUFDRixDQUFDO0FBQUNHLE9BQUEsQ0FBQWIsa0JBQUEsR0FBQUEsa0JBQUE7QUFFRixNQUFNYyxVQUFVLEdBQUdBLENBQ2pCQyxPQUE4QixFQUM5QkMsUUFBcUMsS0FDYjtFQUN4QixPQUFPLElBQUlDLHlDQUFtQixDQUFDRixPQUFPLENBQUNHLElBQUksQ0FBQ0MsWUFBWSxDQUFDQyxNQUFNLENBQUM7QUFDbEUsQ0FBQztBQUVNLFNBQVNDLHlCQUF5QkEsQ0FBQ0MsTUFBZSxFQUFFO0VBQ3pEQSxNQUFNLENBQUNuQyxHQUFHLENBQ1I7SUFDRW9DLElBQUksRUFBRyxHQUFFQyx5QkFBa0IsYUFBWTtJQUN2Q0MsUUFBUSxFQUFFO0VBQ1osQ0FBQyxFQUNELE9BQU9WLE9BQU8sRUFBRVcsT0FBTyxFQUFFeEIsUUFBUSxLQUE0QztJQUMzRSxNQUFNRCxPQUFPLEdBQUdhLFVBQVUsQ0FBQ0MsT0FBTyxFQUFFVyxPQUFPLENBQUM7SUFDNUMsT0FBTzFCLGtCQUFrQixDQUFDQyxPQUFPLEVBQUVDLFFBQVEsRUFBRSxNQUFPWixDQUFzQixJQUN4RUEsQ0FBQyxDQUFDcUMsdUJBQXVCLENBQUMsQ0FDNUIsQ0FBQztFQUNILENBQ0YsQ0FBQztFQUVETCxNQUFNLENBQUNNLElBQUksQ0FDVDtJQUNFTCxJQUFJLEVBQUcsR0FBRUMseUJBQWtCLHVCQUFzQjtJQUNqREMsUUFBUSxFQUFFO01BQ1JJLE1BQU0sRUFBRUMsb0JBQU0sQ0FBQ0MsTUFBTSxDQUFDO1FBQ3BCQyxZQUFZLEVBQUVGLG9CQUFNLENBQUNHLE1BQU0sQ0FBQztNQUM5QixDQUFDLENBQUM7TUFDRjNCLElBQUksRUFBRXdCLG9CQUFNLENBQUNDLE1BQU0sQ0FBQztRQUNsQkcsSUFBSSxFQUFFSixvQkFBTSxDQUFDRyxNQUFNLENBQUMsQ0FBQztRQUNyQkUsVUFBVSxFQUFFTCxvQkFBTSxDQUFDRyxNQUFNLENBQUMsQ0FBQztRQUMzQkcsU0FBUyxFQUFFTixvQkFBTSxDQUFDTyxLQUFLLENBQUNQLG9CQUFNLENBQUNRLE9BQU8sQ0FBQ1Isb0JBQU0sQ0FBQ0csTUFBTSxDQUFDLENBQUMsQ0FBQztNQUN6RCxDQUFDO0lBQ0g7RUFDRixDQUFDLEVBQ0QsT0FBT2xCLE9BQU8sRUFBRVcsT0FBTyxFQUFFeEIsUUFBUSxLQUE0QztJQUMzRSxNQUFNRCxPQUFPLEdBQUdhLFVBQVUsQ0FBQ0MsT0FBTyxFQUFFVyxPQUFPLENBQUM7SUFDNUMsT0FBTzFCLGtCQUFrQixDQUFDQyxPQUFPLEVBQUVDLFFBQVEsRUFBRSxNQUFPWixDQUFzQixJQUFLO01BQzdFLE9BQU9BLENBQUMsQ0FBQ2lELHVCQUF1QixDQUM5QmIsT0FBTyxDQUFDRyxNQUFNLENBQUNHLFlBQVksRUFDM0JOLE9BQU8sQ0FBQ3BCLElBQUksQ0FBQzRCLElBQUksRUFDakJSLE9BQU8sQ0FBQ3BCLElBQUksQ0FBQzZCLFVBQVUsRUFDdkJULE9BQU8sQ0FBQ3BCLElBQUksQ0FBQzhCLFNBQ2YsQ0FBQztJQUNILENBQUMsQ0FBQztFQUNKLENBQ0YsQ0FBQztFQUVEZCxNQUFNLENBQUNuQyxHQUFHLENBQ1I7SUFDRW9DLElBQUksRUFBRyxHQUFFQyx5QkFBa0Isb0JBQW1CO0lBQzlDQyxRQUFRLEVBQUU7TUFDUkksTUFBTSxFQUFFQyxvQkFBTSxDQUFDQyxNQUFNLENBQUM7UUFDcEJHLElBQUksRUFBRUosb0JBQU0sQ0FBQ0csTUFBTSxDQUFDO01BQ3RCLENBQUM7SUFDSDtFQUNGLENBQUMsRUFDRCxPQUFPbEIsT0FBTyxFQUFFVyxPQUFPLEVBQUV4QixRQUFRLEtBQTRDO0lBQzNFLE1BQU1ELE9BQU8sR0FBR2EsVUFBVSxDQUFDQyxPQUFPLEVBQUVXLE9BQU8sQ0FBQztJQUM1QyxPQUFPMUIsa0JBQWtCLENBQ3ZCQyxPQUFPLEVBQ1BDLFFBQVEsRUFDUixNQUFPWixDQUFzQixJQUMzQixDQUNFLE1BQU1BLENBQUMsQ0FBQ3FDLHVCQUF1QixDQUFDO01BQzlCTyxJQUFJLEVBQUVSLE9BQU8sQ0FBQ0csTUFBTSxDQUFDSztJQUN2QixDQUFDLENBQUMsRUFDRk0sSUFBSSxDQUFDLENBQUMsQ0FDWixDQUFDO0VBQ0gsQ0FDRixDQUFDO0VBRURsQixNQUFNLENBQUNuQyxHQUFHLENBQ1I7SUFDRW9DLElBQUksRUFBRyxHQUFFQyx5QkFBa0IsZ0NBQStCO0lBQzFEQyxRQUFRLEVBQUU7TUFDUkksTUFBTSxFQUFFQyxvQkFBTSxDQUFDQyxNQUFNLENBQUM7UUFDcEJVLEVBQUUsRUFBRVgsb0JBQU0sQ0FBQ0csTUFBTSxDQUFDLENBQUM7UUFDbkJWLElBQUksRUFBRU8sb0JBQU0sQ0FBQ0csTUFBTSxDQUFDO01BQ3RCLENBQUM7SUFDSDtFQUNGLENBQUMsRUFDRCxPQUFPbEIsT0FBTyxFQUFFVyxPQUFPLEVBQUV4QixRQUFRLEtBQTRDO0lBQzNFLE1BQU1ELE9BQU8sR0FBR2EsVUFBVSxDQUFDQyxPQUFPLEVBQUVXLE9BQU8sQ0FBQztJQUM1QyxJQUFJO01BQ0YsTUFBTWdCLFdBQVcsR0FBRyxJQUFBQyx5QkFBUSxFQUFDakIsT0FBTyxDQUFDRyxNQUFNLENBQUNOLElBQUksQ0FBQztNQUNqRCxNQUFNcUIsTUFBTSxHQUFHLE1BQU0zQyxPQUFPLENBQUM0QyxTQUFTLENBQUNuQixPQUFPLENBQUNHLE1BQU0sQ0FBQ1ksRUFBRSxFQUFFQyxXQUFXLENBQUM7TUFDdEUsT0FBT3hDLFFBQVEsQ0FBQ0csRUFBRSxDQUFDO1FBQ2pCeUMsT0FBTyxFQUFFO1VBQ1AsY0FBYyxFQUFFMUUsSUFBSSxDQUFDMkUsT0FBTyxDQUFDckIsT0FBTyxDQUFDRyxNQUFNLENBQUNOLElBQUk7UUFDbEQsQ0FBQztRQUNEakIsSUFBSSxFQUFFc0M7TUFDUixDQUFDLENBQUM7SUFDSixDQUFDLENBQUMsT0FBT3JDLEdBQUcsRUFBRTtNQUNaLE9BQU9MLFFBQVEsQ0FBQ1MsTUFBTSxDQUFDO1FBQ3JCQyxVQUFVLEVBQUVMLEdBQUcsQ0FBQ0ssVUFBVSxHQUFHTCxHQUFHLENBQUNLLFVBQVUsR0FBRyxHQUFHO1FBQ2pETixJQUFJLEVBQUVDLEdBQUcsQ0FBQ0c7TUFDWixDQUFDLENBQUM7SUFDSjtFQUNGLENBQ0YsQ0FBQztFQUVEWSxNQUFNLENBQUNuQyxHQUFHLENBQ1I7SUFDRW9DLElBQUksRUFBRyxHQUFFQyx5QkFBa0IseUJBQXdCO0lBQ25EQyxRQUFRLEVBQUU7TUFDUkksTUFBTSxFQUFFQyxvQkFBTSxDQUFDQyxNQUFNLENBQUM7UUFDcEJVLEVBQUUsRUFBRVgsb0JBQU0sQ0FBQ0csTUFBTSxDQUFDO01BQ3BCLENBQUM7SUFDSDtFQUNGLENBQUMsRUFDRCxPQUFPbEIsT0FBTyxFQUFFVyxPQUFPLEVBQUV4QixRQUFRLEtBQTRDO0lBQzNFLE1BQU1ELE9BQU8sR0FBR2EsVUFBVSxDQUFDQyxPQUFPLEVBQUVXLE9BQU8sQ0FBQztJQUM1QyxPQUFPMUIsa0JBQWtCLENBQUNDLE9BQU8sRUFBRUMsUUFBUSxFQUFFLE1BQU9aLENBQXNCLElBQ3hFQSxDQUFDLENBQUMwRCxVQUFVLENBQUN0QixPQUFPLENBQUNHLE1BQU0sQ0FBQ1ksRUFBRSxDQUNoQyxDQUFDO0VBQ0gsQ0FDRixDQUFDO0VBRURuQixNQUFNLENBQUNuQyxHQUFHLENBQ1I7SUFDRW9DLElBQUksRUFBRyxHQUFFQyx5QkFBa0IseUJBQXdCO0lBQ25EQyxRQUFRLEVBQUU7TUFDUkksTUFBTSxFQUFFQyxvQkFBTSxDQUFDQyxNQUFNLENBQUM7UUFDcEJVLEVBQUUsRUFBRVgsb0JBQU0sQ0FBQ0csTUFBTSxDQUFDO01BQ3BCLENBQUM7SUFDSDtFQUNGLENBQUMsRUFDRCxPQUFPbEIsT0FBTyxFQUFFVyxPQUFPLEVBQUV4QixRQUFRLEtBQTRDO0lBQzNFLE1BQU1ELE9BQU8sR0FBR2EsVUFBVSxDQUFDQyxPQUFPLEVBQUVXLE9BQU8sQ0FBQztJQUM1QyxPQUFPMUIsa0JBQWtCLENBQUNDLE9BQU8sRUFBRUMsUUFBUSxFQUFFLE1BQU9aLENBQXNCLElBQ3hFQSxDQUFDLENBQUMyRCxTQUFTLENBQUN2QixPQUFPLENBQUNHLE1BQU0sQ0FBQ1ksRUFBRSxDQUMvQixDQUFDO0VBQ0gsQ0FDRixDQUFDO0VBRURuQixNQUFNLENBQUNuQyxHQUFHLENBQ1I7SUFDRW9DLElBQUksRUFBRyxHQUFFQyx5QkFBa0IsdUJBQXNCO0lBQ2pEQyxRQUFRLEVBQUU7TUFDUkksTUFBTSxFQUFFQyxvQkFBTSxDQUFDQyxNQUFNLENBQUM7UUFDcEJVLEVBQUUsRUFBRVgsb0JBQU0sQ0FBQ0csTUFBTSxDQUFDO01BQ3BCLENBQUM7SUFDSDtFQUNGLENBQUMsRUFDRCxPQUFPbEIsT0FBTyxFQUFFVyxPQUFPLEVBQUV4QixRQUFRLEtBQTRDO0lBQzNFLE1BQU1ELE9BQU8sR0FBR2EsVUFBVSxDQUFDQyxPQUFPLEVBQUVXLE9BQU8sQ0FBQztJQUM1QyxPQUFPMUIsa0JBQWtCLENBQUNDLE9BQU8sRUFBRUMsUUFBUSxFQUFFLE1BQU9aLENBQXNCLElBQ3hFQSxDQUFDLENBQUM0RCxhQUFhLENBQUN4QixPQUFPLENBQUNHLE1BQU0sQ0FBQ1ksRUFBRSxDQUNuQyxDQUFDO0VBQ0gsQ0FDRixDQUFDO0VBRURuQixNQUFNLENBQUNuQyxHQUFHLENBQ1I7SUFDRW9DLElBQUksRUFBRyxHQUFFQyx5QkFBa0IsUUFBTztJQUNsQ0MsUUFBUSxFQUFFO0VBQ1osQ0FBQyxFQUNELE9BQU9WLE9BQU8sRUFBRVcsT0FBTyxFQUFFeEIsUUFBUSxLQUE0QztJQUMzRSxNQUFNRCxPQUFPLEdBQUdhLFVBQVUsQ0FBQ0MsT0FBTyxFQUFFVyxPQUFPLENBQUM7SUFDNUMsT0FBTzFCLGtCQUFrQixDQUFDQyxPQUFPLEVBQUVDLFFBQVEsRUFBRSxNQUFPWixDQUFzQixJQUN4RUEsQ0FBQyxDQUFDNkQsdUJBQXVCLENBQUMsQ0FBQyxDQUFDLENBQzlCLENBQUM7RUFDSCxDQUNGLENBQUM7RUFFRDdCLE1BQU0sQ0FBQzhCLE1BQU0sQ0FDWDtJQUNFN0IsSUFBSSxFQUFHLEdBQUVDLHlCQUFrQixhQUFZO0lBQ3ZDQyxRQUFRLEVBQUU7TUFDUkksTUFBTSxFQUFFQyxvQkFBTSxDQUFDQyxNQUFNLENBQUM7UUFDcEJVLEVBQUUsRUFBRVgsb0JBQU0sQ0FBQ0csTUFBTSxDQUFDO01BQ3BCLENBQUM7SUFDSDtFQUNGLENBQUMsRUFDRCxPQUFPbEIsT0FBTyxFQUFFVyxPQUFPLEVBQUV4QixRQUFRLEtBQTRDO0lBQzNFLE1BQU1ELE9BQU8sR0FBR2EsVUFBVSxDQUFDQyxPQUFPLEVBQUVXLE9BQU8sQ0FBQztJQUM1QyxPQUFPMUIsa0JBQWtCLENBQUNDLE9BQU8sRUFBRUMsUUFBUSxFQUFFLE1BQU9aLENBQXNCLElBQ3hFQSxDQUFDLENBQUMrRCx5QkFBeUIsQ0FBQzNCLE9BQU8sQ0FBQ0csTUFBTSxDQUFDWSxFQUFFLENBQy9DLENBQUM7RUFDSCxDQUNGLENBQUM7RUFFRG5CLE1BQU0sQ0FBQ25DLEdBQUcsQ0FDUjtJQUNFb0MsSUFBSSxFQUFHLEdBQUVDLHlCQUFrQixhQUFZO0lBQ3ZDQyxRQUFRLEVBQUU7TUFDUkksTUFBTSxFQUFFQyxvQkFBTSxDQUFDQyxNQUFNLENBQUM7UUFDcEJVLEVBQUUsRUFBRVgsb0JBQU0sQ0FBQ0csTUFBTSxDQUFDO01BQ3BCLENBQUM7SUFDSDtFQUNGLENBQUMsRUFDRCxPQUFPbEIsT0FBTyxFQUFFVyxPQUFPLEVBQUV4QixRQUFRLEtBQTRDO0lBQzNFLE1BQU1ELE9BQU8sR0FBR2EsVUFBVSxDQUFDQyxPQUFPLEVBQUVXLE9BQU8sQ0FBQztJQUM1QyxPQUFPMUIsa0JBQWtCLENBQUNDLE9BQU8sRUFBRUMsUUFBUSxFQUFFLE1BQU9aLENBQXNCLElBQ3hFQSxDQUFDLENBQUNnRSxzQkFBc0IsQ0FBQztNQUN2QmIsRUFBRSxFQUFFZixPQUFPLENBQUNHLE1BQU0sQ0FBQ1k7SUFDckIsQ0FBQyxDQUNILENBQUM7RUFDSCxDQUNGLENBQUM7QUFDSCJ9