"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.IntegrationReader = void 0;
var _path = _interopRequireDefault(require("path"));
var _semver = _interopRequireDefault(require("semver"));
var _validators = require("../validators");
var _fs_data_adaptor = require("./fs_data_adaptor");
var _utils = require("./utils");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
/**
 * The Integration class represents the data for Integration Templates.
 * It is backed by the repository file system.
 * It includes accessor methods for integration configs, as well as helpers for nested components.
 */
class IntegrationReader {
  constructor(directory, reader) {
    _defineProperty(this, "reader", void 0);
    _defineProperty(this, "directory", void 0);
    _defineProperty(this, "name", void 0);
    this.directory = directory;
    this.name = _path.default.basename(directory);
    this.reader = reader !== null && reader !== void 0 ? reader : new _fs_data_adaptor.FileSystemDataAdaptor(directory);
  }

  /**
   * Retrieve data from correct source regardless of if reader is config-localized or not.
   *
   * TODO refactor to assemble filename from `type` instead of requiring caller to format it.
   *
   * @param item An item which may have data in it.
   * @param fileParams Information about the file to read if the config is not localized.
   * @param format How to package the returned data.
   *               If 'json', return `object | object[]`. If 'binary', return `Buffer`.
   * @returns A result with the data, with a format based on the format field.
   */

  async fetchDataOrReadFile(item, fileParams, format) {
    if (this.reader.isConfigLocalized) {
      if (!item.data) {
        return {
          ok: false,
          error: new Error('The config for the provided reader is localized, but no data field is present. ' + JSON.stringify(item))
        };
      }
      try {
        if (format === 'json') {
          return {
            ok: true,
            value: JSON.parse(item.data)
          };
        } else {
          return {
            ok: true,
            value: Buffer.from(item.data, 'base64')
          };
        }
      } catch (error) {
        return {
          ok: false,
          error
        };
      }
    }
    if (format === 'json') {
      return this.reader.readFile(fileParams.filename, fileParams.type);
    } else {
      return this.reader.readFileRaw(fileParams.filename, fileParams.type);
    }
  }
  async readAsset(asset) {
    const filename = `${asset.name}-${asset.version}.${asset.extension}`;
    const fileParams = {
      filename,
      type: 'assets'
    };
    if (['json', 'ndjson'].includes(asset.extension)) {
      const maybeObject = await this.fetchDataOrReadFile(asset, fileParams, 'json');
      if (!maybeObject.ok) {
        return maybeObject;
      }
      return {
        ok: true,
        value: {
          ...asset,
          data: JSON.stringify(maybeObject.value)
        }
      };
    } else {
      const maybeBuffer = await this.fetchDataOrReadFile(asset, fileParams, 'binary');
      if (!maybeBuffer.ok) {
        return maybeBuffer;
      }
      return {
        ok: true,
        value: {
          ...asset,
          data: JSON.stringify(maybeBuffer.value.toString('utf8'))
        }
      };
    }
  }

  /**
   * Get the latest version of the integration available.
   * This method relies on the fact that integration configs have their versions in their name.
   * Any files that don't match the config naming convention will be ignored.
   *
   * @returns A string with the latest version, or null if no versions are available.
   */
  async getLatestVersion() {
    const versions = await this.reader.findIntegrationVersions();
    if (!versions.ok) {
      return null;
    }
    if (versions.value.length === 0) {
      return null;
    }
    // Sort descending
    versions.value.sort(_semver.default.rcompare);
    return versions.value[0];
  }

  // Get config without pruning or validation.
  async getRawConfig(version) {
    if ((await this.reader.getDirectoryType()) !== 'integration') {
      return {
        ok: false,
        error: new Error(`${this.directory} is not a valid integration directory`)
      };
    }
    const maybeVersion = version ? version : await this.getLatestVersion();
    if (maybeVersion === null) {
      return {
        ok: false,
        error: new Error(`No valid config matching version ${version} is available`)
      };
    }
    const configFile = `${this.name}-${maybeVersion}.json`;

    // Even config-localized readers must support config-read.
    const config = await this.reader.readFile(configFile);
    if (!config.ok) {
      return config;
    }
    return (0, _validators.validateTemplate)(config.value);
  }

  /**
   * Get the configuration of the current integration.
   *
   * @param version The version of the config to retrieve.
   * @returns The config if a valid config matching the version is present, otherwise null.
   */
  async getConfig(version) {
    const maybeConfig = await this.getRawConfig(version);
    if (!maybeConfig.ok) {
      return maybeConfig;
    }
    return (0, _validators.validateTemplate)((0, _utils.pruneConfig)(maybeConfig.value));
  }

  /**
   * Retrieve assets associated with the integration.
   * This method greedily retrieves all assets.
   * If an asset is invalid, an error result is returned.
   *
   * @param version The version of the integration to retrieve assets for.
   * @returns A result containing the parsed assets.
   */
  async getAssets(version) {
    const configResult = await this.getRawConfig(version);
    if (!configResult.ok) {
      return configResult;
    }
    const config = configResult.value;
    const resultValue = [];
    for (const asset of config.assets) {
      const serializedResult = await this.readAsset(asset);
      if (!serializedResult.ok) {
        return serializedResult;
      }
      switch (asset.type) {
        case 'savedObjectBundle':
          resultValue.push({
            type: 'savedObjectBundle',
            workflows: asset.workflows,
            data: JSON.parse(serializedResult.value.data)
          });
          break;
        case 'query':
          resultValue.push({
            type: 'query',
            workflows: asset.workflows,
            query: serializedResult.value.data,
            language: asset.extension
          });
          break;
      }
    }
    return {
      ok: true,
      value: resultValue
    };
  }

  /**
   * Retrieve sample data associated with the integration.
   * If the version is invalid, an error is thrown.
   * If the sample data is invalid, null will be returned
   *
   * @param version The version of the integration to retrieve assets for.
   * @returns An object containing a list of sample data with adjusted timestamps.
   */
  async getSampleData(version) {
    const configResult = await this.getRawConfig(version);
    if (!configResult.ok) {
      return configResult;
    }
    const config = configResult.value;
    const resultValue = {
      sampleData: null
    };
    if (config.sampleData) {
      const jsonContent = await this.fetchDataOrReadFile(config.sampleData, {
        filename: config.sampleData.path,
        type: 'data'
      }, 'json');
      if (!jsonContent.ok) {
        return jsonContent;
      }
      for (const value of jsonContent.value) {
        if (!('@timestamp' in value)) {
          continue;
        }
        // Randomly scatter timestamps across last 10 minutes
        // Assume for now that the ordering of events isn't important, can change to a sequence if needed
        // Also doesn't handle fields like `observedTimestamp` if present
        const newTime = new Date(Date.now() - Math.floor(Math.random() * 1000 * 60 * 10)).toISOString();
        Object.assign(value, {
          '@timestamp': newTime
        });
        if ('observedTimestamp' in value) {
          Object.assign(value, {
            observedTimestamp: newTime
          });
        }
      }
      resultValue.sampleData = jsonContent.value;
    }
    return {
      ok: true,
      value: resultValue
    };
  }

  /**
   * Retrieve schema data associated with the integration.
   * This method greedily retrieves all mappings and schemas.
   * It's assumed that a valid version will be provided.
   * If the version is invalid, an error is thrown.
   * If a schema is invalid, an error will be thrown.
   *
   * @param version The version of the integration to retrieve assets for.
   * @returns An object containing the different types of assets.
   */
  async getSchemas(version) {
    const configResult = await this.getRawConfig(version);
    if (!configResult.ok) {
      return configResult;
    }
    const config = configResult.value;
    const resultValue = {
      mappings: {}
    };
    for (const component of config.components) {
      const schemaFile = `${component.name}-${component.version}.mapping.json`;
      const schema = await this.fetchDataOrReadFile(component, {
        filename: schemaFile,
        type: 'schemas'
      }, 'json');
      if (!schema.ok) {
        return schema;
      }
      resultValue.mappings[component.name] = schema.value;
    }
    return {
      ok: true,
      value: resultValue
    };
  }

  /**
   * Retrieves the data for a static file associated with the integration.
   *
   * @param staticPath The path of the static to retrieve.
   * @returns A buffer with the static's data if present, otherwise null.
   */
  async getStatic(staticPath) {
    // Statics were originally designed to read straight from file system,
    // so we use direct access if possible.
    if (!this.reader.isConfigLocalized) {
      return await this.reader.readFileRaw(staticPath, 'static');
    }

    // Otherwise, we need to search for the right static, by checking each version.
    const versions = await this.reader.findIntegrationVersions();
    if (!versions.ok) {
      return versions;
    }
    for (const version of versions.value) {
      var _statics$logo, _statics$darkModeLogo;
      const config = await this.getRawConfig(version);
      if (!config.ok || !config.value.statics) {
        continue;
      }
      const statics = config.value.statics;
      if (((_statics$logo = statics.logo) === null || _statics$logo === void 0 ? void 0 : _statics$logo.path) === staticPath) {
        if (!('data' in statics.logo)) {
          return {
            ok: false,
            error: new Error('Localized config missing static data')
          };
        }
        return {
          ok: true,
          value: Buffer.from(statics.logo.data, 'base64')
        };
      }
      if ((statics === null || statics === void 0 || (_statics$darkModeLogo = statics.darkModeLogo) === null || _statics$darkModeLogo === void 0 ? void 0 : _statics$darkModeLogo.path) === staticPath) {
        if (!('data' in statics.darkModeLogo)) {
          return {
            ok: false,
            error: new Error('Localized config missing static data')
          };
        }
        return {
          ok: true,
          value: Buffer.from(statics.darkModeLogo.data, 'base64')
        };
      }
      for (const iterStatic of [...((_statics$gallery = statics === null || statics === void 0 ? void 0 : statics.gallery) !== null && _statics$gallery !== void 0 ? _statics$gallery : []), ...((_statics$darkModeGall = statics === null || statics === void 0 ? void 0 : statics.darkModeGallery) !== null && _statics$darkModeGall !== void 0 ? _statics$darkModeGall : [])]) {
        var _statics$gallery, _statics$darkModeGall;
        if (iterStatic.path === staticPath) {
          if (!('data' in iterStatic)) {
            return {
              ok: false,
              error: new Error('Localized config missing static data')
            };
          }
          return {
            ok: true,
            value: Buffer.from(iterStatic.data, 'base64')
          };
        }
      }
    }
    return {
      ok: false,
      error: new Error(`Static not found: ${staticPath}`, {
        code: 'ENOENT'
      })
    };
  }
  async serializeStaticAsset(asset) {
    const data = await this.getStatic(asset.path);
    if (!data.ok) {
      return data;
    }
    return {
      ok: true,
      value: {
        ...asset,
        data: data.value.toString('base64')
      }
    };
  }
  async serializeStatics(statics) {
    const serialized = {};
    if (statics.logo) {
      const serializeResult = await this.serializeStaticAsset(statics.logo);
      serialized.logo = serializeResult.value;
    }
    if (statics.darkModeLogo) {
      const serializeResult = await this.serializeStaticAsset(statics.darkModeLogo);
      serialized.darkModeLogo = serializeResult.value;
    }
    if (statics.gallery) {
      const results = await Promise.all(statics.gallery.map(asset => this.serializeStaticAsset(asset)));
      const foldedResult = (0, _utils.foldResults)(results);
      serialized.gallery = foldedResult.value;
    }
    if (statics.darkModeGallery) {
      const results = await Promise.all(statics.darkModeGallery.map(asset => this.serializeStaticAsset(asset)));
      const foldedResult = (0, _utils.foldResults)(results);
      serialized.darkModeGallery = foldedResult.value;
    }
    return {
      ok: true,
      value: serialized
    };
  }

  /**
   * Serialize the referenced integration as a flat JSON object.
   * Useful for normalizing the format for sending to other locations.
   * This method implements the serialization scheme expected by `JsonCatalogDataAdaptor`.
   *
   * @param version The version of the integration to serialize.
   * @returns A large object which includes all of the integration's data.
   */
  async serialize(version) {
    const configResult = await this.getRawConfig(version);
    if (!configResult.ok) {
      return configResult;
    }

    // Type cast safety: all serializable properties must have the 'data' field.
    // The remainder of the method is populating all such fields.
    const config = configResult.value;
    const componentResults = await Promise.all(config.components.map(component => this.fetchDataOrReadFile(component, {
      filename: `${component.name}-${component.version}.mapping.json`,
      type: 'schemas'
    }, 'json')));
    const componentsResult = (0, _utils.foldResults)(componentResults);
    if (!componentsResult.ok) {
      return componentsResult;
    }
    config.components = config.components.map((component, idx) => {
      return {
        ...component,
        data: JSON.stringify(componentsResult.value[idx])
      };
    });
    const assetResults = await Promise.all(config.assets.map(asset => this.readAsset(asset)));
    const assets = (0, _utils.foldResults)(assetResults);
    if (!assets.ok) {
      return assets;
    }
    config.assets = assets.value;
    if (config.statics) {
      const staticsResult = await this.serializeStatics(config.statics);
      if (!staticsResult.ok) {
        return staticsResult;
      }
      config.statics = staticsResult.value;
    }
    if (config.sampleData) {
      const dataResult = await this.getSampleData(version);
      if (!dataResult.ok) {
        return dataResult;
      }
      config.sampleData = {
        ...config.sampleData,
        data: JSON.stringify(dataResult.value.sampleData)
      };
    }
    return {
      ok: true,
      value: config
    };
  }
}
exports.IntegrationReader = IntegrationReader;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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