"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateTemplate = exports.validateInstance = void 0;
var _ajv = _interopRequireDefault(require("ajv"));
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const ajv = new _ajv.default();
const staticAsset = {
  type: 'object',
  properties: {
    path: {
      type: 'string'
    },
    annotation: {
      type: 'string',
      nullable: true
    },
    data: {
      type: 'string',
      nullable: true
    }
  },
  required: ['path'],
  additionalProperties: false
};
const templateSchema = {
  type: 'object',
  properties: {
    name: {
      type: 'string'
    },
    version: {
      type: 'string'
    },
    displayName: {
      type: 'string',
      nullable: true
    },
    license: {
      type: 'string'
    },
    type: {
      type: 'string'
    },
    labels: {
      type: 'array',
      items: {
        type: 'string'
      },
      nullable: true
    },
    tags: {
      type: 'array',
      items: {
        type: 'string'
      },
      nullable: true
    },
    author: {
      type: 'string',
      nullable: true
    },
    description: {
      type: 'string',
      nullable: true
    },
    sourceUrl: {
      type: 'string',
      nullable: true
    },
    workflows: {
      type: 'array',
      items: {
        type: 'object',
        properties: {
          name: {
            type: 'string'
          },
          label: {
            type: 'string'
          },
          description: {
            type: 'string'
          },
          enabled_by_default: {
            type: 'boolean'
          }
        },
        required: ['name', 'label', 'description', 'enabled_by_default']
      },
      nullable: true
    },
    statics: {
      type: 'object',
      properties: {
        logo: {
          ...staticAsset,
          nullable: true
        },
        gallery: {
          type: 'array',
          items: staticAsset,
          nullable: true
        },
        darkModeLogo: {
          ...staticAsset,
          nullable: true
        },
        darkModeGallery: {
          type: 'array',
          items: staticAsset,
          nullable: true
        }
      },
      additionalProperties: false,
      nullable: true
    },
    components: {
      type: 'array',
      items: {
        type: 'object',
        properties: {
          name: {
            type: 'string'
          },
          version: {
            type: 'string'
          },
          data: {
            type: 'string',
            nullable: true
          }
        },
        required: ['name', 'version']
      }
    },
    assets: {
      type: 'array',
      items: {
        type: 'object',
        properties: {
          name: {
            type: 'string'
          },
          version: {
            type: 'string'
          },
          extension: {
            type: 'string'
          },
          type: {
            type: 'string'
          },
          data: {
            type: 'string',
            nullable: true
          },
          workflows: {
            type: 'array',
            items: {
              type: 'string'
            },
            nullable: true
          }
        },
        required: ['name', 'version', 'extension', 'type'],
        additionalProperties: false
      }
    },
    sampleData: {
      type: 'object',
      properties: {
        path: {
          type: 'string'
        },
        data: {
          type: 'string',
          nullable: true
        }
      },
      required: ['path'],
      additionalProperties: false,
      nullable: true
    }
  },
  required: ['name', 'version', 'license', 'type', 'components', 'assets'],
  additionalProperties: false
};
const instanceSchema = {
  type: 'object',
  properties: {
    name: {
      type: 'string'
    },
    templateName: {
      type: 'string'
    },
    dataSource: {
      type: 'string'
    },
    creationDate: {
      type: 'string'
    },
    assets: {
      type: 'array',
      items: {
        type: 'object',
        properties: {
          assetType: {
            type: 'string'
          },
          assetId: {
            type: 'string'
          },
          isDefaultAsset: {
            type: 'boolean'
          },
          description: {
            type: 'string'
          }
        },
        required: ['assetType', 'assetId', 'isDefaultAsset', 'description']
      }
    }
  },
  required: ['name', 'templateName', 'dataSource', 'creationDate', 'assets']
};
const templateValidator = ajv.compile(templateSchema);
const instanceValidator = ajv.compile(instanceSchema);

/**
 * Validates an integration template against a predefined schema using the AJV library.
 * Since AJV validators use side effects for errors,
 * this is a more conventional wrapper that simplifies calling.
 *
 * @param data The data to be validated as an IntegrationTemplate.
 * @return A Result indicating whether the validation was successful or not.
 *         If validation succeeds, returns an object with 'ok' set to true and the validated data.
 *         If validation fails, returns an object with 'ok' set to false and an Error object describing the validation error.
 */
const validateTemplate = data => {
  if (!templateValidator(data)) {
    return {
      ok: false,
      error: new Error(ajv.errorsText(templateValidator.errors))
    };
  }
  // We assume an invariant that the type of an integration is connected with its component.
  if (data.components.findIndex(x => x.name === data.type) < 0) {
    return {
      ok: false,
      error: new Error(`The integration type '${data.type}' must be included as a component`)
    };
  }
  return {
    ok: true,
    value: data
  };
};

/**
 * Validates an integration instance against a predefined schema using the AJV library.
 *
 * @param data The data to be validated as an IntegrationInstance.
 * @return A Result indicating whether the validation was successful or not.
 *         If validation succeeds, returns an object with 'ok' set to true and the validated data.
 *         If validation fails, returns an object with 'ok' set to false and an Error object describing the validation error.
 */
exports.validateTemplate = validateTemplate;
const validateInstance = data => {
  if (!instanceValidator(data)) {
    return {
      ok: false,
      error: new Error(ajv.errorsText(instanceValidator.errors))
    };
  }
  return {
    ok: true,
    value: data
  };
};
exports.validateInstance = validateInstance;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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