"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerQueryAssistRoutes = registerQueryAssistRoutes;
var _configSchema = require("@osd/config-schema");
var _errors = require("../../../../../src/core/server/opensearch/client/errors");
var _query_assist = require("../../../common/constants/query_assist");
var _generate_field_context = require("../../common/helpers/query_assist/generate_field_context");
var _agents = require("./utils/agents");
var _constants = require("./utils/constants");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function registerQueryAssistRoutes(router) {
  /**
   * Returns whether the PPL agent is configured.
   */
  router.get({
    path: _query_assist.QUERY_ASSIST_API.CONFIGURED,
    validate: {}
  }, async (context, request, response) => {
    const client = context.core.opensearch.client.asCurrentUser;
    try {
      // if the call does not throw any error, then the agent is properly configured
      await (0, _agents.getAgentIdByConfig)(client, _constants.AGENT_CONFIGS.PPL_AGENT);
      return response.ok({
        body: {
          configured: true
        }
      });
    } catch (error) {
      return response.ok({
        body: {
          configured: false,
          error: error.message
        }
      });
    }
  });
  router.post({
    path: _query_assist.QUERY_ASSIST_API.GENERATE_PPL,
    validate: {
      body: _configSchema.schema.object({
        index: _configSchema.schema.string(),
        question: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const client = context.core.opensearch.client.asCurrentUser;
    try {
      const pplRequest = await (0, _agents.requestWithRetryAgentSearch)({
        client,
        configName: _constants.AGENT_CONFIGS.PPL_AGENT,
        body: {
          parameters: {
            index: request.body.index,
            question: request.body.question
          }
        }
      });
      if (!pplRequest.body.inference_results[0].output[0].result) throw new Error('Generated PPL query not found.');
      const result = JSON.parse(pplRequest.body.inference_results[0].output[0].result);
      const ppl = result.ppl.replace(/[\r\n]/g, ' ').trim().replace(/ISNOTNULL/g, 'isnotnull') // https://github.com/opensearch-project/sql/issues/2431
      .replace(/`/g, '') // https://github.com/opensearch-project/dashboards-observability/issues/509, https://github.com/opensearch-project/dashboards-observability/issues/557
      .replace(/\bSPAN\(/g, 'span('); // https://github.com/opensearch-project/dashboards-observability/issues/759
      return response.ok({
        body: ppl
      });
    } catch (error) {
      if ((0, _errors.isResponseError)(error) && error.statusCode === 400 && error.body.error.details === _query_assist.ERROR_DETAILS.GUARDRAILS_TRIGGERED) {
        return response.badRequest({
          body: _query_assist.ERROR_DETAILS.GUARDRAILS_TRIGGERED
        });
      }
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: _query_assist.QUERY_ASSIST_API.SUMMARIZE,
    validate: {
      body: _configSchema.schema.object({
        index: _configSchema.schema.string(),
        question: _configSchema.schema.string(),
        query: _configSchema.schema.maybe(_configSchema.schema.string()),
        response: _configSchema.schema.string(),
        isError: _configSchema.schema.boolean()
      })
    }
  }, async (context, request, response) => {
    const client = context.core.opensearch.client.asCurrentUser;
    const {
      index,
      question,
      query,
      response: _response,
      isError
    } = request.body;
    const queryResponse = JSON.stringify(_response);
    let summaryRequest;
    try {
      var _summaryRequest$body$;
      if (!isError) {
        summaryRequest = await (0, _agents.requestWithRetryAgentSearch)({
          client,
          configName: _constants.AGENT_CONFIGS.RESPONSE_SUMMARY_AGENT,
          body: {
            parameters: {
              index,
              question,
              query,
              response: queryResponse
            }
          }
        });
      } else {
        const [mappings, sampleDoc] = await Promise.all([client.indices.getMapping({
          index
        }), client.search({
          index,
          size: 1
        })]);
        const fields = (0, _generate_field_context.generateFieldContext)(mappings, sampleDoc);
        summaryRequest = await (0, _agents.requestWithRetryAgentSearch)({
          client,
          configName: _constants.AGENT_CONFIGS.ERROR_SUMMARY_AGENT,
          body: {
            parameters: {
              index,
              question,
              query,
              response: queryResponse,
              fields
            }
          }
        });
      }
      const summary = summaryRequest.body.inference_results[0].output[0].result;
      if (!summary) throw new Error('Generated summary not found.');
      const suggestedQuestions = Array.from((((_summaryRequest$body$ = summaryRequest.body.inference_results[0].output[1]) === null || _summaryRequest$body$ === void 0 ? void 0 : _summaryRequest$body$.result) || '').matchAll(/<question>((.|[\r\n])+?)<\/question>/g)).map(m => m[1]);
      return response.ok({
        body: {
          summary,
          suggestedQuestions
        }
      });
    } catch (error) {
      if ((0, _errors.isResponseError)(error) && error.statusCode === 400 && error.body.error.details === _query_assist.ERROR_DETAILS.GUARDRAILS_TRIGGERED) {
        return response.badRequest({
          body: _query_assist.ERROR_DETAILS.GUARDRAILS_TRIGGERED
        });
      }
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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