"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AuthenticationType = void 0;
var _opensearch_security_client = require("../../backend/opensearch_security_client");
var _tenant_resolver = require("../../multitenancy/tenant_resolver");
var _errors = require("../../errors");
var _common = require("../../../common");
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
class AuthenticationType {
  constructor(config, sessionStorageFactory, router, esClient, coreSetup, logger) {
    this.config = config;
    this.sessionStorageFactory = sessionStorageFactory;
    this.router = router;
    this.esClient = esClient;
    this.coreSetup = coreSetup;
    this.logger = logger;
    _defineProperty(this, "type", void 0);
    _defineProperty(this, "securityClient", void 0);
    _defineProperty(this, "authHandler", async (request, response, toolkit) => {
      var _this$config$multiten;
      // skip auth for APIs that do not require auth
      if (this.authNotRequired(request)) {
        return toolkit.authenticated();
      }
      const authState = {};

      // if browser request, auth logic is:
      //   1. check if request includes auth header or parameter(e.g. jwt in url params) is present, if so, authenticate with auth header.
      //   2. if auth header not present, check if auth cookie is present, if no cookie, send to authentication workflow
      //   3. verify whether auth cookie is valid, if not valid, send to authentication workflow
      //   4. if cookie is valid, pass to route handlers
      const authHeaders = {};
      let cookie;
      let authInfo;
      // if this is an REST API call, suppose the request includes necessary auth header
      // see https://www.elastic.co/guide/en/opensearch-dashboards/master/using-api.html
      if (this.requestIncludesAuthInfo(request)) {
        try {
          const additionalAuthHeader = await this.getAdditionalAuthHeader(request);
          Object.assign(authHeaders, additionalAuthHeader);
          authInfo = await this.securityClient.authinfo(request, additionalAuthHeader);
          cookie = this.getCookie(request, authInfo);

          // set tenant from cookie if exist
          const browserCookie = await this.sessionStorageFactory.asScoped(request).get();
          if (browserCookie && (0, _tenant_resolver.isValidTenant)(browserCookie.tenant)) {
            cookie.tenant = browserCookie.tenant;
          }
          this.sessionStorageFactory.asScoped(request).set(cookie);
        } catch (error) {
          return response.unauthorized({
            body: error.message
          });
        }
      } else {
        // no auth header in request, try cookie
        try {
          cookie = await this.sessionStorageFactory.asScoped(request).get();
        } catch (error) {
          this.logger.error(`Error parsing cookie: ${error.message}`);
          cookie = undefined;
        }
        if (!cookie || !(await this.isValidCookie(cookie, request))) {
          // clear cookie
          this.sessionStorageFactory.asScoped(request).clear();

          // for assets, we can still pass it to resource handler as notHandled.
          // marking it as authenticated may result in login pop up when auth challenge
          // is enabled.
          if (request.url.pathname && request.url.pathname.startsWith('/bundles/')) {
            return toolkit.notHandled();
          }

          // send to auth workflow
          return this.handleUnauthedRequest(request, response, toolkit);
        }

        // extend session expiration time
        if (this.config.session.keepalive) {
          cookie.expiryTime = this.getKeepAliveExpiry(cookie, request);
          this.sessionStorageFactory.asScoped(request).set(cookie);
        }
        // cookie is valid
        // build auth header
        const authHeadersFromCookie = this.buildAuthHeaderFromCookie(cookie, request);
        Object.assign(authHeaders, authHeadersFromCookie);
        const additionalAuthHeader = await this.getAdditionalAuthHeader(request);
        Object.assign(authHeaders, additionalAuthHeader);
      }

      // resolve tenant if necessary
      if ((_this$config$multiten = this.config.multitenancy) !== null && _this$config$multiten !== void 0 && _this$config$multiten.enabled) {
        try {
          const tenant = await this.resolveTenant(request, cookie, authHeaders, authInfo);
          // return 401 if no tenant available
          if (!(0, _tenant_resolver.isValidTenant)(tenant)) {
            return response.badRequest({
              body: 'No available tenant for current user, please reach out to your system administrator'
            });
          }
          authState.selectedTenant = tenant;

          // set tenant in header
          if (this.config.multitenancy.enabled && this.config.multitenancy.enable_aggregation_view) {
            // Store all saved objects in a single kibana index.
            Object.assign(authHeaders, {
              securitytenant: _common.GLOBAL_TENANT_SYMBOL
            });
          } else {
            Object.assign(authHeaders, {
              securitytenant: tenant
            });
          }

          // set tenant to cookie
          if (tenant !== cookie.tenant) {
            cookie.tenant = tenant;
            this.sessionStorageFactory.asScoped(request).set(cookie);
          }
        } catch (error) {
          this.logger.error(`Failed to resolve user tenant: ${error}`);
          if (error instanceof _errors.UnauthenticatedError) {
            if (request.url.pathname && request.url.pathname.startsWith('/bundles/')) {
              return toolkit.notHandled();
            }
            this.sessionStorageFactory.asScoped(request).clear();
            return this.handleUnauthedRequest(request, response, toolkit);
          }
          throw error;
        }
      }
      if (!authInfo) {
        authInfo = await this.securityClient.authinfo(request, authHeaders);
      }
      authState.authInfo = authInfo;
      return toolkit.authenticated({
        requestHeaders: authHeaders,
        state: authState
      });
    });
    this.securityClient = new _opensearch_security_client.SecurityClient(esClient);
    this.type = '';
    this.config = config;
  }
  authNotRequired(request) {
    const pathname = request.url.pathname;
    if (!pathname) {
      return false;
    }
    // allow requests to ignored routes
    if (AuthenticationType.ROUTES_TO_IGNORE.includes(pathname)) {
      return true;
    }
    // allow requests to routes that doesn't require authentication
    if (this.config.auth.unauthenticated_routes.indexOf(pathname) > -1) {
      // TODO: use opensearch-dashboards server user
      return true;
    }
    return false;
  }
  async resolveTenant(request, cookie, authHeader, authInfo) {
    if (!authInfo) {
      try {
        authInfo = await this.securityClient.authinfo(request, authHeader);
      } catch (error) {
        throw new _errors.UnauthenticatedError(error);
      }
    }
    const dashboardsInfo = await this.securityClient.dashboardsinfo(request, authHeader);
    return (0, _tenant_resolver.resolveTenant)({
      request,
      username: authInfo.user_name,
      roles: authInfo.roles,
      availableTenants: authInfo.tenants,
      config: this.config,
      cookie,
      multitenancyEnabled: dashboardsInfo.multitenancy_enabled,
      privateTenantEnabled: dashboardsInfo.private_tenant_enabled,
      defaultTenant: dashboardsInfo.default_tenant
    });
  }
  getKeepAliveExpiry(cookie, request) {
    return Date.now() + this.config.session.ttl;
  }
  isPageRequest(request) {
    const path = request.url.pathname || '/';
    return path.startsWith('/app/') || path === '/' || path.startsWith('/goto/');
  }

  // abstract functions for concrete auth types to implement
}
exports.AuthenticationType = AuthenticationType;
_defineProperty(AuthenticationType, "ROUTES_TO_IGNORE", ['/api/core/capabilities',
// FIXME: need to figure out how to bypass this API call
'/app/login']);
_defineProperty(AuthenticationType, "REST_API_CALL_HEADER", 'osd-xsrf');
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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