"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MultipleAuthentication = void 0;
var _authentication_type = require("../authentication_type");
var _common = require("../../../../common");
var _next_url = require("../../../utils/next_url");
var _routes = require("./routes");
var _types = require("../../types");
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
class MultipleAuthentication extends _authentication_type.AuthenticationType {
  constructor(config, sessionStorageFactory, router, esClient, coreSetup, logger) {
    super(config, sessionStorageFactory, router, esClient, coreSetup, logger);
    _defineProperty(this, "authTypes", void 0);
    _defineProperty(this, "authHandlers", void 0);
    this.authTypes = this.config.auth.type;
    this.authHandlers = new Map();
  }
  async init() {
    const routes = new _routes.MultiAuthRoutes(this.router, this.sessionStorageFactory);
    routes.setupRoutes();
    for (let i = 0; i < this.authTypes.length; i++) {
      switch (this.authTypes[i].toLowerCase()) {
        case _common.AuthType.BASIC:
          {
            const BasicAuth = new _types.BasicAuthentication(this.config, this.sessionStorageFactory, this.router, this.esClient, this.coreSetup, this.logger);
            await BasicAuth.init();
            this.authHandlers.set(_common.AuthType.BASIC, BasicAuth);
            break;
          }
        case _common.AuthType.OPEN_ID:
          {
            const OidcAuth = new _types.OpenIdAuthentication(this.config, this.sessionStorageFactory, this.router, this.esClient, this.coreSetup, this.logger);
            await OidcAuth.init();
            this.authHandlers.set(_common.AuthType.OPEN_ID, OidcAuth);
            break;
          }
        case _common.AuthType.SAML:
          {
            const SamlAuth = new _types.SamlAuthentication(this.config, this.sessionStorageFactory, this.router, this.esClient, this.coreSetup, this.logger);
            await SamlAuth.init();
            this.authHandlers.set(_common.AuthType.SAML, SamlAuth);
            break;
          }
        default:
          {
            throw new Error(`Unsupported authentication type: ${this.authTypes[i]}`);
          }
      }
    }
  }

  // override functions inherited from AuthenticationType
  requestIncludesAuthInfo(request) {
    for (const key of this.authHandlers.keys()) {
      if (this.authHandlers.get(key).requestIncludesAuthInfo(request)) {
        return true;
      }
    }
    return false;
  }
  async getAdditionalAuthHeader(request) {
    var _cookie$authType;
    // To Do: refactor this method to improve the effiency to get cookie, get cookie from input parameter
    const cookie = await this.sessionStorageFactory.asScoped(request).get();
    const reqAuthType = cookie === null || cookie === void 0 || (_cookie$authType = cookie.authType) === null || _cookie$authType === void 0 ? void 0 : _cookie$authType.toLowerCase();
    if (reqAuthType && this.authHandlers.has(reqAuthType)) {
      return await this.authHandlers.get(reqAuthType).getAdditionalAuthHeader(request);
    } else {
      return {};
    }
  }
  getCookie(request, authInfo) {
    return {};
  }
  getKeepAliveExpiry(cookie, request) {
    var _cookie$authType2;
    const reqAuthType = cookie === null || cookie === void 0 || (_cookie$authType2 = cookie.authType) === null || _cookie$authType2 === void 0 ? void 0 : _cookie$authType2.toLowerCase();
    if (reqAuthType && this.authHandlers.has(reqAuthType)) {
      return this.authHandlers.get(reqAuthType).getKeepAliveExpiry(cookie, request);
    } else {
      // default to TTL setting
      return Date.now() + this.config.session.ttl;
    }
  }
  async isValidCookie(cookie, request) {
    var _cookie$authType3;
    const reqAuthType = cookie === null || cookie === void 0 || (_cookie$authType3 = cookie.authType) === null || _cookie$authType3 === void 0 ? void 0 : _cookie$authType3.toLowerCase();
    if (reqAuthType && this.authHandlers.has(reqAuthType)) {
      return this.authHandlers.get(reqAuthType).isValidCookie(cookie, request);
    } else {
      return false;
    }
  }
  handleUnauthedRequest(request, response, toolkit) {
    if (this.isPageRequest(request)) {
      const nextUrlParam = (0, _next_url.composeNextUrlQueryParam)(request, this.coreSetup.http.basePath.serverBasePath);
      if (this.config.auth.anonymous_auth_enabled) {
        const redirectLocation = `${this.coreSetup.http.basePath.serverBasePath}${_common.ANONYMOUS_AUTH_LOGIN}?${nextUrlParam}`;
        return response.redirected({
          headers: {
            location: `${redirectLocation}`
          }
        });
      }
      return response.redirected({
        headers: {
          location: `${this.coreSetup.http.basePath.serverBasePath}${_common.LOGIN_PAGE_URI}?${nextUrlParam}`
        }
      });
    } else {
      return response.unauthorized();
    }
  }
  buildAuthHeaderFromCookie(cookie, request) {
    var _cookie$authType4;
    const reqAuthType = cookie === null || cookie === void 0 || (_cookie$authType4 = cookie.authType) === null || _cookie$authType4 === void 0 ? void 0 : _cookie$authType4.toLowerCase();
    if (reqAuthType && this.authHandlers.has(reqAuthType)) {
      return this.authHandlers.get(reqAuthType).buildAuthHeaderFromCookie(cookie, request);
    } else {
      return {};
    }
  }
}
exports.MultipleAuthentication = MultipleAuthentication;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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