"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
exports.registerADRoutes = registerADRoutes;
var _lodash = require("lodash");
var _constants = require("../utils/constants");
var _helpers = require("../utils/helpers");
var _adHelpers = require("./utils/adHelpers");
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 *
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */
function registerADRoutes(apiRouter, adService) {
  apiRouter.post('/detectors', adService.putDetector);
  apiRouter.put('/detectors/{detectorId}', adService.putDetector);
  apiRouter.post('/detectors/_search', adService.searchDetector);
  apiRouter.post('/detectors/results/_search/', adService.searchResults);
  apiRouter.post('/detectors/results/_search', adService.searchResults);
  apiRouter.post('/detectors/results/_search/{resultIndex}/{onlyQueryCustomResultIndex}', adService.searchResults);
  apiRouter.get('/detectors/{detectorId}', adService.getDetector);
  apiRouter.get('/detectors', adService.getDetectors);
  apiRouter.post('/detectors/preview', adService.previewDetector);
  apiRouter.get('/detectors/{id}/results/{isHistorical}/{resultIndex}/{onlyQueryCustomResultIndex}', adService.getAnomalyResults);
  apiRouter.get('/detectors/{id}/results/{isHistorical}', adService.getAnomalyResults);
  apiRouter.delete('/detectors/{detectorId}', adService.deleteDetector);
  apiRouter.post('/detectors/{detectorId}/start', adService.startDetector);
  apiRouter.post('/detectors/{detectorId}/stop/{isHistorical}', adService.stopDetector);
  apiRouter.get('/detectors/{detectorId}/_profile', adService.getDetectorProfile);
  apiRouter.get('/detectors/{detectorName}/_match', adService.matchDetector);
  apiRouter.get('/detectors/_count', adService.getDetectorCount);
  apiRouter.post('/detectors/{detectorId}/_topAnomalies/{isHistorical}', adService.getTopAnomalyResults);
  apiRouter.post('/detectors/_validate/{validationType}', adService.validateDetector);
}
class AdService {
  constructor(client) {
    _defineProperty(this, "client", void 0);
    _defineProperty(this, "deleteDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          detectorId
        } = request.params;
        const response = await this.client.asScoped(request).callAsCurrentUser('ad.deleteDetector', {
          detectorId
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - deleteDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "previewDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const requestBody = JSON.stringify((0, _adHelpers.convertPreviewInputKeysToSnakeCase)(request.body));
        const response = await this.client.asScoped(request).callAsCurrentUser('ad.previewDetector', {
          body: requestBody
        });
        const transformedKeys = (0, _helpers.mapKeysDeep)(response, _helpers.toCamel);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            //@ts-ignore
            response: (0, _adHelpers.anomalyResultMapper)(transformedKeys.anomalyResult)
          }
        });
      } catch (err) {
        console.log('Anomaly detector - previewDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "putDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          detectorId
        } = request.params;
        //@ts-ignore
        const ifSeqNo = request.body.seqNo;
        //@ts-ignore
        const ifPrimaryTerm = request.body.primaryTerm;
        const requestBody = JSON.stringify((0, _adHelpers.convertDetectorKeysToSnakeCase)(request.body));
        let params = {
          detectorId: detectorId,
          ifSeqNo: ifSeqNo,
          ifPrimaryTerm: ifPrimaryTerm,
          body: requestBody
        };
        let response;
        if ((0, _lodash.isNumber)(ifSeqNo) && (0, _lodash.isNumber)(ifPrimaryTerm)) {
          response = await this.client.asScoped(request).callAsCurrentUser('ad.updateDetector', params);
        } else {
          response = await this.client.asScoped(request).callAsCurrentUser('ad.createDetector', {
            body: params.body
          });
        }
        const resp = {
          ...response.anomaly_detector,
          id: response._id,
          primaryTerm: response._primary_term,
          seqNo: response._seq_no
        };
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: (0, _adHelpers.convertDetectorKeysToCamelCase)(resp)
          }
        });
      } catch (err) {
        console.log('Anomaly detector - PutDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "validateDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        let {
          validationType
        } = request.params;
        const requestBody = JSON.stringify((0, _adHelpers.convertPreviewInputKeysToSnakeCase)(request.body));
        const response = await this.client.asScoped(request).callAsCurrentUser('ad.validateDetector', {
          body: requestBody,
          validationType: validationType
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - validateDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          detectorId
        } = request.params;
        const detectorResponse = await this.client.asScoped(request).callAsCurrentUser('ad.getDetector', {
          detectorId
        });

        // Populating static detector fields
        const staticFields = {
          id: detectorResponse._id,
          primaryTerm: detectorResponse._primary_term,
          seqNo: detectorResponse._seq_no,
          ...(0, _adHelpers.convertStaticFieldsToCamelCase)(detectorResponse.anomaly_detector)
        };

        // Get real-time and historical task info to populate the
        // task and job-related fields
        // We wrap these calls in a try/catch, and suppress any index_not_found_exceptions
        // which can occur if no detector jobs have been ran on a new cluster.
        let realtimeTasksResponse = {};
        let historicalTasksResponse = {};
        try {
          realtimeTasksResponse = await this.client.asScoped(request).callAsCurrentUser('ad.searchTasks', {
            body: (0, _adHelpers.getLatestTaskForDetectorQuery)(detectorId, true)
          });
          historicalTasksResponse = await this.client.asScoped(request).callAsCurrentUser('ad.searchTasks', {
            body: (0, _adHelpers.getLatestTaskForDetectorQuery)(detectorId, false)
          });
        } catch (err) {
          if (!(0, _adHelpers.isIndexNotFoundError)(err)) {
            throw err;
          }
        }
        const realtimeTask = (0, _lodash.get)((0, _lodash.get)(realtimeTasksResponse, 'hits.hits', []).map(taskResponse => {
          return {
            id: (0, _lodash.get)(taskResponse, '_id'),
            ...(0, _lodash.get)(taskResponse, '_source')
          };
        }), 0);
        const historicalTask = (0, _lodash.get)((0, _lodash.get)(historicalTasksResponse, 'hits.hits', []).map(taskResponse => {
          return {
            id: (0, _lodash.get)(taskResponse, '_id'),
            ...(0, _lodash.get)(taskResponse, '_source')
          };
        }), 0);
        const taskAndJobFields = (0, _adHelpers.convertTaskAndJobFieldsToCamelCase)(realtimeTask, historicalTask, detectorResponse.anomaly_detector_job);

        // Combine the static and task-and-job-related fields into
        // a final response
        const finalResponse = {
          ...staticFields,
          ...taskAndJobFields
        };
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: finalResponse
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get detector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "startDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        var _request$body, _request$body2;
        const {
          detectorId
        } = request.params;
        //@ts-ignore
        const startTime = (_request$body = request.body) === null || _request$body === void 0 ? void 0 : _request$body.startTime;
        //@ts-ignore
        const endTime = (_request$body2 = request.body) === null || _request$body2 === void 0 ? void 0 : _request$body2.endTime;
        let requestParams = {
          detectorId: detectorId
        };
        let requestPath = 'ad.startDetector';
        // If a start and end time are passed: we want to start a historical analysis
        if ((0, _lodash.isNumber)(startTime) && (0, _lodash.isNumber)(endTime)) {
          requestParams = {
            ...requestParams,
            body: {
              start_time: startTime,
              end_time: endTime
            }
          };
          requestPath = 'ad.startHistoricalDetector';
        }
        const response = await this.client.asScoped(request).callAsCurrentUser(requestPath, requestParams);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - startDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "stopDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        let {
          detectorId,
          isHistorical
        } = request.params;
        isHistorical = JSON.parse(isHistorical);
        const requestPath = isHistorical ? 'ad.stopHistoricalDetector' : 'ad.stopDetector';
        const response = await this.client.asScoped(request).callAsCurrentUser(requestPath, {
          detectorId
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - stopDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getDetectorProfile", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          detectorId
        } = request.params;
        const response = await this.client.asScoped(request).callAsCurrentUser('ad.detectorProfile', {
          detectorId
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - detectorProfile', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "searchDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const requestBody = JSON.stringify(request.body);
        const response = await this.client.asScoped(request).callAsCurrentUser('ad.searchDetector', {
          body: requestBody
        });
        const totalDetectors = (0, _lodash.get)(response, 'hits.total.value', 0);
        const detectors = (0, _lodash.get)(response, 'hits.hits', []).map(detector => ({
          ...(0, _adHelpers.convertDetectorKeysToCamelCase)(detector._source),
          id: detector._id,
          seqNo: detector._seq_no,
          primaryTerm: detector._primary_term
        }));
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              totalDetectors,
              detectors
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to search detectors', err);
        if ((0, _adHelpers.isIndexNotFoundError)(err)) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                totalDetectors: 0,
                detectors: []
              }
            }
          });
        }
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "searchResults", async (context, request, opensearchDashboardsResponse) => {
      try {
        var {
          resultIndex,
          onlyQueryCustomResultIndex
        } = request.params;
        if (!resultIndex || !resultIndex.startsWith(_constants.CUSTOM_AD_RESULT_INDEX_PREFIX)) {
          // Set resultIndex as '' means no custom result index specified, will only search anomaly result from default index.
          resultIndex = '';
        }
        let requestParams = {
          resultIndex: resultIndex,
          onlyQueryCustomResultIndex: onlyQueryCustomResultIndex
        };
        const requestBody = JSON.stringify(request.body);
        const response = !resultIndex ? await this.client.asScoped(request).callAsCurrentUser('ad.searchResults', {
          body: requestBody
        }) : await this.client.asScoped(request).callAsCurrentUser('ad.searchResultsFromCustomResultIndex', {
          ...requestParams,
          body: requestBody
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to search anomaly result', err);
        if ((0, _adHelpers.isIndexNotFoundError)(err)) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                totalDetectors: 0,
                detectors: []
              }
            }
          });
        }
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getDetectors", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          from = 0,
          size = 20,
          search = '',
          indices = '',
          sortDirection = _constants.SORT_DIRECTION.DESC,
          sortField = 'name'
        } = request.query;
        const mustQueries = [];
        if (search.trim()) {
          mustQueries.push({
            query_string: {
              fields: ['name', 'description'],
              default_operator: 'AND',
              query: `*${search.trim().split('-').join('* *')}*`
            }
          });
        }
        if (indices.trim()) {
          mustQueries.push({
            query_string: {
              fields: ['indices'],
              default_operator: 'AND',
              query: `*${indices.trim().split('-').join('* *')}*`
            }
          });
        }
        //Allowed sorting columns
        const sortQueryMap = {
          name: {
            'name.keyword': sortDirection
          },
          indices: {
            'indices.keyword': sortDirection
          },
          lastUpdateTime: {
            last_update_time: sortDirection
          }
        };
        let sort = {};
        const sortQuery = sortQueryMap[sortField];
        if (sortQuery) {
          sort = sortQuery;
        }
        //Preparing search request
        const requestBody = {
          sort,
          size,
          from,
          query: {
            bool: {
              must: mustQueries
            }
          }
        };
        const response = await this.client.asScoped(request).callAsCurrentUser('ad.searchDetector', {
          body: requestBody
        });
        const totalDetectors = (0, _lodash.get)(response, 'hits.total.value', 0);

        //Get all detectors from search detector API
        const allDetectors = (0, _lodash.get)(response, 'hits.hits', []).reduce((acc, detectorResponse) => ({
          ...acc,
          [detectorResponse._id]: {
            id: detectorResponse._id,
            primaryTerm: detectorResponse._primary_term,
            seqNo: detectorResponse._seq_no,
            ...(0, _adHelpers.convertStaticFieldsToCamelCase)(detectorResponse._source)
          }
        }), {});
        const allDetectorsMap = Object.values(allDetectors).reduce((acc, detector) => ({
          ...acc,
          [detector.id]: detector
        }), {});

        //Given each detector from previous result, get aggregation to power list
        const allDetectorIds = Object.keys(allDetectorsMap);
        let requestParams = {
          // If specifying result index, will query anomaly result from both default and custom result indices.
          // If no valid result index specified, just query anomaly result from default result index.
          // Here we specify custom AD result index prefix pattern to query all custom result indices.
          resultIndex: _constants.CUSTOM_AD_RESULT_INDEX_PREFIX + '*',
          onlyQueryCustomResultIndex: 'false'
        };
        const aggregationResult = await this.client.asScoped(request).callAsCurrentUser('ad.searchResultsFromCustomResultIndex', {
          ...requestParams,
          body: (0, _adHelpers.getResultAggregationQuery)(allDetectorIds, {
            from,
            size,
            sortField,
            sortDirection,
            search,
            indices
          })
        });
        const aggsDetectors = (0, _lodash.get)(aggregationResult, 'aggregations.unique_detectors.buckets', []).reduce((acc, agg) => {
          return {
            ...acc,
            [agg.key]: {
              ...allDetectorsMap[agg.key],
              totalAnomalies: agg.total_anomalies_in_24hr.doc_count,
              lastActiveAnomaly: agg.latest_anomaly_time.value
            }
          };
        }, {});

        // Aggregation will not return values where anomalies for detectors are not generated, loop through it and fill values with 0
        const unUsedDetectors = (0, _lodash.pullAll)(allDetectorIds, Object.keys(aggsDetectors)).reduce((acc, unusedDetector) => {
          return {
            ...acc,
            [unusedDetector]: {
              ...allDetectorsMap[unusedDetector],
              totalAnomalies: 0,
              lastActiveAnomaly: 0
            }
          };
        }, {});

        // If sorting criteria is from the aggregation manage pagination in memory.
        let finalDetectors = (0, _lodash.orderBy)({
          ...aggsDetectors,
          ...unUsedDetectors
        }, [sortField], [sortDirection]);
        if (!sortQueryMap[sortField]) {
          finalDetectors = Object.values(finalDetectors).slice(from, from + size).reduce((acc, detector) => ({
            ...acc,
            [detector.id]: detector
          }), {});
        }

        // Fetch the latest realtime and historical tasks for all detectors
        // using terms aggregations
        // We wrap these calls in a try/catch, and suppress any index_not_found_exceptions
        // which can occur if no detector jobs have been ran on a new cluster.
        let realtimeTasksResponse = {};
        let historicalTasksResponse = {};
        try {
          realtimeTasksResponse = await this.client.asScoped(request).callAsCurrentUser('ad.searchTasks', {
            body: (0, _adHelpers.getLatestDetectorTasksQuery)(true)
          });
          historicalTasksResponse = await this.client.asScoped(request).callAsCurrentUser('ad.searchTasks', {
            body: (0, _adHelpers.getLatestDetectorTasksQuery)(false)
          });
        } catch (err) {
          if (!(0, _adHelpers.isIndexNotFoundError)(err)) {
            throw err;
          }
        }
        const realtimeTasks = (0, _lodash.get)(realtimeTasksResponse, 'aggregations.detectors.buckets', []).reduce((acc, bucket) => {
          return {
            ...acc,
            [bucket.key]: {
              realtimeTask: (0, _lodash.get)(bucket, 'latest_tasks.hits.hits.0', undefined)
            }
          };
        }, {});
        const historicalTasks = (0, _lodash.get)(historicalTasksResponse, 'aggregations.detectors.buckets', []).reduce((acc, bucket) => {
          return {
            ...acc,
            [bucket.key]: {
              historicalTask: (0, _lodash.get)(bucket, 'latest_tasks.hits.hits.0', undefined)
            }
          };
        }, {});

        // Get real-time and historical task info by looping through each detector & retrieving
        //    - curState by getting real-time task state
        //    - enabledTime by getting real-time task's execution_start time
        //    - taskId by getting historical task's _id
        finalDetectors.forEach(detector => {
          const realtimeTask = (0, _lodash.get)(realtimeTasks[detector.id], 'realtimeTask._source');
          detector.curState = (0, _adHelpers.getTaskState)(realtimeTask);
          detector.enabledTime = (0, _lodash.get)(realtimeTask, 'execution_start_time');
          detector.taskId = (0, _lodash.get)(historicalTasks[detector.id], 'historicalTask._id');
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              totalDetectors,
              detectorList: Object.values(finalDetectors)
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to search detectors', err);
        if ((0, _adHelpers.isIndexNotFoundError)(err)) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                totalDetectors: 0,
                detectorList: []
              }
            }
          });
        }
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getAnomalyResults", async (context, request, opensearchDashboardsResponse) => {
      let {
        id,
        isHistorical,
        resultIndex,
        onlyQueryCustomResultIndex
      } = request.params;
      if (!resultIndex || !resultIndex.startsWith(_constants.CUSTOM_AD_RESULT_INDEX_PREFIX)) {
        // Set resultIndex as '' means no custom result index specified, will only search anomaly result from default index.
        resultIndex = '';
      }
      isHistorical = JSON.parse(isHistorical);

      // Search by task id if historical, or by detector id if realtime
      const searchTerm = isHistorical ? {
        task_id: id
      } : {
        detector_id: id
      };
      try {
        const {
          from = 0,
          size = 20,
          sortDirection = _constants.SORT_DIRECTION.DESC,
          sortField = _constants.AD_DOC_FIELDS.DATA_START_TIME,
          startTime = 0,
          endTime = 0,
          fieldName = '',
          anomalyThreshold = -1,
          entityList = ''
        } = request.query;
        const entityListAsObj = entityList.length === 0 ? {} : JSON.parse(entityList);
        const entityFilters = (0, _lodash.isEmpty)(entityListAsObj) ? [] : (0, _adHelpers.getFiltersFromEntityList)(entityListAsObj);

        //Allowed sorting columns
        const sortQueryMap = {
          anomalyGrade: {
            anomaly_grade: sortDirection
          },
          confidence: {
            confidence: sortDirection
          },
          [_constants.AD_DOC_FIELDS.DATA_START_TIME]: {
            [_constants.AD_DOC_FIELDS.DATA_START_TIME]: sortDirection
          },
          [_constants.AD_DOC_FIELDS.DATA_END_TIME]: {
            [_constants.AD_DOC_FIELDS.DATA_END_TIME]: sortDirection
          }
        };
        let sort = {};
        const sortQuery = sortQueryMap[sortField];
        if (sortQuery) {
          sort = sortQuery;
        }

        //Preparing search request
        const requestBody = {
          sort,
          size,
          from,
          query: {
            bool: {
              filter: [{
                term: searchTerm
              }, {
                range: {
                  anomaly_grade: {
                    gt: anomalyThreshold
                  }
                }
              }, ...entityFilters]
            }
          }
        };

        // If querying RT results: remove any results that include a task_id, as this indicates
        // a historical result from a historical task.
        if (!isHistorical) {
          requestBody.query.bool = {
            ...requestBody.query.bool,
            ...{
              must_not: {
                exists: {
                  field: 'task_id'
                }
              }
            }
          };
        }
        try {
          const filterSize = requestBody.query.bool.filter.length;
          if (fieldName) {
            (startTime || endTime) && (0, _lodash.set)(requestBody.query.bool.filter, `${filterSize}.range.${fieldName}.format`, 'epoch_millis');
            startTime && (0, _lodash.set)(requestBody.query.bool.filter, `${filterSize}.range.${fieldName}.gte`, startTime);
            endTime && (0, _lodash.set)(requestBody.query.bool.filter, `${filterSize}.range.${fieldName}.lte`, endTime);
          }
        } catch (error) {
          console.log('wrong date range filter', error);
        }
        let requestParams = {
          resultIndex: resultIndex,
          onlyQueryCustomResultIndex: onlyQueryCustomResultIndex
        };
        const response = !resultIndex ? await this.client.asScoped(request).callAsCurrentUser('ad.searchResults', {
          body: requestBody
        }) : await this.client.asScoped(request).callAsCurrentUser('ad.searchResultsFromCustomResultIndex', {
          ...requestParams,
          body: requestBody
        });
        const totalResults = (0, _lodash.get)(response, 'hits.total.value', 0);
        const detectorResult = [];
        const featureResult = {};
        (0, _lodash.get)(response, 'hits.hits', []).forEach(result => {
          detectorResult.push({
            startTime: result._source.data_start_time,
            endTime: result._source.data_end_time,
            plotTime: result._source.data_end_time,
            contributions: result._source.anomaly_grade > 0 ? result._source.relevant_attribution : {},
            confidence: result._source.confidence != null && result._source.confidence !== 'NaN' && result._source.confidence > 0 ? (0, _helpers.toFixedNumberForAnomaly)(Number.parseFloat(result._source.confidence)) : 0,
            anomalyGrade: result._source.anomaly_grade != null && result._source.anomaly_grade !== 'NaN' && result._source.anomaly_grade > 0 ? (0, _helpers.toFixedNumberForAnomaly)(Number.parseFloat(result._source.anomaly_grade)) : 0,
            ...(result._source.entity != null ? {
              entity: result._source.entity
            } : {}),
            // TODO: we should refactor other places to read feature data from
            // AnomalyResult, instead of having separate FeatureData which is hard
            // to know feature data belongs to which anomaly result
            features: this.getFeatureData(result)
          });
          result._source.feature_data.forEach(featureData => {
            if (!featureResult[featureData.feature_id]) {
              featureResult[featureData.feature_id] = [];
            }
            featureResult[featureData.feature_id].push({
              startTime: result._source.data_start_time,
              endTime: result._source.data_end_time,
              plotTime: result._source.data_end_time,
              data: featureData.data != null && featureData.data !== 'NaN' ? (0, _helpers.toFixedNumberForAnomaly)(Number.parseFloat(featureData.data)) : 0,
              name: featureData.feature_name,
              expectedValue: this.getExpectedValue(result, featureData)
            });
          });
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              totalAnomalies: totalResults,
              results: detectorResult,
              featureResults: featureResult
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get results', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getTopAnomalyResults", async (context, request, opensearchDashboardsResponse) => {
      try {
        let {
          detectorId,
          isHistorical
        } = request.params;
        isHistorical = JSON.parse(isHistorical);
        const requestPath = isHistorical ? 'ad.topHistoricalAnomalyResults' : 'ad.topAnomalyResults';
        const response = await this.client.asScoped(request).callAsCurrentUser(requestPath, {
          detectorId: detectorId,
          body: request.body
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - getTopAnomalyResults', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "matchDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          detectorName
        } = request.params;
        const response = await this.client.asScoped(request).callAsCurrentUser('ad.matchDetector', {
          detectorName
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - matchDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getDetectorCount", async (context, request, opensearchDashboardsResponse) => {
      try {
        const response = await this.client.asScoped(request).callAsCurrentUser('ad.detectorCount');
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - getDetectorCount', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getFeatureData", rawResult => {
      const featureResult = {};
      rawResult._source.feature_data.forEach(featureData => {
        featureResult[featureData.feature_id] = {
          startTime: rawResult._source.data_start_time,
          endTime: rawResult._source.data_end_time,
          plotTime: rawResult._source.data_end_time,
          data: featureData.data != null && featureData.data !== 'NaN' ? (0, _helpers.toFixedNumberForAnomaly)(Number.parseFloat(featureData.data)) : 0,
          name: featureData.feature_name,
          expectedValue: this.getExpectedValue(rawResult, featureData)
        };
      });
      return featureResult;
    });
    _defineProperty(this, "getExpectedValue", (rawResult, featureData) => {
      let expectedValue = featureData.data != null && featureData.data !== 'NaN' ? (0, _helpers.toFixedNumberForAnomaly)(Number.parseFloat(featureData.data)) : 0;
      if (rawResult._source.anomaly_grade > 0) {
        const expectedValueList = rawResult._source.expected_values;
        if ((expectedValueList === null || expectedValueList === void 0 ? void 0 : expectedValueList.length) > 0) {
          expectedValueList[0].value_list.forEach(expect => {
            if (expect.feature_id === featureData.feature_id) {
              expectedValue = expect.data;
            }
          });
        }
      }
      return expectedValue;
    });
    this.client = client;
  }
}
exports.default = AdService;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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