"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
var _constants = require("../utils/constants");
var _helpers = require("../utils/helpers");
var _DataStreamService = require("./DataStreamService");
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
class IndexService {
  constructor(osDriver) {
    _defineProperty(this, "osDriver", void 0);
    _defineProperty(this, "getIndices", async (context, request, response) => {
      try {
        // @ts-ignore
        const {
          from,
          size,
          sortField,
          sortDirection,
          terms,
          indices,
          dataStreams,
          showDataStreams,
          expandWildcards,
          exactSearch
        } = request.query;
        const params = {
          index: (0, _helpers.getSearchString)(terms, indices, dataStreams),
          format: "json"
        };
        if (sortField !== "managed" && sortField !== "data_stream") {
          params.s = `${sortField}:${sortDirection}`;
        }
        if (expandWildcards) {
          params.expand_wildcards = expandWildcards;
        }
        if (exactSearch) {
          params.index = exactSearch;
        }
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const [recoverys, tasks, indicesResponse, indexToDataStreamMapping] = await Promise.all([callWithRequest("cat.recovery", {
          format: "json",
          detailed: true
        }).catch(() => []), callWithRequest("cat.tasks", {
          format: "json",
          detailed: true,
          actions: "indices:data/write/reindex"
        }).catch(() => []), callWithRequest("cat.indices", params), (0, _DataStreamService.getIndexToDataStreamMapping)({
          callAsCurrentUser: callWithRequest
        })]);
        const formattedTasks = tasks.map(item => {
          const {
            description
          } = item;
          const regexp = /reindex from \[([^\]]+)\] to \[([^\]]+)\]/i;
          const matchResult = description.match(regexp);
          if (matchResult) {
            const [, fromIndex, toIndex] = matchResult;
            return {
              ...item,
              fromIndex,
              toIndex
            };
          } else {
            return {
              ...item,
              fromIndex: "",
              toIndex: ""
            };
          }
        });
        const onGoingRecovery = recoverys.filter(item => item.stage !== "done");

        // Augment the indices with their parent data stream name.
        indicesResponse.forEach(index => {
          index.data_stream = indexToDataStreamMapping[index.index] || null;
          let extraStatus = index.status;
          if (index.health === "green") {
            if (formattedTasks.find(item => item.toIndex === index.index)) {
              extraStatus = "reindex";
            }
          } else {
            if (onGoingRecovery.find(item => item.index === index.index)) {
              extraStatus = "recovery";
            }
          }
          if (extraStatus) {
            index.extraStatus = extraStatus;
          }
        });
        function customSort(array, key, sortDirection) {
          return array.sort((a, b) => {
            let flag;
            const aValue = a[key];
            const bValue = b[key];
            if (sortDirection === "asc") {
              flag = aValue < bValue;
            } else {
              flag = aValue > bValue;
            }
            return flag ? -1 : 1;
          });
        }
        if (sortField === "status") {
          // add new more status to status field so we need to sort
          customSort(indicesResponse, "extraStatus", sortDirection);
        }

        // Filtering out indices that belong to a data stream. This must be done before pagination.
        const filteredIndices = showDataStreams ? indicesResponse : indicesResponse.filter(index => index.data_stream === null);

        // _cat doesn't support pagination, do our own in server pagination to at least reduce network bandwidth
        const fromNumber = parseInt(from, 10);
        const sizeNumber = parseInt(size, 10);
        const paginatedIndices = filteredIndices.slice(fromNumber, fromNumber + sizeNumber);
        const indexNames = paginatedIndices.map(value => value.index);
        const managedStatus = await this._getManagedStatus(request, indexNames);
        const allIndices = paginatedIndices.map(catIndex => ({
          ...catIndex,
          managed: managedStatus[catIndex.index] ? "Yes" : "No",
          managedPolicy: managedStatus[catIndex.index]
        }));

        // NOTE: Cannot use response.ok due to typescript type checking
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: {
              indices: sortField === "managed" ? customSort(allIndices, "managed", sortDirection) : allIndices,
              totalIndices: filteredIndices.length
            }
          }
        });
      } catch (err) {
        // Throws an error if there is no index matching pattern
        if (err.statusCode === 404 && err.body.error.type === "index_not_found_exception") {
          return response.custom({
            statusCode: 200,
            body: {
              ok: true,
              response: {
                indices: [],
                totalIndices: 0
              }
            }
          });
        }
        console.error("Index Management - IndexService - getIndices:", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });
    _defineProperty(this, "_getManagedStatus", async (request, indexNames) => {
      try {
        const explainParamas = {
          index: indexNames.toString()
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const explainResponse = await callWithRequest("ism.explain", explainParamas);
        const managed = {};
        for (const indexName in explainResponse) {
          if (indexName === "total_managed_indices") continue;
          const explain = explainResponse[indexName];
          managed[indexName] = explain["index.plugins.index_state_management.policy_id"] === null ? "" : explain["index.plugins.index_state_management.policy_id"];
        }
        return managed;
      } catch (err) {
        // otherwise it could be an unauthorized access error to config index or some other error
        // in which case we will return managed status N/A
        console.error("Index Management - IndexService - _getManagedStatus:", err);
        return indexNames.reduce((accu, value) => ({
          ...accu,
          [value]: "N/A"
        }), {});
      }
    });
    _defineProperty(this, "applyPolicy", async (context, request, response) => {
      try {
        const {
          indices,
          policyId
        } = request.body;
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const params = {
          index: indices.join(","),
          body: {
            policy_id: policyId
          }
        };
        const addResponse = await callWithRequest("ism.add", params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: {
              failures: addResponse.failures,
              updatedIndices: addResponse.updated_indices,
              failedIndices: addResponse.failed_indices.map(failedIndex => ({
                indexName: failedIndex.index_name,
                indexUuid: failedIndex.index_uuid,
                reason: failedIndex.reason
              }))
            }
          }
        });
      } catch (err) {
        console.error("Index Management - IndexService - applyPolicy:", err);
        // return { ok: false, error: err.message };
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });
    _defineProperty(this, "editRolloverAlias", async (context, request, response) => {
      try {
        const {
          alias,
          index
        } = request.body;
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const params = {
          index,
          body: {
            [_constants.Setting.RolloverAlias]: alias
          }
        };
        const rollOverResponse = await callWithRequest("indices.putSettings", params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: rollOverResponse
          }
        });
      } catch (err) {
        console.error("Index Management - IndexService - editRolloverAlias", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });
    this.osDriver = osDriver;
  }
}
exports.default = IndexService;
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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