"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
var _lodash = _interopRequireDefault(require("lodash"));
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
class PolicyService {
  constructor(osDriver) {
    _defineProperty(this, "osDriver", void 0);
    /**
     * Calls backend Put Policy API
     */
    _defineProperty(this, "putPolicy", async (context, request, response) => {
      try {
        const {
          id
        } = request.params;
        const {
          seqNo,
          primaryTerm
        } = request.query;
        let method = "ism.putPolicy";
        let params = {
          policyId: id,
          ifSeqNo: seqNo,
          ifPrimaryTerm: primaryTerm,
          body: JSON.stringify(request.body)
        };
        if (seqNo === undefined || primaryTerm === undefined) {
          method = "ism.createPolicy";
          params = {
            policyId: id,
            body: JSON.stringify(request.body)
          };
        }
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const putPolicyResponse = await callWithRequest(method, params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: putPolicyResponse
          }
        });
      } catch (err) {
        console.error("Index Management - PolicyService - putPolicy:", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });
    /**
     * Calls backend Delete Policy API
     */
    _defineProperty(this, "deletePolicy", async (context, request, response) => {
      try {
        const {
          id
        } = request.params;
        const params = {
          policyId: id
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const deletePolicyResponse = await callWithRequest("ism.deletePolicy", params);
        if (deletePolicyResponse.result !== "deleted") {
          return response.custom({
            statusCode: 200,
            body: {
              ok: false,
              error: deletePolicyResponse.result
            }
          });
        }
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: true
          }
        });
      } catch (err) {
        console.error("Index Management - PolicyService - deletePolicy:", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });
    /**
     * Calls backend Get Policy API
     */
    _defineProperty(this, "getPolicy", async (context, request, response) => {
      try {
        const {
          id
        } = request.params;
        const params = {
          policyId: id
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const getResponse = await callWithRequest("ism.getPolicy", params);
        const policy = _lodash.default.get(getResponse, "policy", null);
        const seqNo = _lodash.default.get(getResponse, "_seq_no");
        const primaryTerm = _lodash.default.get(getResponse, "_primary_term");
        if (policy) {
          return response.custom({
            statusCode: 200,
            body: {
              ok: true,
              response: {
                id,
                seqNo: seqNo,
                primaryTerm: primaryTerm,
                policy: policy
              }
            }
          });
        } else {
          return response.custom({
            statusCode: 200,
            body: {
              ok: false,
              error: "Failed to load policy"
            }
          });
        }
      } catch (err) {
        console.error("Index Management - PolicyService - getPolicy:", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });
    /**
     * Calls backend Get Policy API
     */
    _defineProperty(this, "getPolicies", async (context, request, response) => {
      try {
        const {
          from = 0,
          size = 20,
          search,
          sortDirection = "desc",
          sortField = "id"
        } = request.query;
        const policySorts = {
          id: "policy.policy_id.keyword",
          "policy.policy.description": "policy.description.keyword",
          "policy.policy.last_updated_time": "policy.last_updated_time"
        };
        const params = {
          from,
          size,
          sortOrder: sortDirection,
          sortField: policySorts[sortField] || policySorts.id,
          queryString: search.trim() ? `*${search.trim().split(" ").join("* *")}*` : "*"
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const getResponse = await callWithRequest("ism.getPolicies", params);
        const policies = getResponse.policies.map(p => ({
          seqNo: p._seq_no,
          primaryTerm: p._primary_term,
          id: p._id,
          policy: p.policy
        }));
        const totalPolicies = getResponse.total_policies;
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: {
              policies: policies,
              totalPolicies
            }
          }
        });
      } catch (err) {
        if (err.statusCode === 404 && err.body.error.type === "index_not_found_exception") {
          return response.custom({
            statusCode: 200,
            body: {
              ok: true,
              response: {
                policies: [],
                totalPolicies: 0
              }
            }
          });
        }
        console.error("Index Management - PolicyService - getPolicies", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });
    this.osDriver = osDriver;
  }
}
exports.default = PolicyService;
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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