"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.FileSystemDataAdaptor = void 0;
var fs = _interopRequireWildcard(require("fs/promises"));
var _path = _interopRequireDefault(require("path"));
var _utils = require("./utils");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && Object.prototype.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
// Check if a location is a directory without an exception if location not found
const safeIsDirectory = async maybeDirectory => {
  try {
    return (await fs.lstat(maybeDirectory)).isDirectory();
  } catch (_err) {
    return false;
  }
};

/**
 * A CatalogDataAdaptor that reads from the local filesystem.
 * Used to read default Integrations shipped in the in-product catalog at `__data__`.
 */
class FileSystemDataAdaptor {
  /**
   * Creates a new FileSystemCatalogDataAdaptor instance.
   *
   * @param directory The base directory from which to read files. This is not sanitized.
   */
  constructor(directory) {
    _defineProperty(this, "isConfigLocalized", false);
    _defineProperty(this, "directory", void 0);
    this.directory = directory;
  }
  async readFile(filename, type) {
    let content;
    try {
      content = await fs.readFile(_path.default.join(this.directory, type !== null && type !== void 0 ? type : '.', filename), {
        encoding: 'utf-8'
      });
    } catch (err) {
      return {
        ok: false,
        error: err
      };
    }
    // First try to parse as JSON, then NDJSON, then fail.
    try {
      const parsed = JSON.parse(content);
      return {
        ok: true,
        value: parsed
      };
    } catch (err) {
      const parsed = await (0, _utils.tryParseNDJson)(content);
      if (parsed) {
        return {
          ok: true,
          value: parsed
        };
      }
      return {
        ok: false,
        error: new Error('Unable to parse file as JSON or NDJson', {
          cause: err
        })
      };
    }
  }
  async readFileRaw(filename, type) {
    try {
      const buffer = await fs.readFile(_path.default.join(this.directory, type !== null && type !== void 0 ? type : '.', filename));
      return {
        ok: true,
        value: buffer
      };
    } catch (err) {
      return {
        ok: false,
        error: err
      };
    }
  }
  async findIntegrations(dirname = '.') {
    try {
      const integrations = [];
      await this.collectIntegrationsRecursive(dirname, integrations);
      return {
        ok: true,
        value: integrations
      };
    } catch (err) {
      return {
        ok: false,
        error: err
      };
    }
  }
  async collectIntegrationsRecursive(dirname, integrations) {
    const entries = await fs.readdir(_path.default.join(this.directory, dirname));
    for (const entry of entries) {
      const fullPath = _path.default.join(dirname, entry);
      const isDirectory = (await this.getDirectoryType(fullPath)) === 'integration';
      if (isDirectory) {
        integrations.push(fullPath);
      } else if ((await this.getDirectoryType(fullPath)) === 'repository') {
        await this.collectIntegrationsRecursive(fullPath, integrations);
      }
    }
  }
  async findIntegrationVersions(dirname = '.') {
    let files;
    const integPath = _path.default.join(this.directory, dirname);
    try {
      files = await fs.readdir(integPath);
    } catch (err) {
      return {
        ok: false,
        error: err
      };
    }
    const versions = [];
    for (const file of files) {
      // TODO handle nested repositories -- assumes integrations are 1 level deep
      if (_path.default.extname(file) === '.json' && file.startsWith(`${_path.default.basename(integPath)}-`)) {
        const version = file.substring(_path.default.basename(integPath).length + 1, file.length - 5);
        if (!version.match(/^\d+(\.\d+)*$/)) {
          continue;
        }
        versions.push(version);
      }
    }
    return {
      ok: true,
      value: versions
    };
  }
  async getDirectoryType(dirname) {
    const isDir = await safeIsDirectory(_path.default.join(this.directory, dirname !== null && dirname !== void 0 ? dirname : '.'));
    if (!isDir) {
      return 'unknown';
    }
    // Sloppily just check for one mandatory integration directory to distinguish.
    // Improve if we need to distinguish a repository with an integration named "schemas".
    const hasSchemas = await safeIsDirectory(_path.default.join(this.directory, dirname !== null && dirname !== void 0 ? dirname : '.', 'schemas'));
    return hasSchemas ? 'integration' : 'repository';
  }
  join(filename) {
    return new FileSystemDataAdaptor(_path.default.join(this.directory, filename));
  }
}
exports.FileSystemDataAdaptor = FileSystemDataAdaptor;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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