"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isClusterName = exports.createCluster = exports.IsDataTypeAnalogOrDiscrete = exports.getGlobalCommand = exports.getCluster = void 0;
const definition_1 = require("./definition");
const DataTypeValueType = {
    discrete: [
        definition_1.DataType.DATA8, definition_1.DataType.DATA16, definition_1.DataType.DATA24, definition_1.DataType.DATA32, definition_1.DataType.DATA40,
        definition_1.DataType.DATA48, definition_1.DataType.DATA56, definition_1.DataType.DATA64, definition_1.DataType.BOOLEAN,
        definition_1.DataType.BITMAP8, definition_1.DataType.BITMAP16, definition_1.DataType.BITMAP24, definition_1.DataType.BITMAP32, definition_1.DataType.BITMAP40,
        definition_1.DataType.BITMAP48, definition_1.DataType.BITMAP56, definition_1.DataType.BITMAP64, definition_1.DataType.ENUM8, definition_1.DataType.ENUM16,
        definition_1.DataType.OCTET_STR, definition_1.DataType.CHAR_STR, definition_1.DataType.LONG_OCTET_STR, definition_1.DataType.LONG_CHAR_STR, definition_1.DataType.ARRAY,
        definition_1.DataType.STRUCT, definition_1.DataType.SET, definition_1.DataType.BAG, definition_1.DataType.CLUSTER_ID, definition_1.DataType.ATTR_ID, definition_1.DataType.BAC_OID,
        definition_1.DataType.IEEE_ADDR, definition_1.DataType.SEC_KEY,
    ],
    analog: [
        definition_1.DataType.UINT8, definition_1.DataType.UINT16, definition_1.DataType.UINT24, definition_1.DataType.UINT32, definition_1.DataType.UINT40,
        definition_1.DataType.UINT48, definition_1.DataType.UINT56,
        definition_1.DataType.INT8, definition_1.DataType.INT16, definition_1.DataType.INT24, definition_1.DataType.INT32, definition_1.DataType.INT40,
        definition_1.DataType.INT48, definition_1.DataType.INT56, definition_1.DataType.SEMI_PREC, definition_1.DataType.SINGLE_PREC, definition_1.DataType.DOUBLE_PREC,
        definition_1.DataType.TOD, definition_1.DataType.DATE, definition_1.DataType.UTC,
    ],
};
function IsDataTypeAnalogOrDiscrete(dataType) {
    if (DataTypeValueType.discrete.includes(dataType)) {
        return 'DISCRETE';
    }
    else if (DataTypeValueType.analog.includes(dataType)) {
        return 'ANALOG';
    }
    else {
        throw new Error(`Don't know value type for '${definition_1.DataType[dataType]}'`);
    }
}
exports.IsDataTypeAnalogOrDiscrete = IsDataTypeAnalogOrDiscrete;
function getClusterDefinition(key, manufacturerCode, customClusters) {
    let name;
    const clusters = {
        ...customClusters, // Custom clusters have a higher priority than Zcl clusters (custom cluster has a DIFFERENT name than Zcl clusters)
        ...definition_1.Clusters,
        ...customClusters, // Custom clusters should override Zcl clusters (custom cluster has the SAME name than Zcl clusters)
    };
    if (typeof key === 'number') {
        if (manufacturerCode) {
            name = Object.entries(clusters)
                .find((e) => e[1].ID === key && e[1].manufacturerCode === manufacturerCode)?.[0];
        }
        if (!name) {
            name = Object.entries(clusters).find((e) => e[1].ID === key && !e[1].manufacturerCode)?.[0];
        }
        if (!name) {
            name = Object.entries(clusters).find((e) => e[1].ID === key)?.[0];
        }
    }
    else {
        name = key;
    }
    let cluster = clusters[name];
    if (!cluster) {
        if (typeof key === 'number') {
            name = key.toString();
            cluster = { attributes: {}, commands: {}, commandsResponse: {}, manufacturerCode: null, ID: key };
        }
        else {
            throw new Error(`Cluster with name '${key}' does not exist`);
        }
    }
    return { name, cluster };
}
function createCluster(name, cluster, manufacturerCode = null) {
    const attributes = Object.assign({}, ...Object.entries(cluster.attributes).map(([k, v]) => ({ [k]: { ...v, name: k } })));
    const commands = Object.assign({}, ...Object.entries(cluster.commands).map(([k, v]) => ({ [k]: { ...v, name: k } })));
    const commandsResponse = Object.assign({}, ...Object.entries(cluster.commandsResponse).map(([k, v]) => ({ [k]: { ...v, name: k } })));
    const getAttributeInternal = (key) => {
        let result = null;
        if (typeof key === 'number') {
            if (manufacturerCode) {
                result = Object.values(attributes).find((a) => {
                    return a.ID === key && a.manufacturerCode === manufacturerCode;
                });
            }
            if (!result) {
                result = Object.values(attributes).find((a) => a.ID === key && a.manufacturerCode == null);
            }
        }
        else {
            result = Object.values(attributes).find((a) => a.name === key);
        }
        return result;
    };
    const getAttribute = (key) => {
        const result = getAttributeInternal(key);
        if (!result) {
            throw new Error(`Cluster '${name}' has no attribute '${key}'`);
        }
        return result;
    };
    const hasAttribute = (key) => {
        const result = getAttributeInternal(key);
        return !!result;
    };
    const getCommand = (key) => {
        let result = null;
        if (typeof key === 'number') {
            result = Object.values(commands).find((a) => a.ID === key);
        }
        else {
            result = Object.values(commands).find((a) => a.name === key);
        }
        if (!result) {
            throw new Error(`Cluster '${name}' has no command '${key}'`);
        }
        return result;
    };
    const getCommandResponse = (key) => {
        let result = null;
        if (typeof key === 'number') {
            result = Object.values(commandsResponse).find((a) => a.ID === key);
        }
        else {
            result = Object.values(commandsResponse).find((a) => a.name === key);
        }
        if (!result) {
            throw new Error(`Cluster '${name}' has no command response '${key}'`);
        }
        return result;
    };
    return {
        ID: cluster.ID,
        attributes,
        manufacturerCode: cluster.manufacturerCode,
        name,
        commands,
        // eslint-disable-next-line
        commandsResponse: Object.assign({}, ...Object.entries(cluster.commandsResponse).map(([k, v]) => ({ [k]: { ...v, name: k } }))),
        getAttribute,
        hasAttribute,
        getCommand,
        getCommandResponse,
    };
}
exports.createCluster = createCluster;
function getCluster(key, manufacturerCode, customClusters) {
    const { name, cluster } = getClusterDefinition(key, manufacturerCode, customClusters);
    return createCluster(name, cluster, manufacturerCode);
}
exports.getCluster = getCluster;
function getGlobalCommand(key) {
    let name;
    if (typeof key === 'number') {
        for (const commandName in definition_1.Foundation) {
            if (definition_1.Foundation[commandName].ID === key) {
                name = commandName;
                break;
            }
        }
    }
    else {
        name = key;
    }
    const command = definition_1.Foundation[name];
    if (!command) {
        throw new Error(`Global command with key '${key}' does not exist`);
    }
    const result = {
        ID: command.ID,
        name,
        parameters: command.parameters,
    };
    if (command.response != undefined) {
        result.response = command.response;
    }
    return result;
}
exports.getGlobalCommand = getGlobalCommand;
function isClusterName(name) {
    return (name in definition_1.Clusters);
}
exports.isClusterName = isClusterName;
//# sourceMappingURL=utils.js.map