# -*- coding: utf-8 -*- #
# frozen_string_literal: true

module Rouge
  module Lexers
    class OpenEdge < RegexLexer
      tag 'openedge'
      aliases 'abl'
      filenames '*.w', '*.i', '*.p', '*.cls', '*.df'
      mimetypes 'text/x-openedge'

      title 'OpenEdge ABL'
      desc 'The OpenEdge ABL programming language'

      # optional comment or whitespace
      ws = %r((?:\s|//.*?\n|/[*].*?[*]/)+)
      id = /[a-zA-Z_&{}!][a-zA-Z0-9_\-&!}]*/

      def self.keywords
        @keywords ||= Set.new %w(
          ABORT ABS ABSO ABSOL ABSOLU ABSOLUT ABSOLUTE ABSTRACT ACCELERATOR ACCEPT-CHANGES ACCEPT-ROW-CHANGES
          ACCUM ACCUMU ACCUMUL ACCUMULA ACCUMULAT ACCUMULATE ACROSS ACTIVE ACTIVE-FORM ACTIVE-WINDOW ACTOR ADD
          ADD-BUFFER ADD-CALC-COL ADD-CALC-COLU ADD-CALC-COLUM ADD-CALC-COLUMN ADD-COLUMNS-FROM
          ADD-EVENTS-PROC ADD-EVENTS-PROCE ADD-EVENTS-PROCED ADD-EVENTS-PROCEDU ADD-EVENTS-PROCEDUR
          ADD-EVENTS-PROCEDURE ADD-FIELDS-FROM ADD-FIRST ADD-HEADER-ENTRY ADD-INDEX-FIELD ADD-INTERVAL
          ADD-LAST ADD-LIKE-COLUMN ADD-LIKE-FIELD ADD-LIKE-INDEX ADD-NEW-FIELD ADD-NEW-INDEX       
          ADD-SCHEMA-LOCATION ADD-SOURCE-BUFFER ADD-SUPER-PROC ADD-SUPER-PROCE ADD-SUPER-PROCED
          ADD-SUPER-PROCEDU ADD-SUPER-PROCEDUR ADD-SUPER-PROCEDURE ADM-DATA ADVISE AFTER-BUFFER AFTER-ROWID
          AFTER-TABLE ALERT-BOX ALIAS ALL ALLOW-COLUMN-SEARCHING ALLOW-PREV-DESERIALIZATION ALLOW-REPLICATION
          ALTER ALTERNATE-KEY ALWAYS-ON-TOP AMBIG AMBIGU AMBIGUO AMBIGUOU AMBIGUOUS ANALYZ ANALYZE AND
          ANSI-ONLY ANY ANY-KEY ANY-PRINTABLE ANYWHERE APPEND APPEND-CHILD APPEND-LINE APPL-ALERT APPL-ALERT-
          APPL-ALERT-B APPL-ALERT-BO APPL-ALERT-BOX APPL-ALERT-BOXE APPL-ALERT-BOXES APPL-CONTEXT-ID
          APPLICATION APPLY APPLY-CALLBACK APPSERVER-INFO APPSERVER-PASSWORD APPSERVER-USERID ARRAY-M ARRAY-ME
          ARRAY-MES ARRAY-MESS ARRAY-MESSA ARRAY-MESSAG ARRAY-MESSAGE AS ASC ASCE ASCEN ASCEND ASCENDI
          ASCENDIN ASCENDING AS-CURSOR ASK-OVERWRITE ASSEMBLY ASSIGN ASYNCHRONOUS ASYNC-REQUEST-COUNT
          ASYNC-REQUEST-HANDLE AT ATTACH ATTACH-DATA-SOURCE ATTACHED-PAIRLIST ATTACHMENT ATTR ATTR-
          ATTRIBUTE-NAMES ATTRIBUTE-TYPE ATTR-S ATTR-SP ATTR-SPA ATTR-SPAC ATTR-SPACE AUDIT-CONTROL
          AUDIT-ENABLED AUDIT-EVENT-CONTEXT AUDIT-POLICY AUTHENTICATION-FAILED AUTHORIZATION AUTO-COMP
          AUTO-COMPL AUTO-COMPLE AUTO-COMPLET AUTO-COMPLETI AUTO-COMPLETIO AUTO-COMPLETION AUTO-DELETE
          AUTO-DELETE-XML AUTO-ENDKEY AUTO-END-KEY AUTO-GO AUTO-IND AUTO-INDE AUTO-INDEN AUTO-INDENT AUTOMATIC
          AUTO-RESIZE AUTO-RET AUTO-RETU AUTO-RETUR AUTO-RETURN AUTO-SYNCHRONIZE AUTO-VAL AUTO-VALI AUTO-VALID
          AUTO-VALIDA AUTO-VALIDAT AUTO-VALIDATE AUTO-Z AUTO-ZA AUTO-ZAP AVAIL AVAILA AVAILAB AVAILABL
          AVAILABLE AVAILABLE-FORMATS AVE AVER AVERA AVERAG AVERAGE AVG BACK BACKG BACKGR BACKGRO BACKGROU
          BACKGROUN BACKGROUND BACKSPACE BACK-TAB BACKWARD BACKWARDS BASE64 BASE64-DECODE BASE64-ENCODE
          BASE-ADE BASE-KEY BASIC-LOGGING BATCH BATCH-MODE BATCH-SIZE BEFORE-BUFFER BEFORE-H BEFORE-HI
          BEFORE-HID BEFORE-HIDE BEFORE-ROWID BEFORE-TABLE BEGIN-EVENT-GROUP BEGINS BELL BETWEEN BGC BGCO
          BGCOL BGCOLO BGCOLOR BIG-ENDIAN BINARY BIND BIND-WHERE BLANK BLOB BLOCK BLOCK-ITERATION-DISPLAY
          BLOCK-LEV BLOCK-LEVE BLOCK-LEVEL BORDER-B BORDER-BO BORDER-BOT BORDER-BOTT BORDER-BOTTO
          BORDER-BOTTOM BORDER-BOTTOM-C BORDER-BOTTOM-CH BORDER-BOTTOM-CHA BORDER-BOTTOM-CHAR
          BORDER-BOTTOM-CHARS BORDER-BOTTOM-P BORDER-BOTTOM-PI BORDER-BOTTOM-PIX BORDER-BOTTOM-PIXE
          BORDER-BOTTOM-PIXEL BORDER-BOTTOM-PIXELS BORDER-L BORDER-LE BORDER-LEF BORDER-LEFT BORDER-LEFT-C
          BORDER-LEFT-CH BORDER-LEFT-CHA BORDER-LEFT-CHAR BORDER-LEFT-CHARS BORDER-LEFT-P BORDER-LEFT-PI
          BORDER-LEFT-PIX BORDER-LEFT-PIXE BORDER-LEFT-PIXEL BORDER-LEFT-PIXELS BORDER-R BORDER-RI BORDER-RIG
          BORDER-RIGH BORDER-RIGHT BORDER-RIGHT-C BORDER-RIGHT-CH BORDER-RIGHT-CHA BORDER-RIGHT-CHAR
          BORDER-RIGHT-CHARS BORDER-RIGHT-P BORDER-RIGHT-PI BORDER-RIGHT-PIX BORDER-RIGHT-PIXE
          BORDER-RIGHT-PIXEL BORDER-RIGHT-PIXELS BORDER-T BORDER-TO BORDER-TOP BORDER-TOP-C BORDER-TOP-CH
          BORDER-TOP-CHA BORDER-TOP-CHAR BORDER-TOP-CHARS BORDER-TOP-P BORDER-TOP-PI BORDER-TOP-PIX
          BORDER-TOP-PIXE BORDER-TOP-PIXEL BORDER-TOP-PIXELS BOTH BOTTOM BOTTOM-COLUMN BOX BOX-SELECT
          BOX-SELECTA BOX-SELECTAB BOX-SELECTABL BOX-SELECTABLE BREAK BREAK-LINE BROWSE BROWSE-COLUMN-DATA-TYPES
          BROWSE-COLUMN-FORMATS BROWSE-COLUMN-LABELS BROWSE-HEADER BTOS BUFFER BUFFER-CHARS BUFFER-COMP
          BUFFER-COMPA BUFFER-COMPAR BUFFER-COMPARE BUFFER-COPY BUFFER-CREATE BUFFER-DELETE BUFFER-FIELD
          BUFFER-GROUP-ID BUFFER-GROUP-NAME BUFFER-HANDLE BUFFER-LINES BUFFER-N BUFFER-NA BUFFER-NAM
          BUFFER-NAME BUFFER-PARTITION-ID BUFFER-RELEAS BUFFER-RELEASE BUFFER-TENANT-ID BUFFER-TENANT-NAME
          BUFFER-VALIDATE BUFFER-VALUE BUTTON BUTTONS BY BY-POINTER BY-REFERENCE BYTE BYTES-READ BYTES-WRITTEN
          BY-VALUE BY-VARIANT-POINT BY-VARIANT-POINTE BY-VARIANT-POINTER CACHE CACHE-SIZE CALL CALL-NAME
          CALL-TYPE CANCEL-BREAK CANCEL-BUTTON CANCELLED CANCEL-PICK CANCEL-REQUESTS CANCEL-REQUESTS-AFTER
          CAN-CREA CAN-CREAT CAN-CREATE CAN-DELE CAN-DELET CAN-DELETE CAN-DO CAN-DO-DOMAIN-SUPPORT CAN-FIND
          CAN-QUERY CAN-READ CAN-SET CAN-WRIT CAN-WRITE CAPS CAREFUL-PAINT CASE CASE-SEN CASE-SENS CASE-SENSI
          CASE-SENSIT CASE-SENSITI CASE-SENSITIV CASE-SENSITIVE CAST CATCH CDECL CENTER CENTERE CENTERED
          CHAINED CHAR CHARA CHARAC CHARACT CHARACTE CHARACTER CHARACTER_LENGTH CHARSET CHECK CHECKED
          CHECK-MEM-STOMP CHILD-BUFFER CHILD-NUM CHOICES CHOOSE CHR CLASS CLASS-TYPE CLEAR CLEAR-APPL-CONTEXT
          CLEAR-LOG CLEAR-SELECT CLEAR-SELECTI CLEAR-SELECTIO CLEAR-SELECTION CLEAR-SORT-ARROW
          CLEAR-SORT-ARROWS CLIENT-CONNECTION-ID CLIENT-PRINCIPAL CLIENT-TTY CLIENT-TYPE CLIENT-WORKSTATION
          CLIPBOARD CLOB CLONE-NODE CLOSE CLOSE-LOG CODE CODEBASE-LOCATOR CODEPAGE CODEPAGE-CONVERT COL
          COLLATE COL-OF COLON COLON-ALIGN COLON-ALIGNE COLON-ALIGNED COLOR COLOR-TABLE COLUMN COLUMN-BGC
          COLUMN-BGCO COLUMN-BGCOL COLUMN-BGCOLO COLUMN-BGCOLOR COLUMN-CODEPAGE COLUMN-DCOLOR COLUMN-FGC
          COLUMN-FGCO COLUMN-FGCOL COLUMN-FGCOLO COLUMN-FGCOLOR COLUMN-FONT COLUMN-LAB COLUMN-LABE
          COLUMN-LABEL COLUMN-LABEL-BGC COLUMN-LABEL-BGCO COLUMN-LABEL-BGCOL COLUMN-LABEL-BGCOLO
          COLUMN-LABEL-BGCOLOR COLUMN-LABEL-DCOLOR COLUMN-LABEL-FGC COLUMN-LABEL-FGCO COLUMN-LABEL-FGCOL
          COLUMN-LABEL-FGCOLO COLUMN-LABEL-FGCOLOR COLUMN-LABEL-FONT COLUMN-LABEL-HEIGHT-C
          COLUMN-LABEL-HEIGHT-CH COLUMN-LABEL-HEIGHT-CHA COLUMN-LABEL-HEIGHT-CHAR COLUMN-LABEL-HEIGHT-CHARS
          COLUMN-LABEL-HEIGHT-P COLUMN-LABEL-HEIGHT-PI COLUMN-LABEL-HEIGHT-PIX COLUMN-LABEL-HEIGHT-PIXE
          COLUMN-LABEL-HEIGHT-PIXEL COLUMN-LABEL-HEIGHT-PIXELS COLUMN-MOVABLE COLUMN-OF COLUMN-PFC COLUMN-PFCO
          COLUMN-PFCOL COLUMN-PFCOLO COLUMN-PFCOLOR COLUMN-READ-ONLY COLUMN-RESIZABLE COLUMNS COLUMN-SC
          COLUMN-SCR COLUMN-SCRO COLUMN-SCROL COLUMN-SCROLL COLUMN-SCROLLI COLUMN-SCROLLIN COLUMN-SCROLLING
          COMBO-BOX COM-HANDLE COMMAND COMPARE COMPARES COMPILE COMPILER COMPLETE COMPONENT-HANDLE
          COMPONENT-SELF COM-SELF CONFIG-NAME CONNECT CONNECTED CONSTRAINED CONSTRUCTOR CONTAINER-EVENT
          CONTAINS CONTENTS CONTEXT CONTEXT-HELP CONTEXT-HELP-FILE CONTEXT-HELP-ID CONTEXT-POP CONTEXT-POPU
          CONTEXT-POPUP CONTROL CONTROL-BOX CONTROL-CONT CONTROL-CONTA CONTROL-CONTAI CONTROL-CONTAIN
          CONTROL-CONTAINE CONTROL-CONTAINER CONTROL-FRAM CONTROL-FRAME CONVERT CONVERT-3D CONVERT-3D-
          CONVERT-3D-C CONVERT-3D-CO CONVERT-3D-COL CONVERT-3D-COLO CONVERT-3D-COLOR CONVERT-3D-COLORS
          CONVERT-TO-OFFS CONVERT-TO-OFFSE CONVERT-TO-OFFSET COPY COPY-DATASET COPY-LOB COPY-SAX-ATTRIBUTES
          COPY-TEMP-TABLE COUNT COUNT-OF COVERAGE CPCASE CPCOLL CPINT CPINTE CPINTER CPINTERN CPINTERNA
          CPINTERNAL CPLOG CPPRINT CPRCODEIN CPRCODEOUT CPSTREAM CPTERM CRC-VAL CRC-VALU CRC-VALUE CREATE
          CREATE-LIKE CREATE-LIKE-SEQUENTIAL CREATE-NODE CREATE-NODE-NAMESPACE CREATE-ON-ADD
          CREATE-RESULT-LIST-ENTRY CREATE-TEST-FILE CTOS CURRENT CURRENT_DATE CURRENT-CHANGED CURRENT-COLUMN
          CURRENT-ENV CURRENT-ENVI CURRENT-ENVIR CURRENT-ENVIRO CURRENT-ENVIRON CURRENT-ENVIRONM
          CURRENT-ENVIRONME CURRENT-ENVIRONMEN CURRENT-ENVIRONMENT CURRENT-ITERATION CURRENT-LANG
          CURRENT-LANGU CURRENT-LANGUA CURRENT-LANGUAG CURRENT-LANGUAGE CURRENT-QUERY CURRENT-REQUEST-INFO
          CURRENT-RESPONSE-INFO CURRENT-RESULT-ROW CURRENT-ROW-MODIFIED CURRENT-VALUE CURRENT-WINDOW CURS
          CURSO CURSOR CURSOR-CHAR CURSOR-DOWN CURSOR-LEFT CURSOR-LINE CURSOR-OFFSET CURSOR-RIGHT CURSOR-UP
          CUT DATA-B DATABASE DATA-BI DATA-BIN DATA-BIND DATA-ENTRY-RET DATA-ENTRY-RETU DATA-ENTRY-RETUR
          DATA-ENTRY-RETURN DATA-REFRESH-LINE DATA-REFRESH-PAGE DATA-REL DATA-RELA DATA-RELAT DATA-RELATI
          DATA-RELATIO DATA-RELATION DATASERVERS DATASET DATASET-HANDLE DATA-SOURCE DATA-SOURCE-COMPLETE-MAP
          DATA-SOURCE-MODIFIED DATA-SOURCE-ROWID DATA-T DATA-TY DATA-TYP DATA-TYPE DATE DATE-F DATE-FO
          DATE-FOR DATE-FORM DATE-FORMA DATE-FORMAT DATETIME DATETIME-TZ DAY DBCODEPAGE DBCOLLATION DB-CONTEXT
          DB-LIST DBNAME DBPARAM DB-REFERENCES DB-REMOTE-HOST DBREST DBRESTR DBRESTRI DBRESTRIC DBRESTRICT
          DBRESTRICTI DBRESTRICTIO DBRESTRICTION DBRESTRICTIONS DBTASKID DBTYPE DBVERS DBVERSI DBVERSIO
          DBVERSION DCOLOR DDE DDE-ERROR DDE-I DDE-ID DDE-ITEM DDE-NAME DDE-NOTIFY DDE-TOPIC DEBLANK DEBU
          DEBUG DEBUG-ALERT DEBUGGER DEBUG-LIST DEBUG-SET-TENANT DEC DECI DECIM DECIMA DECIMAL DECIMALS
          DECLARE DECLARE-NAMESPACE DECRYPT DEF DEFAULT DEFAULT-ACTION DEFAULT-BUFFER-HANDLE DEFAULT-BUT
          DEFAULT-BUTT DEFAULT-BUTTO DEFAULT-BUTTON DEFAULT-COMMIT DEFAULT-EX DEFAULT-EXT DEFAULT-EXTE
          DEFAULT-EXTEN DEFAULT-EXTENS DEFAULT-EXTENSI DEFAULT-EXTENSIO DEFAULT-EXTENSION DEFAULT-NOXL
          DEFAULT-NOXLA DEFAULT-NOXLAT DEFAULT-NOXLATE DEFAULT-POP-UP DEFAULT-STRING DEFAULT-VALUE
          DEFAULT-WINDOW DEFAUT-B DEFER-LOB-FETCH DEFI DEFIN DEFINE DEFINED DEFINE-USER-EVENT-MANAGER DEL
          DELEGATE DELETE DELETE-CHAR DELETE-CHARACTER DELETE-COLUMN DELETE-CURRENT-ROW DELETE-END-LINE
          DELETE-FIELD DELETE-HEADER-ENTRY DELETE-LINE DELETE-NODE DELETE-RESULT-LIST-ENTRY
          DELETE-SELECTED-ROW DELETE-SELECTED-ROWS DELETE-WORD DELIMITER DESC DESCE DESCEN DESCEND DESCENDI
          DESCENDIN DESCENDING DESCRIPT DESCRIPTI DESCRIPTIO DESCRIPTION DESELECT DESELECT-EXTEND
          DESELECT-FOCUSED-ROW DESELECTION DESELECTION-EXTEND DESELECT-ROWS DESELECT-SELECTED-ROW DESTRUCTOR
          DETACH DETACH-DATA-SOURCE DIALOG-BOX DIALOG-HELP DICT DICTI DICTIO DICTION DICTIONA DICTIONAR
          DICTIONARY DIR DIRECTORY DISABLE DISABLE-AUTO-ZAP DISABLE-CONNECTIONS DISABLED DISABLE-DUMP-TRIGGERS
          DISABLE-LOAD-TRIGGERS DISCON DISCONN DISCONNE DISCONNEC DISCONNECT DISMISS-MENU DISP DISPL DISPLA
          DISPLAY DISPLAY-MESSAGE DISPLAY-T DISPLAY-TIMEZONE DISPLAY-TY DISPLAY-TYP DISPLAY-TYPE DISTINCT
          DLL-CALL-TYPE DO DOMAIN-DESCRIPTION DOMAIN-NAME DOMAIN-TYPE DOS DOS-END DOTNET-CLR-LOADED DOUBLE
          DOWN DRAG-ENABLED DROP DROP-DOWN DROP-DOWN-LIST DROP-FILE-NOTIFY DROP-TARGET DS-CLOSE-CURSOR
          DSLOG-MANAGER DUMP DUMP-LOGGING-NOW DYNAMIC DYNAMIC-CAST DYNAMIC-CURRENT-VALUE DYNAMIC-ENUM
          DYNAMIC-FUNC DYNAMIC-FUNCT DYNAMIC-FUNCTI DYNAMIC-FUNCTIO DYNAMIC-FUNCTION DYNAMIC-INVOKE
          DYNAMIC-NEW DYNAMIC-NEXT-VALUE DYNAMIC-PROPERTY EACH ECHO EDGE EDGE-C EDGE-CH EDGE-CHA EDGE-CHAR
          EDGE-CHARS EDGE-P EDGE-PI EDGE-PIX EDGE-PIXE EDGE-PIXEL EDGE-PIXELS EDIT-CAN-PASTE EDIT-CAN-UNDO
          EDIT-CLEAR EDIT-COPY EDIT-CUT EDITING EDITOR EDITOR-BACKTAB EDITOR-TAB EDIT-PASTE EDIT-UNDO ELSE
          EMPTY EMPTY-DATASET EMPTY-SELECTION EMPTY-TEMP-TABLE ENABLE ENABLE-CONNECTIONS ENABLED
          ENABLED-FIELDS ENCODE ENCODE-DOMAIN-ACCESS-CODE ENCODING ENCRYPT ENCRYPT-AUDIT-MAC-KEY
          ENCRYPTION-SALT END END-BOX-SELECTION END-DOCUMENT END-ELEMENT END-ERROR END-EVENT-GROUP
          END-FILE-DROP ENDKEY END-KEY END-MOVE END-RESIZE END-ROW-RESIZE END-SEARCH END-USER-PROMPT ENTERED
          ENTER-MENUBAR ENTITY-EXPANSION-LIMIT ENTRY ENTRY-TYPES-LIST ENUM EQ ERROR ERROR-COL ERROR-COLU
          ERROR-COLUM ERROR-COLUMN ERROR-OBJECT ERROR-OBJECT-DETAIL ERROR-ROW ERROR-STACK-TRACE ERROR-STAT
          ERROR-STATU ERROR-STATUS ERROR-STRING ESCAPE ETIME EVENT EVENT-GROUP-ID EVENT-PROCEDURE
          EVENT-PROCEDURE-CONTEXT EVENTS EVENT-T EVENT-TY EVENT-TYP EVENT-TYPE EXCEPT EXCLUSIVE EXCLUSIVE-ID
          EXCLUSIVE-L EXCLUSIVE-LO EXCLUSIVE-LOC EXCLUSIVE-LOCK EXCLUSIVE-WEB EXCLUSIVE-WEB- EXCLUSIVE-WEB-U
          EXCLUSIVE-WEB-US EXCLUSIVE-WEB-USE EXCLUSIVE-WEB-USER EXECUTE EXECUTION-LOG EXISTS EXIT EXIT-CODE
          EXP EXPAND EXPANDABLE EXPIRE EXPLICIT EXPORT EXPORT-PRINCIPAL EXTENDED EXTENT EXTERNAL EXTRACT FALSE
          FALSE-LEAKS FETCH FETCH-SELECTED-ROW FGC FGCO FGCOL FGCOLO FGCOLOR FIELD FIELDS FILE FILE-ACCESS-D
          FILE-ACCESS-DA FILE-ACCESS-DAT FILE-ACCESS-DATE FILE-ACCESS-T FILE-ACCESS-TI FILE-ACCESS-TIM
          FILE-ACCESS-TIME FILE-CREATE-D FILE-CREATE-DA FILE-CREATE-DAT FILE-CREATE-DATE FILE-CREATE-T
          FILE-CREATE-TI FILE-CREATE-TIM FILE-CREATE-TIME FILE-INFO FILE-INFOR FILE-INFORM FILE-INFORMA
          FILE-INFORMAT FILE-INFORMATI FILE-INFORMATIO FILE-INFORMATION FILE-MOD-D FILE-MOD-DA FILE-MOD-DAT
          FILE-MOD-DATE FILE-MOD-T FILE-MOD-TI FILE-MOD-TIM FILE-MOD-TIME FILENAME FILE-NAME FILE-OFF
          FILE-OFFS FILE-OFFSE FILE-OFFSET FILE-SIZE FILE-TYPE FILL FILLED FILL-IN FILL-MODE FILL-WHERE-STRING
          FILTERS FINAL FINALLY FIND FIND-BY-ROWID FIND-CASE-SENSITIVE FIND-CURRENT FINDER FIND-FIRST
          FIND-GLOBAL FIND-LAST FIND-NEXT FIND-NEXT-OCCURRENCE FIND-PREVIOUS FIND-PREV-OCCURRENCE FIND-SELECT
          FIND-UNIQUE FIND-WRAP-AROUND FIREHOSE-CURSOR FIRST FIRST-ASYNC FIRST-ASYNC- FIRST-ASYNCH-REQUEST
          FIRST-ASYNC-R FIRST-ASYNC-RE FIRST-ASYNC-REQ FIRST-ASYNC-REQU FIRST-ASYNC-REQUE FIRST-ASYNC-REQUES
          FIRST-ASYNC-REQUEST FIRST-BUFFER FIRST-CHILD FIRST-COLUMN FIRST-DATASET FIRST-DATA-SOURCE FIRST-FORM
          FIRST-OBJECT FIRST-OF FIRST-PROC FIRST-PROCE FIRST-PROCED FIRST-PROCEDU FIRST-PROCEDUR
          FIRST-PROCEDURE FIRST-QUERY FIRST-SERV FIRST-SERVE FIRST-SERVER FIRST-SERVER-SOCKET FIRST-SOCKET
          FIRST-TAB-I FIRST-TAB-IT FIRST-TAB-ITE FIRST-TAB-ITEM FIT-LAST-COLUMN FIX-CODEPAGE FIXED-ONLY FLAGS
          FLAT-BUTTON FLOAT FOCUS FOCUSED-ROW FOCUSED-ROW-SELECTED FOCUS-IN FONT FONT-TABLE FOR FORCE-FILE
          FORE FOREG FOREGR FOREGRO FOREGROU FOREGROUN FOREGROUND FOREIGN-KEY-HIDDEN FORM FORMA FORMAT
          FORMATTE FORMATTED FORM-INPUT FORM-LONG-INPUT FORWARD FORWARD-ONLY FORWARDS FRAGMEN FRAGMENT FRAM
          FRAME FRAME-COL FRAME-DB FRAME-DOWN FRAME-FIELD FRAME-FILE FRAME-INDE FRAME-INDEX FRAME-LINE
          FRAME-NAME FRAME-ROW FRAME-SPA FRAME-SPAC FRAME-SPACI FRAME-SPACIN FRAME-SPACING FRAME-VAL
          FRAME-VALU FRAME-VALUE FRAME-X FRAME-Y FREQUENCY FROM FROM-C FROM-CH FROM-CHA FROM-CHAR FROM-CHARS
          FROM-CUR FROM-CURR FROM-CURRE FROM-CURREN FROM-CURRENT FROMNOREORDER FROM-P FROM-PI FROM-PIX
          FROM-PIXE FROM-PIXEL FROM-PIXELS FULL-HEIGHT FULL-HEIGHT-C FULL-HEIGHT-CH FULL-HEIGHT-CHA
          FULL-HEIGHT-CHAR FULL-HEIGHT-CHARS FULL-HEIGHT-P FULL-HEIGHT-PI FULL-HEIGHT-PIX FULL-HEIGHT-PIXE
          FULL-HEIGHT-PIXEL FULL-HEIGHT-PIXELS FULL-PATHN FULL-PATHNA FULL-PATHNAM FULL-PATHNAME FULL-WIDTH
          FULL-WIDTH- FULL-WIDTH-C FULL-WIDTH-CH FULL-WIDTH-CHA FULL-WIDTH-CHAR FULL-WIDTH-CHARS FULL-WIDTH-P
          FULL-WIDTH-PI FULL-WIDTH-PIX FULL-WIDTH-PIXE FULL-WIDTH-PIXEL FULL-WIDTH-PIXELS FUNCTION
          FUNCTION-CALL-TYPE GATEWAY GATEWAYS GE GENERATE-MD5 GENERATE-PBE-KEY GENERATE-PBE-SALT
          GENERATE-RANDOM-KEY GENERATE-UUID GET GET-ATTR-CALL-TYPE GET-ATTRIBUTE GET-ATTRIBUTE-NODE
          GET-BINARY-DATA GET-BITS GET-BLUE GET-BLUE- GET-BLUE-V GET-BLUE-VA GET-BLUE-VAL GET-BLUE-VALU
          GET-BLUE-VALUE GET-BROWSE-COL GET-BROWSE-COLU GET-BROWSE-COLUM GET-BROWSE-COLUMN GET-BUFFER-HANDLE
          GETBYTE GET-BYTE GET-BYTE-ORDER GET-BYTES GET-BYTES-AVAILABLE GET-CALLBACK-PROC-CONTEXT
          GET-CALLBACK-PROC-NAME GET-CGI-LIST GET-CGI-LONG-VALUE GET-CGI-VALUE GET-CHANGES GET-CHILD
          GET-CHILD-REL GET-CHILD-RELA GET-CHILD-RELAT GET-CHILD-RELATI GET-CHILD-RELATIO GET-CHILD-RELATION
          GET-CLASS GET-CLIENT GET-CODEPAGE GET-CODEPAGES GET-COLL GET-COLLA GET-COLLAT GET-COLLATI
          GET-COLLATIO GET-COLLATION GET-COLLATIONS GET-COLUMN GET-CONFIG-VALUE GET-CURR GET-CURRE GET-CURREN
          GET-CURRENT GET-DATASET-BUFFER GET-DB-CLIENT GET-DIR GET-DOCUMENT-ELEMENT GET-DOUBLE
          GET-DROPPED-FILE GET-DYNAMIC GET-EFFECTIVE-TENANT-ID GET-EFFECTIVE-TENANT-NAME GET-ERROR-COLUMN
          GET-ERROR-ROW GET-FILE GET-FILE-NAME GET-FILE-OFFSE GET-FILE-OFFSET GET-FIRS GET-FIRST GET-FLOAT
          GET-GREEN GET-GREEN- GET-GREEN-V GET-GREEN-VA GET-GREEN-VAL GET-GREEN-VALU GET-GREEN-VALUE
          GET-HEADER-ENTR GET-HEADER-ENTRY GET-INDEX-BY-NAMESPACE-NAME GET-INDEX-BY-QNAME GET-INT64
          GET-ITERATION GET-KEY-VAL GET-KEY-VALU GET-KEY-VALUE GET-LAST GET-LOCALNAME-BY-INDEX GET-LONG
          GET-MESSAGE GET-MESSAGE-TYPE GET-NEXT GET-NODE GET-NUMBER GET-PARENT GET-POINTER-VALUE GET-PREV
          GET-PRINTERS GET-PROPERTY GET-QNAME-BY-INDEX GET-RED GET-RED- GET-RED-V GET-RED-VA GET-RED-VAL
          GET-RED-VALU GET-RED-VALUE GET-REL GET-RELA GET-RELAT GET-RELATI GET-RELATIO GET-RELATION
          GET-REPOSITIONED-ROW GET-RGB GET-RGB- GET-RGB-V GET-RGB-VA GET-RGB-VAL GET-RGB-VALU GET-RGB-VALUE
          GET-ROW GET-SAFE-USER GET-SELECTED GET-SELECTED- GET-SELECTED-W GET-SELECTED-WI GET-SELECTED-WID
          GET-SELECTED-WIDG GET-SELECTED-WIDGE GET-SELECTED-WIDGET GET-SERIALIZED GET-SHORT GET-SIGNATURE
          GET-SIZE GET-SOCKET-OPTION GET-SOURCE-BUFFER GET-STRING GET-TAB-ITEM GET-TEXT-HEIGHT
          GET-TEXT-HEIGHT-C GET-TEXT-HEIGHT-CH GET-TEXT-HEIGHT-CHA GET-TEXT-HEIGHT-CHAR GET-TEXT-HEIGHT-CHARS
          GET-TEXT-HEIGHT-P GET-TEXT-HEIGHT-PI GET-TEXT-HEIGHT-PIX GET-TEXT-HEIGHT-PIXE GET-TEXT-HEIGHT-PIXEL
          GET-TEXT-HEIGHT-PIXELS GET-TEXT-WIDTH GET-TEXT-WIDTH-C GET-TEXT-WIDTH-CH GET-TEXT-WIDTH-CHA
          GET-TEXT-WIDTH-CHAR GET-TEXT-WIDTH-CHARS GET-TEXT-WIDTH-P GET-TEXT-WIDTH-PI GET-TEXT-WIDTH-PIX
          GET-TEXT-WIDTH-PIXE GET-TEXT-WIDTH-PIXEL GET-TEXT-WIDTH-PIXELS GET-TOP-BUFFER GET-TYPE-BY-INDEX
          GET-TYPE-BY-NAMESPACE-NAME GET-TYPE-BY-QNAME GET-UNSIGNED-LONG GET-UNSIGNED-SHORT GET-URI-BY-INDEX
          GET-VALUE-BY-INDEX GET-VALUE-BY-NAMESPACE-NAME GET-VALUE-BY-QNAME GET-WAIT GET-WAIT- GET-WAIT-S
          GET-WAIT-ST GET-WAIT-STA GET-WAIT-STAT GET-WAIT-STATE GLOBAL GO GO-ON GO-PEND GO-PENDI GO-PENDIN
          GO-PENDING GOTO GRANT GRANT-ARCHIVE GRAPHIC-E GRAPHIC-ED GRAPHIC-EDG GRAPHIC-EDGE GRAYED
          GRID-FACTOR-H GRID-FACTOR-HO GRID-FACTOR-HOR GRID-FACTOR-HORI GRID-FACTOR-HORIZ GRID-FACTOR-HORIZO
          GRID-FACTOR-HORIZON GRID-FACTOR-HORIZONT GRID-FACTOR-HORIZONTA GRID-FACTOR-HORIZONTAL GRID-FACTOR-V
          GRID-FACTOR-VE GRID-FACTOR-VER GRID-FACTOR-VERT GRID-FACTOR-VERTI GRID-FACTOR-VERTIC
          GRID-FACTOR-VERTICA GRID-FACTOR-VERTICAL GRID-SET GRID-SNAP GRID-UNIT-HEIGHT GRID-UNIT-HEIGHT-C
          GRID-UNIT-HEIGHT-CH GRID-UNIT-HEIGHT-CHA GRID-UNIT-HEIGHT-CHAR GRID-UNIT-HEIGHT-CHARS
          GRID-UNIT-HEIGHT-P GRID-UNIT-HEIGHT-PI GRID-UNIT-HEIGHT-PIX GRID-UNIT-HEIGHT-PIXE
          GRID-UNIT-HEIGHT-PIXEL GRID-UNIT-HEIGHT-PIXELS GRID-UNIT-WIDTH GRID-UNIT-WIDTH-C GRID-UNIT-WIDTH-CH
          GRID-UNIT-WIDTH-CHA GRID-UNIT-WIDTH-CHAR GRID-UNIT-WIDTH-CHARS GRID-UNIT-WIDTH-P GRID-UNIT-WIDTH-PI
          GRID-UNIT-WIDTH-PIX GRID-UNIT-WIDTH-PIXE GRID-UNIT-WIDTH-PIXEL GRID-UNIT-WIDTH-PIXELS GRID-VISIBLE
          GROUP GROUP-BOX GT GUID HANDLE HANDLER HAS-LOBS HAS-RECORDS HAVING HEADER HEIGHT HEIGHT-C HEIGHT-CH
          HEIGHT-CHA HEIGHT-CHAR HEIGHT-CHARS HEIGHT-P HEIGHT-PI HEIGHT-PIX HEIGHT-PIXE HEIGHT-PIXEL
          HEIGHT-PIXELS HELP HELP-CON HELP-CONT HELP-CONTE HELP-CONTEX HELP-CONTEXT HELPFILE-N HELPFILE-NA
          HELPFILE-NAM HELPFILE-NAME HELP-TOPIC HEX-DECODE HEX-ENCODE HIDDEN HIDE HINT HOME HORI HORIZ
          HORIZ-END HORIZ-HOME HORIZO HORIZON HORIZONT HORIZONTA HORIZONTAL HORIZ-SCROLL-DRAG HOST-BYTE-ORDER
          HTML-CHARSET HTML-END-OF-LINE HTML-END-OF-PAGE HTML-FRAME-BEGIN HTML-FRAME-END HTML-HEADER-BEGIN
          HTML-HEADER-END HTML-TITLE-BEGIN HTML-TITLE-END HWND ICFPARAM ICFPARAME ICFPARAMET ICFPARAMETE
          ICFPARAMETER ICON IF IGNORE-CURRENT-MOD IGNORE-CURRENT-MODI IGNORE-CURRENT-MODIF
          IGNORE-CURRENT-MODIFI IGNORE-CURRENT-MODIFIE IGNORE-CURRENT-MODIFIED IMAGE IMAGE-DOWN
          IMAGE-INSENSITIVE IMAGE-SIZE IMAGE-SIZE-C IMAGE-SIZE-CH IMAGE-SIZE-CHA IMAGE-SIZE-CHAR
          IMAGE-SIZE-CHARS IMAGE-SIZE-P IMAGE-SIZE-PI IMAGE-SIZE-PIX IMAGE-SIZE-PIXE IMAGE-SIZE-PIXEL
          IMAGE-SIZE-PIXELS IMAGE-UP IMMEDIATE-DISPLAY IMPLEMENTS IMPORT IMPORT-NODE IMPORT-PRINCIPAL IN
          INCREMENT-EXCLUSIVE-ID INDEX INDEXED-REPOSITION INDEX-HINT INDEX-INFO INDEX-INFOR INDEX-INFORM
          INDEX-INFORMA INDEX-INFORMAT INDEX-INFORMATI INDEX-INFORMATIO INDEX-INFORMATION INDICATOR INFO INFOR
          INFORM INFORMA INFORMAT INFORMATI INFORMATIO INFORMATION IN-HANDLE INHERIT-BGC INHERIT-BGCO
          INHERIT-BGCOL INHERIT-BGCOLO INHERIT-BGCOLOR INHERIT-COLOR-MODE INHERIT-FGC INHERIT-FGCO
          INHERIT-FGCOL INHERIT-FGCOLO INHERIT-FGCOLOR INHERITS INIT INITIAL INITIAL-DIR INITIAL-FILTER
          INITIALIZE INITIALIZE-DOCUMENT-TYPE INITIATE INNER INNER-CHARS INNER-LINES INPUT INPUT-O INPUT-OU
          INPUT-OUT INPUT-OUTP INPUT-OUTPU INPUT-OUTPUT INPUT-VALUE INSERT INSERT-ATTRIBUTE INSERT-B INSERT-BA
          INSERT-BAC INSERT-BACK INSERT-BACKT INSERT-BACKTA INSERT-BACKTAB INSERT-BEFORE INSERT-COLUMN
          INSERT-FIELD INSERT-FIELD-DATA INSERT-FIELD-LABEL INSERT-FILE INSERT-MODE INSERT-ROW INSERT-STRING
          INSERT-T INSERT-TA INSERT-TAB INSTANTIATING-PROCEDURE INT INT64 INTE INTEG INTEGE INTEGER INTERFACE
          INTERNAL-ENTRIES INTERVAL INTO INVOKE IS IS-ATTR IS-ATTR- IS-ATTR-S IS-ATTR-SP IS-ATTR-SPA
          IS-ATTR-SPAC IS-ATTR-SPACE IS-CLAS IS-CLASS IS-CODEPAGE-FIXED IS-COLUMN-CODEPAGE IS-DB-MULTI-TENANT
          IS-JSON IS-LEAD-BYTE IS-MULTI-TENANT ISO-DATE IS-OPEN IS-PARAMETER-SET IS-PARTITIONE IS-PARTITIONED
          IS-ROW-SELECTED IS-SELECTED IS-XML ITEM ITEMS-PER-ROW ITERATION-CHANGED JOIN JOIN-BY-SQLDB
          JOIN-ON-SELECT KBLABEL KEEP-CONNECTION-OPEN KEEP-FRAME-Z KEEP-FRAME-Z- KEEP-FRAME-Z-O
          KEEP-FRAME-Z-OR KEEP-FRAME-Z-ORD KEEP-FRAME-Z-ORDE KEEP-FRAME-Z-ORDER KEEP-MESSAGES
          KEEP-SECURITY-CACHE KEEP-TAB-ORDER KEY KEYCACHE-JOIN KEYCODE KEY-CODE KEYFUNC KEY-FUNC KEYFUNCT
          KEY-FUNCT KEYFUNCTI KEY-FUNCTI KEYFUNCTIO KEY-FUNCTIO KEYFUNCTION KEY-FUNCTION KEYLABEL KEY-LABEL
          KEYS KEYWORD KEYWORD-ALL LABEL LABEL-BGC LABEL-BGCO LABEL-BGCOL LABEL-BGCOLO LABEL-BGCOLOR LABEL-DC
          LABEL-DCO LABEL-DCOL LABEL-DCOLO LABEL-DCOLOR LABEL-FGC LABEL-FGCO LABEL-FGCOL LABEL-FGCOLO
          LABEL-FGCOLOR LABEL-FONT LABEL-PFC LABEL-PFCO LABEL-PFCOL LABEL-PFCOLO LABEL-PFCOLOR LABELS
          LABELS-HAVE-COLONS LANDSCAPE LANGUAGE LANGUAGES LARGE LARGE-TO-SMALL LAST LAST-ASYNC LAST-ASYNC-
          LAST-ASYNCH-REQUEST LAST-ASYNC-R LAST-ASYNC-RE LAST-ASYNC-REQ LAST-ASYNC-REQU LAST-ASYNC-REQUE
          LAST-ASYNC-REQUES LAST-ASYNC-REQUEST LAST-BATCH LAST-CHILD LAST-EVEN LAST-EVENT LAST-FORM LASTKEY
          LAST-KEY LAST-OBJECT LAST-OF LAST-PROCE LAST-PROCED LAST-PROCEDU LAST-PROCEDUR LAST-PROCEDURE
          LAST-SERV LAST-SERVE LAST-SERVER LAST-SERVER-SOCKET LAST-SOCKET LAST-TAB-I LAST-TAB-IT LAST-TAB-ITE
          LAST-TAB-ITEM LC LDBNAME LE LEADING LEAK-DETECTION LEAVE LEFT LEFT-ALIGN LEFT-ALIGNE LEFT-ALIGNED
          LEFT-END LEFT-TRIM LENGTH LIBRARY LIBRARY-CALLING-CONVENTION LIKE LIKE-SEQUENTIAL LINE LINE-COUNT
          LINE-COUNTE LINE-COUNTER LINE-DOWN LINE-LEFT LINE-RIGHT LINE-UP LIST-EVENTS LISTI LISTIN LISTING
          LISTINGS LIST-ITEM-PAIRS LIST-ITEMS LIST-PROPERTY-NAMES LIST-QUERY-ATTRS LIST-SET-ATTRS LIST-WIDGETS
          LITERAL-QUESTION LITTLE-ENDIAN LOAD LOAD-DOMAINS LOAD-FROM LOAD-ICON LOAD-IMAGE LOAD-IMAGE-DOWN
          LOAD-IMAGE-INSENSITIVE LOAD-IMAGE-UP LOAD-MOUSE-P LOAD-MOUSE-PO LOAD-MOUSE-POI LOAD-MOUSE-POIN
          LOAD-MOUSE-POINT LOAD-MOUSE-POINTE LOAD-MOUSE-POINTER LOAD-PICTURE LOAD-RESULT-INTO LOAD-SMALL-ICON
          LOB-DIR LOCAL-HOST LOCAL-NAME LOCAL-PORT LOCAL-VERSION-INFO LOCATOR-COLUMN-NUMBER
          LOCATOR-LINE-NUMBER LOCATOR-PUBLIC-ID LOCATOR-SYSTEM-ID LOCATOR-TYPE LOCKED LOCK-REGISTRATION LOG
          LOG-AUDIT-EVENT LOG-ENTRY-TYPES LOGFILE-NAME LOGGING-LEVEL LOGICAL LOG-ID LOGIN-EXPIRATION-TIMESTAMP
          LOGIN-HOST LOGIN-STATE LOG-MANAGER LOGOUT LOG-THRESHOLD LONG LONGCH LONGCHA LONGCHAR
          LONGCHAR-TO-NODE-VALUE LOOKAHEAD LOOKUP LOWER LT MACHINE-CLASS MAIN-MENU MANDATORY MANUAL-HIGHLIGHT
          MAP MARGIN-EXTRA MARGIN-HEIGHT MARGIN-HEIGHT-C MARGIN-HEIGHT-CH MARGIN-HEIGHT-CHA MARGIN-HEIGHT-CHAR
          MARGIN-HEIGHT-CHARS MARGIN-HEIGHT-P MARGIN-HEIGHT-PI MARGIN-HEIGHT-PIX MARGIN-HEIGHT-PIXE
          MARGIN-HEIGHT-PIXEL MARGIN-HEIGHT-PIXELS MARGIN-WIDTH MARGIN-WIDTH-C MARGIN-WIDTH-CH
          MARGIN-WIDTH-CHA MARGIN-WIDTH-CHAR MARGIN-WIDTH-CHARS MARGIN-WIDTH-P MARGIN-WIDTH-PI
          MARGIN-WIDTH-PIX MARGIN-WIDTH-PIXE MARGIN-WIDTH-PIXEL MARGIN-WIDTH-PIXELS MARK-NEW MARK-ROW-STATE
          MATCHES MAX MAX-BUTTON MAX-CHARS MAX-DATA-GUESS MAX-HEIGHT MAX-HEIGHT-C MAX-HEIGHT-CH MAX-HEIGHT-CHA
          MAX-HEIGHT-CHAR MAX-HEIGHT-CHARS MAX-HEIGHT-P MAX-HEIGHT-PI MAX-HEIGHT-PIX MAX-HEIGHT-PIXE
          MAX-HEIGHT-PIXEL MAX-HEIGHT-PIXELS MAXIMIZE MAXIMUM MAXIMUM-LEVEL MAX-ROWS MAX-SIZE MAX-VAL MAX-VALU
          MAX-VALUE MAX-WIDTH MAX-WIDTH-C MAX-WIDTH-CH MAX-WIDTH-CHA MAX-WIDTH-CHAR MAX-WIDTH-CHARS
          MAX-WIDTH-P MAX-WIDTH-PI MAX-WIDTH-PIX MAX-WIDTH-PIXE MAX-WIDTH-PIXEL MAX-WIDTH-PIXELS MD5-DIGEST
          MD5-VALUE MEMBER MEMPTR MEMPTR-TO-NODE-VALUE MENU MENUBAR MENU-BAR MENU-DROP MENU-ITEM MENU-K
          MENU-KE MENU-KEY MENU-M MENU-MO MENU-MOU MENU-MOUS MENU-MOUSE MERGE-BY-FIELD MERGE-CHANGES
          MERGE-ROW-CHANGES MESSAGE MESSAGE-AREA MESSAGE-AREA-FONT MESSAGE-AREA-MSG MESSAGE-DIGEST
          MESSAGE-LINE MESSAGE-LINES METHOD MIN MIN-BUTTON MIN-COLUMN-WIDTH-C MIN-COLUMN-WIDTH-CH
          MIN-COLUMN-WIDTH-CHA MIN-COLUMN-WIDTH-CHAR MIN-COLUMN-WIDTH-CHARS MIN-COLUMN-WIDTH-P
          MIN-COLUMN-WIDTH-PI MIN-COLUMN-WIDTH-PIX MIN-COLUMN-WIDTH-PIXE MIN-COLUMN-WIDTH-PIXEL
          MIN-COLUMN-WIDTH-PIXELS MIN-HEIGHT MIN-HEIGHT-C MIN-HEIGHT-CH MIN-HEIGHT-CHA MIN-HEIGHT-CHAR
          MIN-HEIGHT-CHARS MIN-HEIGHT-P MIN-HEIGHT-PI MIN-HEIGHT-PIX MIN-HEIGHT-PIXE MIN-HEIGHT-PIXEL
          MIN-HEIGHT-PIXELS MINI MINIM MINIMU MINIMUM MIN-SCHEMA-MARSHAL MIN-SCHEMA-MARSHALL MIN-SIZE MIN-VAL
          MIN-VALU MIN-VALUE MIN-WIDTH MIN-WIDTH-C MIN-WIDTH-CH MIN-WIDTH-CHA MIN-WIDTH-CHAR MIN-WIDTH-CHARS
          MIN-WIDTH-P MIN-WIDTH-PI MIN-WIDTH-PIX MIN-WIDTH-PIXE MIN-WIDTH-PIXEL MIN-WIDTH-PIXELS MOD MODIFIED
          MODULO MONTH MOUSE MOUSE-P MOUSE-PO MOUSE-POI MOUSE-POIN MOUSE-POINT MOUSE-POINTE MOUSE-POINTER
          MOVABLE MOVE MOVE-AFTER MOVE-AFTER- MOVE-AFTER-T MOVE-AFTER-TA MOVE-AFTER-TAB MOVE-AFTER-TAB-
          MOVE-AFTER-TAB-I MOVE-AFTER-TAB-IT MOVE-AFTER-TAB-ITE MOVE-AFTER-TAB-ITEM MOVE-BEFOR MOVE-BEFORE
          MOVE-BEFORE- MOVE-BEFORE-T MOVE-BEFORE-TA MOVE-BEFORE-TAB MOVE-BEFORE-TAB- MOVE-BEFORE-TAB-I
          MOVE-BEFORE-TAB-IT MOVE-BEFORE-TAB-ITE MOVE-BEFORE-TAB-ITEM MOVE-COL MOVE-COLU MOVE-COLUM
          MOVE-COLUMN MOVE-TO-B MOVE-TO-BO MOVE-TO-BOT MOVE-TO-BOTT MOVE-TO-BOTTO MOVE-TO-BOTTOM MOVE-TO-EOF
          MOVE-TO-T MOVE-TO-TO MOVE-TO-TOP MPE MTIME MULTI-COMPILE MULTIPLE MULTIPLE-KEY MULTITASKING-INTERVAL
          MUST-EXIST MUST-UNDERSTAND NAME NAMESPACE-PREFIX NAMESPACE-URI NATIVE NE NEEDS-APPSERVER-PROMPT
          NEEDS-PROMPT NESTED NEW NEW-INSTANCE NEW-LINE NEW-ROW NEXT NEXT-COL NEXT-COLU NEXT-COLUM NEXT-COLUMN
          NEXT-ERROR NEXT-FRAME NEXT-PROMPT NEXT-ROWID NEXT-SIBLING NEXT-TAB-I NEXT-TAB-ITE NEXT-TAB-ITEM
          NEXT-VALUE NEXT-WORD NO NO-APPLY NO-ARRAY-M NO-ARRAY-ME NO-ARRAY-MES NO-ARRAY-MESS NO-ARRAY-MESSA
          NO-ARRAY-MESSAG NO-ARRAY-MESSAGE NO-ASSIGN NO-ATTR NO-ATTR-L NO-ATTR-LI NO-ATTR-LIS NO-ATTR-LIST
          NO-ATTR-S NO-ATTR-SP NO-ATTR-SPA NO-ATTR-SPAC NO-ATTR-SPACE NO-AUTO-TRI NO-AUTO-TRIM
          NO-AUTO-VALIDATE NO-BIND-WHERE NO-BOX NO-COLUMN-SC NO-COLUMN-SCR NO-COLUMN-SCRO NO-COLUMN-SCROL
          NO-COLUMN-SCROLL NO-COLUMN-SCROLLI NO-COLUMN-SCROLLIN NO-COLUMN-SCROLLING NO-CONSOLE NO-CONVERT
          NO-CONVERT-3D NO-CONVERT-3D- NO-CONVERT-3D-C NO-CONVERT-3D-CO NO-CONVERT-3D-COL NO-CONVERT-3D-COLO
          NO-CONVERT-3D-COLOR NO-CONVERT-3D-COLORS NO-CURRENT-VALUE NO-DEBUG NODE-TYPE NODE-VALUE
          NODE-VALUE-TO-LONGCHAR NODE-VALUE-TO-MEMPTR NO-DRAG NO-ECHO NO-EMPTY-SPACE NO-ERROR NO-F NO-FI
          NO-FIL NO-FILL NO-FIREHOSE-CURSOR NO-FOCUS NO-HELP NO-HIDE NO-INDEX-HINT NO-INHERIT-BGC
          NO-INHERIT-BGCO NO-INHERIT-BGCOL NO-INHERIT-BGCOLO NO-INHERIT-BGCOLOR NO-INHERIT-FGC NO-INHERIT-FGCO
          NO-INHERIT-FGCOL NO-INHERIT-FGCOLO NO-INHERIT-FGCOLOR NO-JOIN-BY-SQLDB NO-KEYCACHE-JOIN NO-LABEL
          NO-LABELS NO-LOBS NO-LOCK NO-LOOKAHEAD NO-MAP NO-MES NO-MESS NO-MESSA NO-MESSAG NO-MESSAGE
          NONAMESPACE-SCHEMA-LOCATION NONE NON-SERIALIZABLE NO-PAUSE NO-PREFE NO-PREFET NO-PREFETC NO-PREFETCH
          NO-QUERY-O NO-QUERY-OR NO-QUERY-ORD NO-QUERY-ORDE NO-QUERY-ORDER NO-QUERY-ORDER- NO-QUERY-ORDER-A
          NO-QUERY-ORDER-AD NO-QUERY-ORDER-ADD NO-QUERY-ORDER-ADDE NO-QUERY-ORDER-ADDED NO-QUERY-U NO-QUERY-UN
          NO-QUERY-UNI NO-QUERY-UNIQ NO-QUERY-UNIQU NO-QUERY-UNIQUE NO-QUERY-UNIQUE- NO-QUERY-UNIQUE-A
          NO-QUERY-UNIQUE-AD NO-QUERY-UNIQUE-ADD NO-QUERY-UNIQUE-ADDE NO-QUERY-UNIQUE-ADDED NO-RETURN-VAL
          NO-RETURN-VALU NO-RETURN-VALUE NORMALIZE NO-ROW-MARKERS NO-SCHEMA-MARSHAL NO-SCHEMA-MARSHALL
          NO-SCROLLBAR-V NO-SCROLLBAR-VE NO-SCROLLBAR-VER NO-SCROLLBAR-VERT NO-SCROLLBAR-VERTI
          NO-SCROLLBAR-VERTIC NO-SCROLLBAR-VERTICA NO-SCROLLBAR-VERTICAL NO-SCROLLING NO-SEPARATE-CONNECTION
          NO-SEPARATORS NOT NO-TAB NO-TAB- NO-TAB-S NO-TAB-ST NO-TAB-STO NO-TAB-STOP NOT-ACTIVE NO-UND NO-UNDE
          NO-UNDER NO-UNDERL NO-UNDERLI NO-UNDERLIN NO-UNDERLINE NO-UNDO NO-VAL NO-VALI NO-VALID NO-VALIDA
          NO-VALIDAT NO-VALIDATE NOW NO-WAIT NO-WORD-WRAP NULL NUM-ALI NUM-ALIA NUM-ALIAS NUM-ALIASE
          NUM-ALIASES NUM-BUFFERS NUM-BUT NUM-BUTT NUM-BUTTO NUM-BUTTON NUM-BUTTONS NUM-CHILD-RELATIONS
          NUM-CHILDREN NUM-COL NUM-COLU NUM-COLUM NUM-COLUMN NUM-COLUMNS NUM-COPIES NUM-DBS NUM-DROPPED-FILES
          NUM-ENTRIES NUMERIC NUMERIC-DEC NUMERIC-DECI NUMERIC-DECIM NUMERIC-DECIMA NUMERIC-DECIMAL
          NUMERIC-DECIMAL- NUMERIC-DECIMAL-P NUMERIC-DECIMAL-PO NUMERIC-DECIMAL-POI NUMERIC-DECIMAL-POIN
          NUMERIC-DECIMAL-POINT NUMERIC-F NUMERIC-FO NUMERIC-FOR NUMERIC-FORM NUMERIC-FORMA NUMERIC-FORMAT
          NUMERIC-SEP NUMERIC-SEPA NUMERIC-SEPAR NUMERIC-SEPARA NUMERIC-SEPARAT NUMERIC-SEPARATO
          NUMERIC-SEPARATOR NUM-FIELDS NUM-FORMATS NUM-HEADER-ENTRIES NUM-ITEMS NUM-ITERATIONS NUM-LINES
          NUM-LOCKED-COL NUM-LOCKED-COLU NUM-LOCKED-COLUM NUM-LOCKED-COLUMN NUM-LOCKED-COLUMNS NUM-LOG-FILES
          NUM-MESSAGES NUM-PARAMETERS NUM-REFERENCES NUM-RELATIONS NUM-REPL NUM-REPLA NUM-REPLAC NUM-REPLACE
          NUM-REPLACED NUM-RESULTS NUM-SELECTED NUM-SELECTED-ROWS NUM-SELECTED-WIDGETS NUM-SOURCE-BUFFERS
          NUM-TABS NUM-TOP-BUFFERS NUM-TO-RETAIN NUM-VISIBLE-COL NUM-VISIBLE-COLU NUM-VISIBLE-COLUM
          NUM-VISIBLE-COLUMN NUM-VISIBLE-COLUMNS OBJECT OCTET_LENGTH OCTET-LENGTH OF OFF OFF-END OFF-HOME OK
          OK-CANCEL OLD OLE-INVOKE-LOCA OLE-INVOKE-LOCAL OLE-INVOKE-LOCALE OLE-NAMES-LOCA OLE-NAMES-LOCAL
          OLE-NAMES-LOCALE ON ON-FRAME ON-FRAME- ON-FRAME-B ON-FRAME-BO ON-FRAME-BOR ON-FRAME-BORD
          ON-FRAME-BORDE ON-FRAME-BORDER OPEN OPEN-LINE-ABOVE OPSYS OPTION OPTIONS OPTIONS-FILE OR
          ORDERED-JOIN ORDINAL ORIENTATION ORIGIN-HANDLE ORIGIN-ROWID OS2 OS400 OS-APPEND OS-COMMAND OS-COPY
          OS-CREATE-DIR OS-DELETE OS-DIR OS-DRIVE OS-DRIVES OS-ERROR OS-GETENV OS-RENAME OTHERWISE OUTER
          OUTER-JOIN OUT-OF-DATA OUTPUT OVERLAY OVERRIDE OWNER OWNER-DOCUMENT PACKAGE-PRIVATE
          PACKAGE-PROTECTED PAGE PAGE-BOT PAGE-BOTT PAGE-BOTTO PAGE-BOTTOM PAGED PAGE-DOWN PAGE-LEFT PAGE-NUM
          PAGE-NUMB PAGE-NUMBE PAGE-NUMBER PAGE-RIGHT PAGE-RIGHT-TEXT PAGE-SIZE PAGE-TOP PAGE-UP PAGE-WID
          PAGE-WIDT PAGE-WIDTH PARAM PARAME PARAMET PARAMETE PARAMETER PARENT PARENT-BUFFER
          PARENT-FIELDS-AFTER PARENT-FIELDS-BEFORE PARENT-ID-FIELD PARENT-ID-RELATION PARENT-REL PARENT-RELA
          PARENT-RELAT PARENT-RELATI PARENT-RELATIO PARENT-RELATION PARENT-WINDOW-CLOSE PARSE-STATUS
          PARTIAL-KEY PASCAL PASSWORD-FIELD PASTE PATHNAME PAUSE PBE-HASH-ALG PBE-HASH-ALGO PBE-HASH-ALGOR
          PBE-HASH-ALGORI PBE-HASH-ALGORIT PBE-HASH-ALGORITH PBE-HASH-ALGORITHM PBE-KEY-ROUNDS PDBNAME PERF
          PERFO PERFOR PERFORM PERFORMA PERFORMAN PERFORMANC PERFORMANCE PERSIST PERSISTE PERSISTEN PERSISTENT
          PERSISTENT-CACHE-DISABLED PERSISTENT-PROCEDURE PFC PFCO PFCOL PFCOLO PFCOLOR PICK PICK-AREA
          PICK-BOTH PIXELS PIXELS-PER-COL PIXELS-PER-COLU PIXELS-PER-COLUM PIXELS-PER-COLUMN PIXELS-PER-ROW
          POPUP-M POPUP-ME POPUP-MEN POPUP-MENU POPUP-O POPUP-ON POPUP-ONL POPUP-ONLY PORTRAIT POSITION
          PRECISION PREFER-DATASET PREPARED PREPARE-STRING PREPROC PREPROCE PREPROCES PREPROCESS PRESEL
          PRESELE PRESELEC PRESELECT PREV PREV-COL PREV-COLU PREV-COLUM PREV-COLUMN PREV-FRAME PREV-SIBLING
          PREV-TAB-I PREV-TAB-IT PREV-TAB-ITE PREV-TAB-ITEM PREV-WORD PRIMARY PRIMARY-PASSPHRASE PRINTER
          PRINTER-CONTROL-HANDLE PRINTER-HDC PRINTER-NAME PRINTER-PORT PRINTER-SETUP PRIVATE PRIVATE-D
          PRIVATE-DA PRIVATE-DAT PRIVATE-DATA PRIVILEGES PROCE PROCED PROCEDU PROCEDUR PROCEDURE
          PROCEDURE-CALL-TYPE PROCEDURE-COMPLETE PROCEDURE-NAME PROCEDURE-TYPE PROCESS PROCESS-ARCHITECTURE
          PROC-HA PROC-HAN PROC-HAND PROC-HANDL PROC-HANDLE PROC-ST PROC-STA PROC-STAT PROC-STATU PROC-STATUS
          PROC-TEXT PROC-TEXT-BUFFER PROFILE-FILE PROFILER PROFILING PROGRAM-NAME PROGRESS PROGRESS-S
          PROGRESS-SO PROGRESS-SOU PROGRESS-SOUR PROGRESS-SOURC PROGRESS-SOURCE PROMPT PROMPT-F PROMPT-FO
          PROMPT-FOR PROMSGS PROPATH PROPERTY PROTECTED PROVERS PROVERSI PROVERSIO PROVERSION PROXY
          PROXY-PASSWORD PROXY-USERID PUBLIC PUBLIC-ID PUBLISH PUBLISHED-EVENTS PUT PUT-BITS PUTBYTE PUT-BYTE
          PUT-BYTES PUT-DOUBLE PUT-FLOAT PUT-INT64 PUT-KEY-VAL PUT-KEY-VALU PUT-KEY-VALUE PUT-LONG PUT-SHORT
          PUT-STRING PUT-UNSIGNED-LONG PUT-UNSIGNED-SHORT QUALIFIED-USER-ID QUERY QUERY-CLOSE QUERY-OFF-END
          QUERY-OPEN QUERY-PREPARE QUERY-TUNING QUESTION QUIT QUOTER RADIO-BUTTONS RADIO-SET RANDOM RAW
          RAW-TRANSFER RCODE-INFO RCODE-INFOR RCODE-INFORM RCODE-INFORMA RCODE-INFORMAT RCODE-INFORMATI
          RCODE-INFORMATIO RCODE-INFORMATION READ READ-AVAILABLE READ-EXACT-NUM READ-FILE READ-JSON READKEY
          READ-ONLY READ-RESPONSE READ-XML READ-XMLSCHEMA REAL RECALL RECID RECORD-LEN RECORD-LENG
          RECORD-LENGT RECORD-LENGTH RECT RECTA RECTAN RECTANG RECTANGL RECTANGLE RECURSIVE REFERENCE-ONLY
          REFRESH REFRESHABLE REFRESH-AUDIT-POLICY REGISTER-DOMAIN REINSTATE REJECT-CHANGES REJECTED
          REJECT-ROW-CHANGES RELATION-FI RELATION-FIE RELATION-FIEL RELATION-FIELD RELATION-FIELDS
          RELATIONS-ACTIVE RELEASE REMOTE REMOTE-HOST REMOTE-PORT REMOVE-ATTRIBUTE REMOVE-CHILD
          REMOVE-EVENTS-PROC REMOVE-EVENTS-PROCE REMOVE-EVENTS-PROCED REMOVE-EVENTS-PROCEDU
          REMOVE-EVENTS-PROCEDUR REMOVE-EVENTS-PROCEDURE REMOVE-SUPER-PROC REMOVE-SUPER-PROCE
          REMOVE-SUPER-PROCED REMOVE-SUPER-PROCEDU REMOVE-SUPER-PROCEDUR REMOVE-SUPER-PROCEDURE REPEAT REPLACE
          REPLACE-CHILD REPLACE-SELECTION-TEXT REPLICATION-CREATE REPLICATION-DELETE REPLICATION-WRITE REPORTS
          REPOSITION REPOSITION-BACK REPOSITION-BACKW REPOSITION-BACKWA REPOSITION-BACKWAR REPOSITION-BACKWARD
          REPOSITION-BACKWARDS REPOSITION-FORW REPOSITION-FORWA REPOSITION-FORWAR REPOSITION-FORWARD
          REPOSITION-FORWARDS REPOSITION-MODE REPOSITION-PARENT-REL REPOSITION-PARENT-RELA
          REPOSITION-PARENT-RELAT REPOSITION-PARENT-RELATI REPOSITION-PARENT-RELATIO REPOSITION-PARENT-RELATION
          REPOSITION-TO-ROW REPOSITION-TO-ROWID REQUEST REQUEST-INFO RESET RESIZA RESIZAB RESIZABL RESIZABLE
          RESIZE RESPONSE-INFO RESTART-ROW RESTART-ROWID RESULT RESUME-DISPLAY RETAIN RETAIN-S RETAIN-SH
          RETAIN-SHA RETAIN-SHAP RETAIN-SHAPE RETRY RETRY-CANCEL RETURN RETURN-ALIGN RETURN-INS RETURN-INSE
          RETURN-INSER RETURN-INSERT RETURN-INSERTE RETURN-INSERTED RETURNS RETURN-TO-START-DI
          RETURN-TO-START-DIR RETURN-VAL RETURN-VALU RETURN-VALUE RETURN-VALUE-DATA-TYPE RETURN-VALUE-DLL-TYPE
          REVERSE-FROM REVERT REVOKE RGB-V RGB-VA RGB-VAL RGB-VALU RGB-VALUE RIGHT RIGHT-ALIGN RIGHT-ALIGNE
          RIGHT-ALIGNED RIGHT-END RIGHT-TRIM R-INDEX ROLE ROLES ROUND ROUNDED ROUTINE-LEVEL ROW ROW-CREATED
          ROW-DELETED ROW-DISPLAY ROW-ENTRY ROW-HEIGHT ROW-HEIGHT-C ROW-HEIGHT-CH ROW-HEIGHT-CHA
          ROW-HEIGHT-CHAR ROW-HEIGHT-CHARS ROW-HEIGHT-P ROW-HEIGHT-PI ROW-HEIGHT-PIX ROW-HEIGHT-PIXE
          ROW-HEIGHT-PIXEL ROW-HEIGHT-PIXELS ROWID ROW-LEAVE ROW-MA ROW-MAR ROW-MARK ROW-MARKE ROW-MARKER
          ROW-MARKERS ROW-MODIFIED ROW-OF ROW-RESIZABLE ROW-STATE ROW-UNMODIFIED RULE RULE-ROW RULE-Y RUN
          RUN-PROC RUN-PROCE RUN-PROCED RUN-PROCEDU RUN-PROCEDUR RUN-PROCEDURE SAVE SAVE-AS SAVE-FILE
          SAVE-ROW-CHANGES SAVE-WHERE-STRING SAX-ATTRIBUTES SAX-COMPLE SAX-COMPLET SAX-COMPLETE SAX-PARSE
          SAX-PARSE-FIRST SAX-PARSE-NEXT SAX-PARSER-ERROR SAX-READER SAX-RUNNING SAX-UNINITIALIZED
          SAX-WRITE-BEGIN SAX-WRITE-COMPLETE SAX-WRITE-CONTENT SAX-WRITE-ELEMENT SAX-WRITE-ERROR
          SAX-WRITE-IDLE SAX-WRITER SAX-WRITE-TAG SAX-XML SCHEMA SCHEMA-CHANGE SCHEMA-LOCATION SCHEMA-MARSHAL
          SCHEMA-PATH SCREEN SCREEN-IO SCREEN-LINES SCREEN-VAL SCREEN-VALU SCREEN-VALUE SCROLL SCROLLABLE
          SCROLLBAR-DRAG SCROLLBAR-H SCROLLBAR-HO SCROLLBAR-HOR SCROLLBAR-HORI SCROLLBAR-HORIZ
          SCROLLBAR-HORIZO SCROLLBAR-HORIZON SCROLLBAR-HORIZONT SCROLLBAR-HORIZONTA SCROLLBAR-HORIZONTAL
          SCROLL-BARS SCROLLBAR-V SCROLLBAR-VE SCROLLBAR-VER SCROLLBAR-VERT SCROLLBAR-VERTI SCROLLBAR-VERTIC
          SCROLLBAR-VERTICA SCROLLBAR-VERTICAL SCROLL-DELTA SCROLLED-ROW-POS SCROLLED-ROW-POSI
          SCROLLED-ROW-POSIT SCROLLED-ROW-POSITI SCROLLED-ROW-POSITIO SCROLLED-ROW-POSITION SCROLL-HORIZONTAL
          SCROLLING SCROLL-LEFT SCROLL-MODE SCROLL-NOTIFY SCROLL-OFFSET SCROLL-RIGHT SCROLL-TO-CURRENT-ROW
          SCROLL-TO-I SCROLL-TO-IT SCROLL-TO-ITE SCROLL-TO-ITEM SCROLL-TO-SELECTED-ROW SCROLL-VERTICAL SDBNAME
          SEAL SEAL-TIMESTAMP SEARCH SEARCH-SELF SEARCH-TARGER SEARCH-TARGET SECTION SECURITY-POLICY SEEK
          SELECT SELECTABLE SELECT-ALL SELECTED SELECTED-ITEMS SELECT-EXTEND SELECT-FOCUSED-ROW SELECTION
          SELECTION-END SELECTION-EXTEND SELECTION-LIST SELECTION-START SELECTION-TEXT SELECT-NEXT-ROW
          SELECT-ON-JOIN SELECT-PREV-ROW SELECT-REPOSITIONED-ROW SELECT-ROW SELF SEND SEND-SQL
          SEND-SQL-STATEMENT SENSITIVE SEPARATE-CONNECTION SEPARATOR-FGC SEPARATOR-FGCO SEPARATOR-FGCOL
          SEPARATOR-FGCOLO SEPARATOR-FGCOLOR SEPARATORS SERIALIZABLE SERIALIZE-HIDDEN SERIALIZE-NAME
          SERIALIZE-ROW SERVER SERVER-CONNECTION-BO SERVER-CONNECTION-BOU SERVER-CONNECTION-BOUN
          SERVER-CONNECTION-BOUND SERVER-CONNECTION-BOUND-RE SERVER-CONNECTION-BOUND-REQ SERVER-CONNECTION-BOUND-REQU
          SERVER-CONNECTION-BOUND-REQUE SERVER-CONNECTION-BOUND-REQUES SERVER-CONNECTION-BOUND-REQUEST
          SERVER-CONNECTION-CO SERVER-CONNECTION-CON SERVER-CONNECTION-CONT SERVER-CONNECTION-CONTE
          SERVER-CONNECTION-CONTEX SERVER-CONNECTION-CONTEXT SERVER-CONNECTION-ID SERVER-OPERATING-MODE
          SERVER-SOCKET SESSION SESSION-END SESSION-ID SET SET-ACTOR SET-APPL-CONTEXT SET-ATTR-CALL-TYPE
          SET-ATTRIBUTE SET-ATTRIBUTE-NODE SET-BLUE SET-BLUE- SET-BLUE-V SET-BLUE-VA SET-BLUE-VAL
          SET-BLUE-VALU SET-BLUE-VALUE SET-BREAK SET-BUFFERS SET-BYTE-ORDER SET-CALLBACK SET-CALLBACK-PROCEDURE
          SET-CELL-FOCUS SET-CLIENT SET-COMMIT SET-CONNECT-PROCEDURE SET-CONTENTS SET-CURRENT-VALUE
          SET-DB-CLIENT SET-DB-LOGGING SET-DYNAMIC SET-EFFECTIVE-TENANT SET-EVENT-MANAGER-OPTION SET-GREEN
          SET-GREEN- SET-GREEN-V SET-GREEN-VA SET-GREEN-VAL SET-GREEN-VALU SET-GREEN-VALUE SET-INPUT-SOURCE
          SET-MUST-UNDERSTAND SET-NODE SET-NUMERIC-FORM SET-NUMERIC-FORMA SET-NUMERIC-FORMAT SET-OPTION
          SET-OUTPUT-DESTINATION SET-PARAMETER SET-POINTER-VAL SET-POINTER-VALU SET-POINTER-VALUE SET-PROPERTY
          SET-READ-RESPONSE-PROCEDURE SET-RED SET-RED- SET-RED-V SET-RED-VA SET-RED-VAL SET-RED-VALU
          SET-RED-VALUE SET-REPOSITIONED-ROW SET-RGB SET-RGB- SET-RGB-V SET-RGB-VA SET-RGB-VAL SET-RGB-VALU
          SET-RGB-VALUE SET-ROLE SET-ROLLBACK SET-SAFE-USER SET-SELECTION SET-SERIALIZED SET-SIZE
          SET-SOCKET-OPTION SET-SORT-ARROW SET-STATE SETTINGS SETUSER SETUSERI SETUSERID SET-WAIT SET-WAIT-
          SET-WAIT-S SET-WAIT-ST SET-WAIT-STA SET-WAIT-STAT SET-WAIT-STATE SHA1-DIGEST SHARE SHARE- SHARED
          SHARE-L SHARE-LO SHARE-LOC SHARE-LOCK SHORT SHOW-IN-TASK SHOW-IN-TASKB SHOW-IN-TASKBA
          SHOW-IN-TASKBAR SHOW-STAT SHOW-STATS SIDE-LAB SIDE-LABE SIDE-LABEL SIDE-LABEL-H SIDE-LABEL-HA
          SIDE-LABEL-HAN SIDE-LABEL-HAND SIDE-LABEL-HANDL SIDE-LABEL-HANDLE SIDE-LABELS SIGNATURE
          SIGNATURE-VALUE SILENT SIMPLE SINGLE SINGLE-CHARACTER SINGLE-RUN SINGLETON SIZE SIZE-C SIZE-CH
          SIZE-CHA SIZE-CHAR SIZE-CHARS SIZE-P SIZE-PI SIZE-PIX SIZE-PIXE SIZE-PIXEL SIZE-PIXELS SKIP
          SKIP-DELETED-REC SKIP-DELETED-RECO SKIP-DELETED-RECOR SKIP-DELETED-RECORD SKIP-GROUP-DUPLICATES
          SKIP-SCHEMA-CHECK SLIDER SMALL-ICON SMALLINT SMALL-TITLE SOAP-FAULT SOAP-FAULT-ACTOR SOAP-FAULT-CODE
          SOAP-FAULT-DETAIL SOAP-FAULT-MISUNDERSTOOD-HEADER SOAP-FAULT-NODE SOAP-FAULT-ROLE SOAP-FAULT-STRING
          SOAP-FAULT-SUBCODE SOAP-HEADER SOAP-HEADER-ENTRYREF SOAP-VERSION SOCKET SOME SORT SORT-ASCENDING
          SORT-NUMBER SOURCE SOURCE-PROCEDURE SPACE SQL SQRT SSL-SERVER-NAME STANDALONE START
          START-BOX-SELECTION START-DOCUMENT START-ELEMENT START-EXTEND-BOX-SELECTION STARTING START-MEM-CHECK
          START-MOVE START-RESIZE START-ROW-RESIZE START-SEARCH STARTUP-PARAMETERS STATE-DETAIL STATIC
          STATISTICS STATUS STATUS-AREA STATUS-AREA-FONT STATUS-AREA-MSG STDCALL STOMP-DETECTION
          STOMP-FREQUENCY STOP STOP-AFTER STOP-DISPLAY STOP-MEM-CHECK STOP-OBJECT STOP-PARSING STOPPE STOPPED
          STORED-PROC STORED-PROCE STORED-PROCED STORED-PROCEDU STORED-PROCEDUR STORED-PROCEDURE STREAM
          STREAM-HANDLE STREAM-IO STRETCH-TO-FIT STRICT STRICT-ENTITY-RESOLUTION STRING STRING-VALUE
          STRING-XREF SUB- SUB-AVE SUB-AVER SUB-AVERA SUB-AVERAG SUB-AVERAGE SUB-COUNT SUB-MAX SUB-MAXI
          SUB-MAXIM SUB-MAXIMU SUB-MAXIMUM SUB-MENU SUB-MENU-HELP SUB-MIN SUB-MINI SUB-MINIM SUB-MINIMU
          SUB-MINIMUM SUBSCRIBE SUBST SUBSTI SUBSTIT SUBSTITU SUBSTITUT SUBSTITUTE SUBSTR SUBSTRI SUBSTRIN
          SUBSTRING SUB-TOTAL SUBTYPE SUM SUMMARY SUM-MAX SUPER SUPER-PROC SUPER-PROCE SUPER-PROCED
          SUPER-PROCEDU SUPER-PROCEDUR SUPER-PROCEDURE SUPER-PROCEDURES SUPPRESS-NAMESPACE-PROCESSING
          SUPPRESS-W SUPPRESS-WA SUPPRESS-WAR SUPPRESS-WARN SUPPRESS-WARNI SUPPRESS-WARNIN SUPPRESS-WARNING
          SUPPRESS-WARNINGS SUPPRESS-WARNINGS-LIST SUSPEND SYMMETRIC-ENCRYPTION-ALGORITHM SYMMETRIC-ENCRYPTION-IV
          SYMMETRIC-ENCRYPTION-KEY SYMMETRIC-SUPPORT SYNCHRONIZE SYSTEM-ALERT SYSTEM-ALERT- SYSTEM-ALERT-B
          SYSTEM-ALERT-BO SYSTEM-ALERT-BOX SYSTEM-ALERT-BOXE SYSTEM-ALERT-BOXES SYSTEM-DIALOG SYSTEM-HELP
          SYSTEM-ID TAB TABLE TABLE-CRC-LIST TABLE-HANDLE TABLE-LIST TABLE-NUM TABLE-NUMB TABLE-NUMBE
          TABLE-NUMBER TABLE-SCAN TAB-POSITION TAB-STOP TARGET TARGET-PROCEDURE TEMP-DIR TEMP-DIRE TEMP-DIREC
          TEMP-DIRECT TEMP-DIRECTO TEMP-DIRECTOR TEMP-DIRECTORY TEMP-TABLE TEMP-TABLE-PREPAR
          TEMP-TABLE-PREPARE TENANT TENANT-ID TENANT-NAME TENANT-NAME-TO-ID TENANT-WHERE TERM TERMINAL
          TERMINATE TEXT TEXT-CURSOR TEXT-SEG TEXT-SEG- TEXT-SEG-G TEXT-SEG-GR TEXT-SEG-GRO TEXT-SEG-GROW
          TEXT-SEG-GROWT TEXT-SEG-GROWTH TEXT-SELECTED THEN THIS-OBJECT THIS-PROCEDURE THREAD-SAFE THREE-D
          THROUGH THROW THRU TIC-MARKS TIME TIME-SOURCE TIMEZONE TITLE TITLE-BGC TITLE-BGCO TITLE-BGCOL
          TITLE-BGCOLO TITLE-BGCOLOR TITLE-DC TITLE-DCO TITLE-DCOL TITLE-DCOLO TITLE-DCOLOR TITLE-FGC
          TITLE-FGCO TITLE-FGCOL TITLE-FGCOLO TITLE-FGCOLOR TITLE-FO TITLE-FON TITLE-FONT TO TODAY TOGGLE-BOX
          TOOLTIP TOOLTIPS TOP TOP-COLUMN TOPIC TOP-NAV-QUERY TOP-ONLY TO-ROWID TOTAL TRACE-FILTER TRACING
          TRACKING-CHANGES TRAILING TRANS TRANSACT TRANSACTI TRANSACTIO TRANSACTION TRANSACTION-MODE
          TRANS-INIT-PROC TRANS-INIT-PROCE TRANS-INIT-PROCED TRANS-INIT-PROCEDU TRANS-INIT-PROCEDUR
          TRANS-INIT-PROCEDURE TRANSPAR TRANSPARE TRANSPAREN TRANSPARENT TRIGGER TRIGGERS TRIM TRUE TRUNC
          TRUNCA TRUNCAT TRUNCATE TTCODEPAGE TYPE TYPE-OF UNBOX UNBUFF UNBUFFE UNBUFFER UNBUFFERE UNBUFFERED
          UNDERL UNDERLI UNDERLIN UNDERLINE UNDO UNDO-THROW-SCOPE UNFORM UNFORMA UNFORMAT UNFORMATT UNFORMATTE
          UNFORMATTED UNION UNIQUE UNIQUE-ID UNIQUE-MATCH UNIX UNIX-END UNLESS-HIDDEN UNLOAD UNSIGNED-BYTE
          UNSIGNED-INT64 UNSIGNED-INTEGER UNSIGNED-LONG UNSIGNED-SHORT UNSUBSCRIBE UP UPDATE UPDATE-ATTRIBUTE
          UPPER URL URL-DECODE URL-ENCODE URL-PASSWORD URL-USERID USE USE-DIC USE-DICT USE-DICT- USE-DICT-E
          USE-DICT-EX USE-DICT-EXP USE-DICT-EXPS USE-FILENAME USE-INDEX USER USER-DATA USE-REVVIDEO USERID
          USER-ID USE-TEXT USE-UNDERLINE USE-WIDGET-POOL USING UTC-OFFSET V6DISPLAY V6FRAME VALIDATE
          VALIDATE-DOMAIN-ACCESS-CODE VALIDATE-EXPRESSIO VALIDATE-EXPRESSION VALIDATE-MESSAGE VALIDATE-SEAL
          VALIDATE-XML VALIDATION-ENABLED VALID-EVENT VALID-HANDLE VALID-OBJECT VALUE VALUE-CHANGED VALUES VAR
          VARI VARIA VARIAB VARIABL VARIABLE VERB VERBO VERBOS VERBOSE VERSION VERT VERTI VERTIC VERTICA
          VERTICAL VIEW VIEW-AS VIEW-FIRST-COLUMN-ON-REOPEN VIRTUAL-HEIGHT VIRTUAL-HEIGHT-C VIRTUAL-HEIGHT-CH
          VIRTUAL-HEIGHT-CHA VIRTUAL-HEIGHT-CHAR VIRTUAL-HEIGHT-CHARS VIRTUAL-HEIGHT-P VIRTUAL-HEIGHT-PI
          VIRTUAL-HEIGHT-PIX VIRTUAL-HEIGHT-PIXE VIRTUAL-HEIGHT-PIXEL VIRTUAL-HEIGHT-PIXELS VIRTUAL-WIDTH
          VIRTUAL-WIDTH-C VIRTUAL-WIDTH-CH VIRTUAL-WIDTH-CHA VIRTUAL-WIDTH-CHAR VIRTUAL-WIDTH-CHARS
          VIRTUAL-WIDTH-P VIRTUAL-WIDTH-PI VIRTUAL-WIDTH-PIX VIRTUAL-WIDTH-PIXE VIRTUAL-WIDTH-PIXEL
          VIRTUAL-WIDTH-PIXELS VISIBLE VMS VOID WAIT WAIT-FOR WARNING WC-ADMIN-APP WEB-CON WEB-CONT WEB-CONTE
          WEB-CONTEX WEB-CONTEXT WEB-NOTIFY WEEKDAY WHEN WHERE WHERE-STRING WHILE WIDGET WIDGET-E WIDGET-EN
          WIDGET-ENT WIDGET-ENTE WIDGET-ENTER WIDGET-H WIDGET-HA WIDGET-HAN WIDGET-HAND WIDGET-HANDL
          WIDGET-HANDLE WIDGET-ID WIDGET-L WIDGET-LE WIDGET-LEA WIDGET-LEAV WIDGET-LEAVE WIDGET-POOL WIDTH
          WIDTH-C WIDTH-CH WIDTH-CHA WIDTH-CHAR WIDTH-CHARS WIDTH-P WIDTH-PI WIDTH-PIX WIDTH-PIXE WIDTH-PIXEL
          WIDTH-PIXELS WINDOW WINDOW-CLOSE WINDOW-DELAYED-MIN WINDOW-DELAYED-MINI WINDOW-DELAYED-MINIM
          WINDOW-DELAYED-MINIMI WINDOW-DELAYED-MINIMIZ WINDOW-DELAYED-MINIMIZE WINDOW-MAXIM WINDOW-MAXIMI
          WINDOW-MAXIMIZ WINDOW-MAXIMIZE WINDOW-MAXIMIZED WINDOW-MINIM WINDOW-MINIMI WINDOW-MINIMIZ
          WINDOW-MINIMIZE WINDOW-MINIMIZED WINDOW-NAME WINDOW-NORMAL WINDOW-RESIZED WINDOW-RESTORED WINDOW-STA
          WINDOW-STAT WINDOW-STATE WINDOW-SYS WINDOW-SYST WINDOW-SYSTE WINDOW-SYSTEM WITH WORD-INDEX WORD-WRAP
          WORK-AREA-HEIGHT-P WORK-AREA-HEIGHT-PI WORK-AREA-HEIGHT-PIX WORK-AREA-HEIGHT-PIXE
          WORK-AREA-HEIGHT-PIXEL WORK-AREA-HEIGHT-PIXELS WORK-AREA-WIDTH-P WORK-AREA-WIDTH-PI
          WORK-AREA-WIDTH-PIX WORK-AREA-WIDTH-PIXE WORK-AREA-WIDTH-PIXEL WORK-AREA-WIDTH-PIXELS WORK-AREA-X
          WORK-AREA-Y WORKFILE WORK-TAB WORK-TABL WORK-TABLE WRITE WRITE-CDATA WRITE-CHARACTERS WRITE-COMMENT
          WRITE-DATA WRITE-DATA-ELEMENT WRITE-EMPTY-ELEMENT WRITE-ENTITY-REF WRITE-EXTERNAL-DTD WRITE-FRAGMENT
          WRITE-JSON WRITE-MESSAGE WRITE-PROCESSING-INSTRUCTION WRITE-STATUS WRITE-XML WRITE-XMLSCHEMA X XCODE
          XCODE-SESSION-KEY X-DOCUMENT XML-DATA-TYPE XML-ENTITY-EXPANSION-LIMIT XML-NODE-NAME XML-NODE-TYPE
          XML-SCHEMA-PAT XML-SCHEMA-PATH XML-STRICT-ENTITY-RESOLUTION XML-SUPPRESS-NAMESPACE-PROCESSING
          X-NODEREF X-OF XOR XREF XREF-XML Y YEAR YEAR-OFFSET YES YES-NO YES-NO-CANCEL Y-OF          
        )
      end

      def self.keywords_prepro
        @keywords_prepro ||= Set.new %w(
          &ANALYZE-SUSPEND &ANALYZE-RESUME
          &ELSE &ELSEIF &ENDIF &GLOB &GLOBAL-DEFINE &IF &MESSAGE &SCOP &SCOPED-DEFINE &THEN &UNDEF &UNDEFINE &WEBSTREAM
          {&BATCH} {&BATCH-MODE} {&FILE-NAME} {&LINE-NUMBE} {&LINE-NUMBER} {&OPSYS} {&PROCESS-ARCHITECTURE} {&SEQUENCE}
          {&WINDOW-SYS} {&WINDOW-SYSTEM}
        )
      end

      def self.keywords_type
        @keywords_type ||= Set.new %w(
          BLOB CHARACTER CHAR CLOB COM-HANDLE DATE DATETIME DATETIME-TZ DECIMAL
          DEC HANDLE INT64 INTEGER INT LOGICAL LONGCHAR MEMPTR RAW RECID ROWID
          WIDGET-HANDLE VOID
        )
      end

      state :statements do
        mixin :singlelinecomment
        mixin :multilinecomment
        mixin :string
        mixin :numeric
        mixin :constant
        mixin :operator
        mixin :whitespace
        mixin :preproc
        mixin :decorator
        mixin :function
      end

      state :whitespace do
        rule %r/\s+/m, Text 
      end

      state :singlelinecomment do
        rule %r(//.*$), Comment::Single
      end

      state :multilinecomment do
        rule %r(/[*]), Comment::Multiline, :multilinecomment_content
      end

      state :multilinecomment_content do
        rule %r([*]/), Comment::Multiline, :pop!
        rule %r(/[*]), Comment::Multiline, :multilinecomment_content
        rule %r([^*/]+), Comment::Multiline
        rule %r([*/]), Comment::Multiline
      end

      state :string do
        rule %r/"/, Str::Delimiter, :doublequotedstring
        rule %r/'/, Str::Delimiter, :singlequotedstring
      end

      state :numeric do
        rule %r((?:\d+[.]\d*|[.]\d+)(?:e[+-]?\d+[lu]*)?)i, Num::Float
        rule %r(\d+e[+-]?\d+[lu]*)i, Num::Float
        rule %r/0x[0-9a-f]+[lu]*/i, Num::Hex
        rule %r/0[0-7]+[lu]*/i, Num::Oct
        rule %r/\d+[lu]*/i, Num::Integer
      end

      state :constant do
        rule %r/(?:TRUE|FALSE|YES|NO(?!\-)|\?)\b/i, Keyword::Constant
      end

      state :operator do
        rule %r([~!%^*+=\|?:<>/-]), Operator
        rule %r/[()\[\],.]/, Punctuation
      end
      
      state :doublequotedstring do
        rule %r/\~[~nrt"]?/, Str::Escape
        rule %r/[^\\"]+/, Str::Double
        rule %r/"/, Str::Delimiter, :pop!
      end

      state :singlequotedstring do
        rule %r/\~[~nrt']?/, Str::Escape
        rule %r/[^\\']+/, Str::Single
        rule %r/'/, Str::Delimiter, :pop!
      end
      
      state :preproc do
        rule %r/(\&analyze-suspend|\&analyze-resume)/i, Comment::Preproc, :analyze_suspend_resume_content
        rule %r/(\&scoped-define|\&global-define)\s*([\.\w\\\/-]*)/i , Comment::Preproc, :analyze_suspend_resume_content
        mixin :include_file
      end

      state :analyze_suspend_resume_content do
        rule %r/.*(?=(?:\/\/|\/\*))/, Comment::Preproc, :pop!
        rule %r/.*\n/, Comment::Preproc, :pop!
        rule %r/.*/, Comment::Preproc, :pop!
      end
      
      state :preproc_content do
        rule %r/\n/, Text, :pop!
        rule %r/\s+/, Text
 
        rule %r/({?&)(\S+)/ do
          groups Comment::Preproc, Name::Other
        end
 
        rule %r/"/, Str, :string
        mixin :numeric
 
        rule %r/\S+/, Name
      end

      state :include_file do
        rule %r/(\{(?:[^\{\}]|(\g<0>))+\})/i , Comment::Preproc
      end

      state :decorator do
        rule %r/@#{id}/, Name::Decorator #, :decorator_content
      end

      state :function do
        rule %r/\!?#{id}(?=\s*\()/i , Name::Function
      end

      state :root do
        mixin :statements

        rule id do |m|
          name = m[0].upcase
          if self.class.keywords_prepro.include? name
            token Comment::Preproc
          elsif self.class.keywords.include? name
            token Keyword
          elsif self.class.keywords_type.include? name
            token Keyword::Type
          else
            token Name::Variable
          end
        end
      end
    end
  end
end
