"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.IntegrationInstanceBuilder = void 0;
var _uuid = require("uuid");
var _utils = require("./repository/utils");
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
class IntegrationInstanceBuilder {
  constructor(client) {
    _defineProperty(this, "client", void 0);
    this.client = client;
  }
  async build(integration, options) {
    const instance = await (0, _utils.deepCheck)(integration);
    if (!instance.ok) {
      return Promise.reject(instance.error);
    }
    const assets = await integration.getAssets();
    if (!assets.ok) {
      return Promise.reject(assets.error);
    }
    const remapped = this.remapIDs(this.getSavedObjectBundles(assets.value));
    const withDataSource = this.remapDataSource(remapped, options.dataSource);
    const refs = await this.postAssets(withDataSource);
    const builtInstance = await this.buildInstance(integration, refs, options);
    return builtInstance;
  }
  getSavedObjectBundles(assets, includeWorkflows) {
    return assets.filter(asset => {
      // At this stage we only care about installing bundles
      if (asset.type !== 'savedObjectBundle') {
        return false;
      }
      // If no workflows present: default to all workflows
      // Otherwise only install if workflow is present
      if (!asset.workflows || !includeWorkflows) {
        return true;
      }
      return includeWorkflows.some(w => {
        var _asset$workflows;
        return (_asset$workflows = asset.workflows) === null || _asset$workflows === void 0 ? void 0 : _asset$workflows.includes(w);
      });
    }).map(asset => asset.data).flat();
  }
  remapDataSource(assets, dataSource) {
    if (!dataSource) return assets;
    assets = assets.map(asset => {
      if (asset.type === 'index-pattern') {
        asset.attributes.title = dataSource;
      }
      return asset;
    });
    return assets;
  }
  remapIDs(assets) {
    const toRemap = assets.filter(asset => asset.id);
    const idMap = new Map();
    return toRemap.map(item => {
      if (!idMap.has(item.id)) {
        idMap.set(item.id, (0, _uuid.v4)());
      }
      item.id = idMap.get(item.id);
      for (let ref = 0; ref < item.references.length; ref++) {
        const refId = item.references[ref].id;
        if (!idMap.has(refId)) {
          idMap.set(refId, (0, _uuid.v4)());
        }
        item.references[ref].id = idMap.get(refId);
      }
      return item;
    });
  }
  async postAssets(assets) {
    try {
      const response = await this.client.bulkCreate(assets);
      const refs = response.saved_objects.map(obj => {
        var _obj$attributes;
        return {
          assetType: obj.type,
          assetId: obj.id,
          status: 'available',
          // Assuming a successfully created object is available
          isDefaultAsset: obj.type === 'dashboard',
          // Assuming for now that dashboards are default
          description: (_obj$attributes = obj.attributes) === null || _obj$attributes === void 0 ? void 0 : _obj$attributes.title
        };
      });
      return Promise.resolve(refs);
    } catch (err) {
      return Promise.reject(err);
    }
  }
  async buildInstance(integration, refs, options) {
    const config = await integration.getConfig();
    if (!config.ok) {
      return Promise.reject(new Error('Attempted to create instance with invalid template', config.error));
    }
    return Promise.resolve({
      name: options.name,
      templateName: config.value.name,
      dataSource: options.dataSource,
      creationDate: new Date().toISOString(),
      assets: refs
    });
  }
}
exports.IntegrationInstanceBuilder = IntegrationInstanceBuilder;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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