"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerDataConnectionsRoute = registerDataConnectionsRoute;
var _configSchema = require("@osd/config-schema");
var _shared = require("../../../common/constants/shared");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function registerDataConnectionsRoute(router) {
  router.get({
    path: `${_shared.DATACONNECTIONS_BASE}/{name}`,
    validate: {
      params: _configSchema.schema.object({
        name: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    try {
      const dataConnectionsresponse = await context.observability_plugin.observabilityClient.asScoped(request).callAsCurrentUser('ppl.getDataConnectionById', {
        dataconnection: request.params.name
      });
      return response.ok({
        body: dataConnectionsresponse
      });
    } catch (error) {
      console.error('Issue in fetching data connection:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.delete({
    path: `${_shared.DATACONNECTIONS_BASE}/{name}`,
    validate: {
      params: _configSchema.schema.object({
        name: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    try {
      const dataConnectionsresponse = await context.observability_plugin.observabilityClient.asScoped(request).callAsCurrentUser('ppl.deleteDataConnection', {
        dataconnection: request.params.name
      });
      return response.ok({
        body: dataConnectionsresponse
      });
    } catch (error) {
      console.error('Issue in deleting data connection:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: `${_shared.DATACONNECTIONS_BASE}${_shared.EDIT}`,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        allowedRoles: _configSchema.schema.arrayOf(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      const dataConnectionsresponse = await context.observability_plugin.observabilityClient.asScoped(request).callAsCurrentUser('ppl.modifyDataConnection', {
        body: {
          name: request.body.name,
          allowedRoles: request.body.allowedRoles
        }
      });
      return response.ok({
        body: dataConnectionsresponse
      });
    } catch (error) {
      console.error('Issue in modifying data connection:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: `${_shared.DATACONNECTIONS_BASE}${_shared.EDIT}${_shared.DATACONNECTIONS_UPDATE_STATUS}`,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        status: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    try {
      const dataConnectionsresponse = await context.observability_plugin.observabilityClient.asScoped(request).callAsCurrentUser('ppl.modifyDataConnection', {
        body: {
          name: request.body.name,
          status: request.body.status
        }
      });
      return response.ok({
        body: dataConnectionsresponse
      });
    } catch (error) {
      console.error('Issue in modifying data connection:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: `${_shared.DATACONNECTIONS_BASE}`,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        connector: _configSchema.schema.string(),
        allowedRoles: _configSchema.schema.arrayOf(_configSchema.schema.string()),
        properties: _configSchema.schema.any()
      })
    }
  }, async (context, request, response) => {
    try {
      const dataConnectionsresponse = await context.observability_plugin.observabilityClient.asScoped(request).callAsCurrentUser('ppl.createDataSource', {
        body: {
          name: request.body.name,
          connector: request.body.connector,
          allowedRoles: request.body.allowedRoles,
          properties: request.body.properties
        }
      });
      return response.ok({
        body: dataConnectionsresponse
      });
    } catch (error) {
      console.error('Issue in creating data source:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.response
      });
    }
  });
  router.get({
    path: `${_shared.DATACONNECTIONS_BASE}`,
    validate: false
  }, async (context, request, response) => {
    try {
      const dataConnectionsresponse = await context.observability_plugin.observabilityClient.asScoped(request).callAsCurrentUser('ppl.getDataConnections');
      return response.ok({
        body: dataConnectionsresponse
      });
    } catch (error) {
      console.error('Issue in fetching data sources:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.response
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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