/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.sql.datasources.storage;

import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import org.apache.commons.io.IOUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.opensearch.action.DocWriteRequest;
import org.opensearch.action.DocWriteResponse;
import org.opensearch.action.admin.indices.create.CreateIndexRequest;
import org.opensearch.action.admin.indices.create.CreateIndexResponse;
import org.opensearch.action.delete.DeleteRequest;
import org.opensearch.action.delete.DeleteResponse;
import org.opensearch.action.index.IndexRequest;
import org.opensearch.action.index.IndexResponse;
import org.opensearch.action.search.SearchRequest;
import org.opensearch.action.search.SearchResponse;
import org.opensearch.action.support.WriteRequest;
import org.opensearch.action.update.UpdateRequest;
import org.opensearch.action.update.UpdateResponse;
import org.opensearch.client.Client;
import org.opensearch.cluster.service.ClusterService;
import org.opensearch.common.action.ActionFuture;
import org.opensearch.common.util.concurrent.ThreadContext;
import org.opensearch.common.xcontent.XContentType;
import org.opensearch.index.engine.DocumentMissingException;
import org.opensearch.index.engine.VersionConflictEngineException;
import org.opensearch.index.query.QueryBuilder;
import org.opensearch.index.query.QueryBuilders;
import org.opensearch.search.SearchHit;
import org.opensearch.search.builder.SearchSourceBuilder;
import org.opensearch.sql.datasource.model.DataSourceMetadata;
import org.opensearch.sql.datasources.encryptor.Encryptor;
import org.opensearch.sql.datasources.exceptions.DataSourceNotFoundException;
import org.opensearch.sql.datasources.service.DataSourceMetadataStorage;
import org.opensearch.sql.datasources.utils.XContentParserUtils;

public class OpenSearchDataSourceMetadataStorage
implements DataSourceMetadataStorage {
    public static final String DATASOURCE_INDEX_NAME = ".ql-datasources";
    private static final String DATASOURCE_INDEX_MAPPING_FILE_NAME = "datasources-index-mapping.yml";
    private static final Integer DATASOURCE_QUERY_RESULT_SIZE = 10000;
    private static final String DATASOURCE_INDEX_SETTINGS_FILE_NAME = "datasources-index-settings.yml";
    private static final Logger LOG = LogManager.getLogger();
    private final Client client;
    private final ClusterService clusterService;
    private final Encryptor encryptor;

    public OpenSearchDataSourceMetadataStorage(Client client, ClusterService clusterService, Encryptor encryptor) {
        this.client = client;
        this.clusterService = clusterService;
        this.encryptor = encryptor;
    }

    @Override
    public List<DataSourceMetadata> getDataSourceMetadata() {
        if (!this.clusterService.state().routingTable().hasIndex(DATASOURCE_INDEX_NAME)) {
            this.createDataSourcesIndex();
            return Collections.emptyList();
        }
        return this.searchInDataSourcesIndex((QueryBuilder)QueryBuilders.matchAllQuery());
    }

    @Override
    public Optional<DataSourceMetadata> getDataSourceMetadata(String datasourceName) {
        if (!this.clusterService.state().routingTable().hasIndex(DATASOURCE_INDEX_NAME)) {
            this.createDataSourcesIndex();
            return Optional.empty();
        }
        return this.searchInDataSourcesIndex((QueryBuilder)QueryBuilders.termQuery((String)"name.keyword", (String)datasourceName)).stream().findFirst().map(x -> this.encryptDecryptAuthenticationData((DataSourceMetadata)x, false));
    }

    @Override
    public void createDataSourceMetadata(DataSourceMetadata dataSourceMetadata) {
        IndexResponse indexResponse;
        this.encryptDecryptAuthenticationData(dataSourceMetadata, true);
        if (!this.clusterService.state().routingTable().hasIndex(DATASOURCE_INDEX_NAME)) {
            this.createDataSourcesIndex();
        }
        IndexRequest indexRequest = new IndexRequest(DATASOURCE_INDEX_NAME);
        indexRequest.id(dataSourceMetadata.getName());
        indexRequest.opType(DocWriteRequest.OpType.CREATE);
        indexRequest.setRefreshPolicy(WriteRequest.RefreshPolicy.IMMEDIATE);
        try (ThreadContext.StoredContext storedContext = this.client.threadPool().getThreadContext().stashContext();){
            indexRequest.source(XContentParserUtils.convertToXContent(dataSourceMetadata));
            ActionFuture indexResponseActionFuture = this.client.index(indexRequest);
            indexResponse = (IndexResponse)indexResponseActionFuture.actionGet();
        }
        catch (VersionConflictEngineException exception) {
            throw new IllegalArgumentException("A datasource already exists with name: " + dataSourceMetadata.getName());
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
        if (!indexResponse.getResult().equals((Object)DocWriteResponse.Result.CREATED)) {
            throw new RuntimeException("Saving dataSource metadata information failed with result : " + indexResponse.getResult().getLowercase());
        }
        LOG.debug("DatasourceMetadata : {}  successfully created", (Object)dataSourceMetadata.getName());
    }

    @Override
    public void updateDataSourceMetadata(DataSourceMetadata dataSourceMetadata) {
        UpdateResponse updateResponse;
        this.encryptDecryptAuthenticationData(dataSourceMetadata, true);
        UpdateRequest updateRequest = new UpdateRequest(DATASOURCE_INDEX_NAME, dataSourceMetadata.getName());
        try (ThreadContext.StoredContext storedContext = this.client.threadPool().getThreadContext().stashContext();){
            updateRequest.doc(XContentParserUtils.convertToXContent(dataSourceMetadata));
            updateRequest.setRefreshPolicy(WriteRequest.RefreshPolicy.IMMEDIATE);
            ActionFuture updateResponseActionFuture = this.client.update(updateRequest);
            updateResponse = (UpdateResponse)updateResponseActionFuture.actionGet();
        }
        catch (DocumentMissingException exception) {
            throw new DataSourceNotFoundException("Datasource with name: " + dataSourceMetadata.getName() + " doesn't exist");
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
        if (!updateResponse.getResult().equals((Object)DocWriteResponse.Result.UPDATED) && !updateResponse.getResult().equals((Object)DocWriteResponse.Result.NOOP)) {
            throw new RuntimeException("Saving dataSource metadata information failed with result : " + updateResponse.getResult().getLowercase());
        }
        LOG.debug("DatasourceMetadata : {}  successfully updated", (Object)dataSourceMetadata.getName());
    }

    @Override
    public void deleteDataSourceMetadata(String datasourceName) {
        ActionFuture deleteResponseActionFuture;
        DeleteRequest deleteRequest = new DeleteRequest(DATASOURCE_INDEX_NAME);
        deleteRequest.id(datasourceName);
        deleteRequest.setRefreshPolicy(WriteRequest.RefreshPolicy.IMMEDIATE);
        try (ThreadContext.StoredContext storedContext = this.client.threadPool().getThreadContext().stashContext();){
            deleteResponseActionFuture = this.client.delete(deleteRequest);
        }
        DeleteResponse deleteResponse = (DeleteResponse)deleteResponseActionFuture.actionGet();
        if (!deleteResponse.getResult().equals((Object)DocWriteResponse.Result.DELETED)) {
            if (deleteResponse.getResult().equals((Object)DocWriteResponse.Result.NOT_FOUND)) {
                throw new DataSourceNotFoundException("Datasource with name: " + datasourceName + " doesn't exist");
            }
            throw new RuntimeException("Deleting dataSource metadata information failed with result : " + deleteResponse.getResult().getLowercase());
        }
        LOG.debug("DatasourceMetadata : {}  successfully deleted", (Object)datasourceName);
    }

    private void createDataSourcesIndex() {
        try {
            ActionFuture createIndexResponseActionFuture;
            InputStream mappingFileStream = OpenSearchDataSourceMetadataStorage.class.getClassLoader().getResourceAsStream(DATASOURCE_INDEX_MAPPING_FILE_NAME);
            InputStream settingsFileStream = OpenSearchDataSourceMetadataStorage.class.getClassLoader().getResourceAsStream(DATASOURCE_INDEX_SETTINGS_FILE_NAME);
            CreateIndexRequest createIndexRequest = new CreateIndexRequest(DATASOURCE_INDEX_NAME);
            createIndexRequest.mapping(IOUtils.toString((InputStream)mappingFileStream, (Charset)StandardCharsets.UTF_8), XContentType.YAML).settings(IOUtils.toString((InputStream)settingsFileStream, (Charset)StandardCharsets.UTF_8), XContentType.YAML);
            try (ThreadContext.StoredContext ignored = this.client.threadPool().getThreadContext().stashContext();){
                createIndexResponseActionFuture = this.client.admin().indices().create(createIndexRequest);
            }
            CreateIndexResponse createIndexResponse = (CreateIndexResponse)createIndexResponseActionFuture.actionGet();
            if (!createIndexResponse.isAcknowledged()) {
                throw new RuntimeException("Index creation is not acknowledged.");
            }
            LOG.info("Index: {} creation Acknowledged", (Object)DATASOURCE_INDEX_NAME);
        }
        catch (Throwable e) {
            throw new RuntimeException("Internal server error while creating.ql-datasources index:: " + e.getMessage());
        }
    }

    private List<DataSourceMetadata> searchInDataSourcesIndex(QueryBuilder query) {
        ActionFuture searchResponseActionFuture;
        SearchRequest searchRequest = new SearchRequest();
        searchRequest.indices(new String[]{DATASOURCE_INDEX_NAME});
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        searchSourceBuilder.query(query);
        searchSourceBuilder.size(DATASOURCE_QUERY_RESULT_SIZE.intValue());
        searchRequest.source(searchSourceBuilder);
        searchRequest.preference("_primary_first");
        try (ThreadContext.StoredContext ignored = this.client.threadPool().getThreadContext().stashContext();){
            searchResponseActionFuture = this.client.search(searchRequest);
        }
        SearchResponse searchResponse = (SearchResponse)searchResponseActionFuture.actionGet();
        if (searchResponse.status().getStatus() != 200) {
            throw new RuntimeException("Fetching dataSource metadata information failed with status : " + searchResponse.status());
        }
        ArrayList<DataSourceMetadata> list = new ArrayList<DataSourceMetadata>();
        for (SearchHit searchHit : searchResponse.getHits().getHits()) {
            DataSourceMetadata dataSourceMetadata;
            String sourceAsString = searchHit.getSourceAsString();
            try {
                dataSourceMetadata = XContentParserUtils.toDataSourceMetadata(sourceAsString);
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
            list.add(dataSourceMetadata);
        }
        return list;
    }

    private DataSourceMetadata encryptDecryptAuthenticationData(DataSourceMetadata dataSourceMetadata, Boolean isEncryption) {
        Map propertiesMap = dataSourceMetadata.getProperties();
        this.handleBasicAuthPropertiesEncryptionDecryption(propertiesMap, isEncryption);
        this.handleSigV4PropertiesEncryptionDecryption(propertiesMap, isEncryption);
        return dataSourceMetadata;
    }

    private void handleBasicAuthPropertiesEncryptionDecryption(Map<String, String> propertiesMap, Boolean isEncryption) {
        ArrayList<String> list = new ArrayList<String>();
        propertiesMap.keySet().stream().filter(s -> s.endsWith("auth.username")).findFirst().ifPresent(list::add);
        propertiesMap.keySet().stream().filter(s -> s.endsWith("auth.password")).findFirst().ifPresent(list::add);
        this.encryptOrDecrypt(propertiesMap, isEncryption, list);
    }

    private void encryptOrDecrypt(Map<String, String> propertiesMap, Boolean isEncryption, List<String> keyIdentifiers) {
        for (String key : keyIdentifiers) {
            if (isEncryption.booleanValue()) {
                propertiesMap.put(key, this.encryptor.encrypt(propertiesMap.get(key)));
                continue;
            }
            propertiesMap.put(key, this.encryptor.decrypt(propertiesMap.get(key)));
        }
    }

    private void handleSigV4PropertiesEncryptionDecryption(Map<String, String> propertiesMap, Boolean isEncryption) {
        ArrayList<String> list = new ArrayList<String>();
        propertiesMap.keySet().stream().filter(s -> s.endsWith("auth.access_key")).findFirst().ifPresent(list::add);
        propertiesMap.keySet().stream().filter(s -> s.endsWith("auth.secret_key")).findFirst().ifPresent(list::add);
        this.encryptOrDecrypt(propertiesMap, isEncryption, list);
    }
}

