"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerQueryAssistRoutes = registerQueryAssistRoutes;
var _configSchema = require("@osd/config-schema");
var _errors = require("../../../../../src/core/server/opensearch/client/errors");
var _query_assist = require("../../../common/constants/query_assist");
var _generate_field_context = require("../../common/helpers/query_assist/generate_field_context");
var _agents = require("./utils/agents");
var _constants = require("./utils/constants");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function registerQueryAssistRoutes(router) {
  /**
   * Returns whether the PPL agent is configured.
   */
  router.get({
    path: _query_assist.QUERY_ASSIST_API.CONFIGURED,
    validate: {}
  }, async (context, request, response) => {
    const client = context.core.opensearch.client.asCurrentUser;
    try {
      // if the call does not throw any error, then the agent is properly configured
      await (0, _agents.getAgentIdByConfig)(client, _constants.AGENT_CONFIGS.PPL_AGENT);
      return response.ok({
        body: {
          configured: true
        }
      });
    } catch (error) {
      return response.ok({
        body: {
          configured: false,
          error: error.message
        }
      });
    }
  });
  router.post({
    path: _query_assist.QUERY_ASSIST_API.GENERATE_PPL,
    validate: {
      body: _configSchema.schema.object({
        index: _configSchema.schema.string(),
        question: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const client = context.core.opensearch.client.asCurrentUser;
    try {
      const pplRequest = await (0, _agents.requestWithRetryAgentSearch)({
        client,
        configName: _constants.AGENT_CONFIGS.PPL_AGENT,
        body: {
          parameters: {
            index: request.body.index,
            question: request.body.question
          }
        }
      });
      if (!pplRequest.body.inference_results[0].output[0].result) throw new Error('Generated PPL query not found.');
      const result = JSON.parse(pplRequest.body.inference_results[0].output[0].result);
      const ppl = result.ppl.replace(/[\r\n]/g, ' ').trim().replace(/ISNOTNULL/g, 'isnotnull') // https://github.com/opensearch-project/sql/issues/2431
      .replace(/`/g, '') // https://github.com/opensearch-project/dashboards-observability/issues/509, https://github.com/opensearch-project/dashboards-observability/issues/557
      .replace(/\bSPAN\(/g, 'span('); // https://github.com/opensearch-project/dashboards-observability/issues/759
      return response.ok({
        body: ppl
      });
    } catch (error) {
      if ((0, _errors.isResponseError)(error) && error.statusCode === 400 && error.body.includes(_query_assist.ERROR_DETAILS.GUARDRAILS_TRIGGERED)) {
        return response.badRequest({
          body: _query_assist.ERROR_DETAILS.GUARDRAILS_TRIGGERED
        });
      }
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: _query_assist.QUERY_ASSIST_API.SUMMARIZE,
    validate: {
      body: _configSchema.schema.object({
        index: _configSchema.schema.string(),
        question: _configSchema.schema.string(),
        query: _configSchema.schema.maybe(_configSchema.schema.string()),
        response: _configSchema.schema.string(),
        isError: _configSchema.schema.boolean()
      })
    }
  }, async (context, request, response) => {
    const client = context.core.opensearch.client.asCurrentUser;
    const {
      index,
      question,
      query,
      response: _response,
      isError
    } = request.body;
    const queryResponse = JSON.stringify(_response);
    let summaryRequest;
    try {
      var _summaryRequest$body$;
      if (!isError) {
        summaryRequest = await (0, _agents.requestWithRetryAgentSearch)({
          client,
          configName: _constants.AGENT_CONFIGS.RESPONSE_SUMMARY_AGENT,
          body: {
            parameters: {
              index,
              question,
              query,
              response: queryResponse
            }
          }
        });
      } else {
        const [mappings, sampleDoc] = await Promise.all([client.indices.getMapping({
          index
        }), client.search({
          index,
          size: 1
        })]);
        const fields = (0, _generate_field_context.generateFieldContext)(mappings, sampleDoc);
        summaryRequest = await (0, _agents.requestWithRetryAgentSearch)({
          client,
          configName: _constants.AGENT_CONFIGS.ERROR_SUMMARY_AGENT,
          body: {
            parameters: {
              index,
              question,
              query,
              response: queryResponse,
              fields
            }
          }
        });
      }
      const summary = summaryRequest.body.inference_results[0].output[0].result;
      if (!summary) throw new Error('Generated summary not found.');
      const suggestedQuestions = Array.from((((_summaryRequest$body$ = summaryRequest.body.inference_results[0].output[1]) === null || _summaryRequest$body$ === void 0 ? void 0 : _summaryRequest$body$.result) || '').matchAll(/<question>((.|[\r\n])+?)<\/question>/g)).map(m => m[1]);
      return response.ok({
        body: {
          summary,
          suggestedQuestions
        }
      });
    } catch (error) {
      if ((0, _errors.isResponseError)(error) && error.statusCode === 400 && error.body.includes(_query_assist.ERROR_DETAILS.GUARDRAILS_TRIGGERED)) {
        return response.badRequest({
          body: _query_assist.ERROR_DETAILS.GUARDRAILS_TRIGGERED
        });
      }
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJuYW1lcyI6WyJfY29uZmlnU2NoZW1hIiwicmVxdWlyZSIsIl9lcnJvcnMiLCJfcXVlcnlfYXNzaXN0IiwiX2dlbmVyYXRlX2ZpZWxkX2NvbnRleHQiLCJfYWdlbnRzIiwiX2NvbnN0YW50cyIsInJlZ2lzdGVyUXVlcnlBc3Npc3RSb3V0ZXMiLCJyb3V0ZXIiLCJnZXQiLCJwYXRoIiwiUVVFUllfQVNTSVNUX0FQSSIsIkNPTkZJR1VSRUQiLCJ2YWxpZGF0ZSIsImNvbnRleHQiLCJyZXF1ZXN0IiwicmVzcG9uc2UiLCJjbGllbnQiLCJjb3JlIiwib3BlbnNlYXJjaCIsImFzQ3VycmVudFVzZXIiLCJnZXRBZ2VudElkQnlDb25maWciLCJBR0VOVF9DT05GSUdTIiwiUFBMX0FHRU5UIiwib2siLCJib2R5IiwiY29uZmlndXJlZCIsImVycm9yIiwibWVzc2FnZSIsInBvc3QiLCJHRU5FUkFURV9QUEwiLCJzY2hlbWEiLCJvYmplY3QiLCJpbmRleCIsInN0cmluZyIsInF1ZXN0aW9uIiwicHBsUmVxdWVzdCIsInJlcXVlc3RXaXRoUmV0cnlBZ2VudFNlYXJjaCIsImNvbmZpZ05hbWUiLCJwYXJhbWV0ZXJzIiwiaW5mZXJlbmNlX3Jlc3VsdHMiLCJvdXRwdXQiLCJyZXN1bHQiLCJFcnJvciIsIkpTT04iLCJwYXJzZSIsInBwbCIsInJlcGxhY2UiLCJ0cmltIiwiaXNSZXNwb25zZUVycm9yIiwic3RhdHVzQ29kZSIsImluY2x1ZGVzIiwiRVJST1JfREVUQUlMUyIsIkdVQVJEUkFJTFNfVFJJR0dFUkVEIiwiYmFkUmVxdWVzdCIsImN1c3RvbSIsIlNVTU1BUklaRSIsInF1ZXJ5IiwibWF5YmUiLCJpc0Vycm9yIiwiYm9vbGVhbiIsIl9yZXNwb25zZSIsInF1ZXJ5UmVzcG9uc2UiLCJzdHJpbmdpZnkiLCJzdW1tYXJ5UmVxdWVzdCIsIl9zdW1tYXJ5UmVxdWVzdCRib2R5JCIsIlJFU1BPTlNFX1NVTU1BUllfQUdFTlQiLCJtYXBwaW5ncyIsInNhbXBsZURvYyIsIlByb21pc2UiLCJhbGwiLCJpbmRpY2VzIiwiZ2V0TWFwcGluZyIsInNlYXJjaCIsInNpemUiLCJmaWVsZHMiLCJnZW5lcmF0ZUZpZWxkQ29udGV4dCIsIkVSUk9SX1NVTU1BUllfQUdFTlQiLCJzdW1tYXJ5Iiwic3VnZ2VzdGVkUXVlc3Rpb25zIiwiQXJyYXkiLCJmcm9tIiwibWF0Y2hBbGwiLCJtYXAiLCJtIl0sInNvdXJjZXMiOlsicm91dGVzLnRzIl0sInNvdXJjZXNDb250ZW50IjpbIi8qXG4gKiBDb3B5cmlnaHQgT3BlblNlYXJjaCBDb250cmlidXRvcnNcbiAqIFNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wXG4gKi9cblxuaW1wb3J0IHsgc2NoZW1hIH0gZnJvbSAnQG9zZC9jb25maWctc2NoZW1hJztcbmltcG9ydCB7XG4gIElPcGVuU2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlLFxuICBJUm91dGVyLFxuICBSZXNwb25zZUVycm9yLFxufSBmcm9tICcuLi8uLi8uLi8uLi8uLi9zcmMvY29yZS9zZXJ2ZXInO1xuaW1wb3J0IHsgaXNSZXNwb25zZUVycm9yIH0gZnJvbSAnLi4vLi4vLi4vLi4vLi4vc3JjL2NvcmUvc2VydmVyL29wZW5zZWFyY2gvY2xpZW50L2Vycm9ycyc7XG5pbXBvcnQgeyBFUlJPUl9ERVRBSUxTLCBRVUVSWV9BU1NJU1RfQVBJIH0gZnJvbSAnLi4vLi4vLi4vY29tbW9uL2NvbnN0YW50cy9xdWVyeV9hc3Npc3QnO1xuaW1wb3J0IHsgZ2VuZXJhdGVGaWVsZENvbnRleHQgfSBmcm9tICcuLi8uLi9jb21tb24vaGVscGVycy9xdWVyeV9hc3Npc3QvZ2VuZXJhdGVfZmllbGRfY29udGV4dCc7XG5pbXBvcnQgeyBnZXRBZ2VudElkQnlDb25maWcsIHJlcXVlc3RXaXRoUmV0cnlBZ2VudFNlYXJjaCB9IGZyb20gJy4vdXRpbHMvYWdlbnRzJztcbmltcG9ydCB7IEFHRU5UX0NPTkZJR1MgfSBmcm9tICcuL3V0aWxzL2NvbnN0YW50cyc7XG5cbmV4cG9ydCBmdW5jdGlvbiByZWdpc3RlclF1ZXJ5QXNzaXN0Um91dGVzKHJvdXRlcjogSVJvdXRlcikge1xuICAvKipcbiAgICogUmV0dXJucyB3aGV0aGVyIHRoZSBQUEwgYWdlbnQgaXMgY29uZmlndXJlZC5cbiAgICovXG4gIHJvdXRlci5nZXQoXG4gICAge1xuICAgICAgcGF0aDogUVVFUllfQVNTSVNUX0FQSS5DT05GSUdVUkVELFxuICAgICAgdmFsaWRhdGU6IHt9LFxuICAgIH0sXG4gICAgYXN5bmMgKFxuICAgICAgY29udGV4dCxcbiAgICAgIHJlcXVlc3QsXG4gICAgICByZXNwb25zZVxuICAgICk6IFByb21pc2U8SU9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2U8YW55IHwgUmVzcG9uc2VFcnJvcj4+ID0+IHtcbiAgICAgIGNvbnN0IGNsaWVudCA9IGNvbnRleHQuY29yZS5vcGVuc2VhcmNoLmNsaWVudC5hc0N1cnJlbnRVc2VyO1xuICAgICAgdHJ5IHtcbiAgICAgICAgLy8gaWYgdGhlIGNhbGwgZG9lcyBub3QgdGhyb3cgYW55IGVycm9yLCB0aGVuIHRoZSBhZ2VudCBpcyBwcm9wZXJseSBjb25maWd1cmVkXG4gICAgICAgIGF3YWl0IGdldEFnZW50SWRCeUNvbmZpZyhjbGllbnQsIEFHRU5UX0NPTkZJR1MuUFBMX0FHRU5UKTtcbiAgICAgICAgcmV0dXJuIHJlc3BvbnNlLm9rKHsgYm9keTogeyBjb25maWd1cmVkOiB0cnVlIH0gfSk7XG4gICAgICB9IGNhdGNoIChlcnJvcikge1xuICAgICAgICByZXR1cm4gcmVzcG9uc2Uub2soeyBib2R5OiB7IGNvbmZpZ3VyZWQ6IGZhbHNlLCBlcnJvcjogZXJyb3IubWVzc2FnZSB9IH0pO1xuICAgICAgfVxuICAgIH1cbiAgKTtcblxuICByb3V0ZXIucG9zdChcbiAgICB7XG4gICAgICBwYXRoOiBRVUVSWV9BU1NJU1RfQVBJLkdFTkVSQVRFX1BQTCxcbiAgICAgIHZhbGlkYXRlOiB7XG4gICAgICAgIGJvZHk6IHNjaGVtYS5vYmplY3Qoe1xuICAgICAgICAgIGluZGV4OiBzY2hlbWEuc3RyaW5nKCksXG4gICAgICAgICAgcXVlc3Rpb246IHNjaGVtYS5zdHJpbmcoKSxcbiAgICAgICAgfSksXG4gICAgICB9LFxuICAgIH0sXG4gICAgYXN5bmMgKFxuICAgICAgY29udGV4dCxcbiAgICAgIHJlcXVlc3QsXG4gICAgICByZXNwb25zZVxuICAgICk6IFByb21pc2U8SU9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2U8YW55IHwgUmVzcG9uc2VFcnJvcj4+ID0+IHtcbiAgICAgIGNvbnN0IGNsaWVudCA9IGNvbnRleHQuY29yZS5vcGVuc2VhcmNoLmNsaWVudC5hc0N1cnJlbnRVc2VyO1xuICAgICAgdHJ5IHtcbiAgICAgICAgY29uc3QgcHBsUmVxdWVzdCA9IGF3YWl0IHJlcXVlc3RXaXRoUmV0cnlBZ2VudFNlYXJjaCh7XG4gICAgICAgICAgY2xpZW50LFxuICAgICAgICAgIGNvbmZpZ05hbWU6IEFHRU5UX0NPTkZJR1MuUFBMX0FHRU5ULFxuICAgICAgICAgIGJvZHk6IHtcbiAgICAgICAgICAgIHBhcmFtZXRlcnM6IHtcbiAgICAgICAgICAgICAgaW5kZXg6IHJlcXVlc3QuYm9keS5pbmRleCxcbiAgICAgICAgICAgICAgcXVlc3Rpb246IHJlcXVlc3QuYm9keS5xdWVzdGlvbixcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSk7XG4gICAgICAgIGlmICghcHBsUmVxdWVzdC5ib2R5LmluZmVyZW5jZV9yZXN1bHRzWzBdLm91dHB1dFswXS5yZXN1bHQpXG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdHZW5lcmF0ZWQgUFBMIHF1ZXJ5IG5vdCBmb3VuZC4nKTtcbiAgICAgICAgY29uc3QgcmVzdWx0ID0gSlNPTi5wYXJzZShwcGxSZXF1ZXN0LmJvZHkuaW5mZXJlbmNlX3Jlc3VsdHNbMF0ub3V0cHV0WzBdLnJlc3VsdCkgYXMge1xuICAgICAgICAgIHBwbDogc3RyaW5nO1xuICAgICAgICAgIGV4ZWN1dGlvblJlc3VsdDogc3RyaW5nO1xuICAgICAgICB9O1xuICAgICAgICBjb25zdCBwcGwgPSByZXN1bHQucHBsXG4gICAgICAgICAgLnJlcGxhY2UoL1tcXHJcXG5dL2csICcgJylcbiAgICAgICAgICAudHJpbSgpXG4gICAgICAgICAgLnJlcGxhY2UoL0lTTk9UTlVMTC9nLCAnaXNub3RudWxsJykgLy8gaHR0cHM6Ly9naXRodWIuY29tL29wZW5zZWFyY2gtcHJvamVjdC9zcWwvaXNzdWVzLzI0MzFcbiAgICAgICAgICAucmVwbGFjZSgvYC9nLCAnJykgLy8gaHR0cHM6Ly9naXRodWIuY29tL29wZW5zZWFyY2gtcHJvamVjdC9kYXNoYm9hcmRzLW9ic2VydmFiaWxpdHkvaXNzdWVzLzUwOSwgaHR0cHM6Ly9naXRodWIuY29tL29wZW5zZWFyY2gtcHJvamVjdC9kYXNoYm9hcmRzLW9ic2VydmFiaWxpdHkvaXNzdWVzLzU1N1xuICAgICAgICAgIC5yZXBsYWNlKC9cXGJTUEFOXFwoL2csICdzcGFuKCcpOyAvLyBodHRwczovL2dpdGh1Yi5jb20vb3BlbnNlYXJjaC1wcm9qZWN0L2Rhc2hib2FyZHMtb2JzZXJ2YWJpbGl0eS9pc3N1ZXMvNzU5XG4gICAgICAgIHJldHVybiByZXNwb25zZS5vayh7IGJvZHk6IHBwbCB9KTtcbiAgICAgIH0gY2F0Y2ggKGVycm9yKSB7XG4gICAgICAgIGlmIChcbiAgICAgICAgICBpc1Jlc3BvbnNlRXJyb3IoZXJyb3IpICYmXG4gICAgICAgICAgZXJyb3Iuc3RhdHVzQ29kZSA9PT0gNDAwICYmXG4gICAgICAgICAgZXJyb3IuYm9keS5pbmNsdWRlcyhFUlJPUl9ERVRBSUxTLkdVQVJEUkFJTFNfVFJJR0dFUkVEKVxuICAgICAgICApIHtcbiAgICAgICAgICByZXR1cm4gcmVzcG9uc2UuYmFkUmVxdWVzdCh7IGJvZHk6IEVSUk9SX0RFVEFJTFMuR1VBUkRSQUlMU19UUklHR0VSRUQgfSk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHJlc3BvbnNlLmN1c3RvbSh7IHN0YXR1c0NvZGU6IGVycm9yLnN0YXR1c0NvZGUgfHwgNTAwLCBib2R5OiBlcnJvci5tZXNzYWdlIH0pO1xuICAgICAgfVxuICAgIH1cbiAgKTtcblxuICByb3V0ZXIucG9zdChcbiAgICB7XG4gICAgICBwYXRoOiBRVUVSWV9BU1NJU1RfQVBJLlNVTU1BUklaRSxcbiAgICAgIHZhbGlkYXRlOiB7XG4gICAgICAgIGJvZHk6IHNjaGVtYS5vYmplY3Qoe1xuICAgICAgICAgIGluZGV4OiBzY2hlbWEuc3RyaW5nKCksXG4gICAgICAgICAgcXVlc3Rpb246IHNjaGVtYS5zdHJpbmcoKSxcbiAgICAgICAgICBxdWVyeTogc2NoZW1hLm1heWJlKHNjaGVtYS5zdHJpbmcoKSksXG4gICAgICAgICAgcmVzcG9uc2U6IHNjaGVtYS5zdHJpbmcoKSxcbiAgICAgICAgICBpc0Vycm9yOiBzY2hlbWEuYm9vbGVhbigpLFxuICAgICAgICB9KSxcbiAgICAgIH0sXG4gICAgfSxcbiAgICBhc3luYyAoXG4gICAgICBjb250ZXh0LFxuICAgICAgcmVxdWVzdCxcbiAgICAgIHJlc3BvbnNlXG4gICAgKTogUHJvbWlzZTxJT3BlblNlYXJjaERhc2hib2FyZHNSZXNwb25zZTxhbnkgfCBSZXNwb25zZUVycm9yPj4gPT4ge1xuICAgICAgY29uc3QgY2xpZW50ID0gY29udGV4dC5jb3JlLm9wZW5zZWFyY2guY2xpZW50LmFzQ3VycmVudFVzZXI7XG4gICAgICBjb25zdCB7IGluZGV4LCBxdWVzdGlvbiwgcXVlcnksIHJlc3BvbnNlOiBfcmVzcG9uc2UsIGlzRXJyb3IgfSA9IHJlcXVlc3QuYm9keTtcbiAgICAgIGNvbnN0IHF1ZXJ5UmVzcG9uc2UgPSBKU09OLnN0cmluZ2lmeShfcmVzcG9uc2UpO1xuICAgICAgbGV0IHN1bW1hcnlSZXF1ZXN0O1xuXG4gICAgICB0cnkge1xuICAgICAgICBpZiAoIWlzRXJyb3IpIHtcbiAgICAgICAgICBzdW1tYXJ5UmVxdWVzdCA9IGF3YWl0IHJlcXVlc3RXaXRoUmV0cnlBZ2VudFNlYXJjaCh7XG4gICAgICAgICAgICBjbGllbnQsXG4gICAgICAgICAgICBjb25maWdOYW1lOiBBR0VOVF9DT05GSUdTLlJFU1BPTlNFX1NVTU1BUllfQUdFTlQsXG4gICAgICAgICAgICBib2R5OiB7XG4gICAgICAgICAgICAgIHBhcmFtZXRlcnM6IHsgaW5kZXgsIHF1ZXN0aW9uLCBxdWVyeSwgcmVzcG9uc2U6IHF1ZXJ5UmVzcG9uc2UgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSk7XG4gICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgY29uc3QgW21hcHBpbmdzLCBzYW1wbGVEb2NdID0gYXdhaXQgUHJvbWlzZS5hbGwoW1xuICAgICAgICAgICAgY2xpZW50LmluZGljZXMuZ2V0TWFwcGluZyh7IGluZGV4IH0pLFxuICAgICAgICAgICAgY2xpZW50LnNlYXJjaCh7IGluZGV4LCBzaXplOiAxIH0pLFxuICAgICAgICAgIF0pO1xuICAgICAgICAgIGNvbnN0IGZpZWxkcyA9IGdlbmVyYXRlRmllbGRDb250ZXh0KG1hcHBpbmdzLCBzYW1wbGVEb2MpO1xuICAgICAgICAgIHN1bW1hcnlSZXF1ZXN0ID0gYXdhaXQgcmVxdWVzdFdpdGhSZXRyeUFnZW50U2VhcmNoKHtcbiAgICAgICAgICAgIGNsaWVudCxcbiAgICAgICAgICAgIGNvbmZpZ05hbWU6IEFHRU5UX0NPTkZJR1MuRVJST1JfU1VNTUFSWV9BR0VOVCxcbiAgICAgICAgICAgIGJvZHk6IHtcbiAgICAgICAgICAgICAgcGFyYW1ldGVyczogeyBpbmRleCwgcXVlc3Rpb24sIHF1ZXJ5LCByZXNwb25zZTogcXVlcnlSZXNwb25zZSwgZmllbGRzIH0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0pO1xuICAgICAgICB9XG4gICAgICAgIGNvbnN0IHN1bW1hcnkgPSBzdW1tYXJ5UmVxdWVzdC5ib2R5LmluZmVyZW5jZV9yZXN1bHRzWzBdLm91dHB1dFswXS5yZXN1bHQ7XG4gICAgICAgIGlmICghc3VtbWFyeSkgdGhyb3cgbmV3IEVycm9yKCdHZW5lcmF0ZWQgc3VtbWFyeSBub3QgZm91bmQuJyk7XG4gICAgICAgIGNvbnN0IHN1Z2dlc3RlZFF1ZXN0aW9ucyA9IEFycmF5LmZyb20oXG4gICAgICAgICAgKHN1bW1hcnlSZXF1ZXN0LmJvZHkuaW5mZXJlbmNlX3Jlc3VsdHNbMF0ub3V0cHV0WzFdPy5yZXN1bHQgfHwgJycpLm1hdGNoQWxsKFxuICAgICAgICAgICAgLzxxdWVzdGlvbj4oKC58W1xcclxcbl0pKz8pPFxcL3F1ZXN0aW9uPi9nXG4gICAgICAgICAgKVxuICAgICAgICApLm1hcCgobSkgPT4gKG0gYXMgdW5rbm93bltdKVsxXSk7XG4gICAgICAgIHJldHVybiByZXNwb25zZS5vayh7XG4gICAgICAgICAgYm9keToge1xuICAgICAgICAgICAgc3VtbWFyeSxcbiAgICAgICAgICAgIHN1Z2dlc3RlZFF1ZXN0aW9ucyxcbiAgICAgICAgICB9LFxuICAgICAgICB9KTtcbiAgICAgIH0gY2F0Y2ggKGVycm9yKSB7XG4gICAgICAgIGlmIChcbiAgICAgICAgICBpc1Jlc3BvbnNlRXJyb3IoZXJyb3IpICYmXG4gICAgICAgICAgZXJyb3Iuc3RhdHVzQ29kZSA9PT0gNDAwICYmXG4gICAgICAgICAgZXJyb3IuYm9keS5pbmNsdWRlcyhFUlJPUl9ERVRBSUxTLkdVQVJEUkFJTFNfVFJJR0dFUkVEKVxuICAgICAgICApIHtcbiAgICAgICAgICByZXR1cm4gcmVzcG9uc2UuYmFkUmVxdWVzdCh7IGJvZHk6IEVSUk9SX0RFVEFJTFMuR1VBUkRSQUlMU19UUklHR0VSRUQgfSk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHJlc3BvbnNlLmN1c3RvbSh7IHN0YXR1c0NvZGU6IGVycm9yLnN0YXR1c0NvZGUgfHwgNTAwLCBib2R5OiBlcnJvci5tZXNzYWdlIH0pO1xuICAgICAgfVxuICAgIH1cbiAgKTtcbn1cbiJdLCJtYXBwaW5ncyI6Ijs7Ozs7O0FBS0EsSUFBQUEsYUFBQSxHQUFBQyxPQUFBO0FBTUEsSUFBQUMsT0FBQSxHQUFBRCxPQUFBO0FBQ0EsSUFBQUUsYUFBQSxHQUFBRixPQUFBO0FBQ0EsSUFBQUcsdUJBQUEsR0FBQUgsT0FBQTtBQUNBLElBQUFJLE9BQUEsR0FBQUosT0FBQTtBQUNBLElBQUFLLFVBQUEsR0FBQUwsT0FBQTtBQWZBO0FBQ0E7QUFDQTtBQUNBOztBQWNPLFNBQVNNLHlCQUF5QkEsQ0FBQ0MsTUFBZSxFQUFFO0VBQ3pEO0FBQ0Y7QUFDQTtFQUNFQSxNQUFNLENBQUNDLEdBQUcsQ0FDUjtJQUNFQyxJQUFJLEVBQUVDLDhCQUFnQixDQUFDQyxVQUFVO0lBQ2pDQyxRQUFRLEVBQUUsQ0FBQztFQUNiLENBQUMsRUFDRCxPQUNFQyxPQUFPLEVBQ1BDLE9BQU8sRUFDUEMsUUFBUSxLQUN3RDtJQUNoRSxNQUFNQyxNQUFNLEdBQUdILE9BQU8sQ0FBQ0ksSUFBSSxDQUFDQyxVQUFVLENBQUNGLE1BQU0sQ0FBQ0csYUFBYTtJQUMzRCxJQUFJO01BQ0Y7TUFDQSxNQUFNLElBQUFDLDBCQUFrQixFQUFDSixNQUFNLEVBQUVLLHdCQUFhLENBQUNDLFNBQVMsQ0FBQztNQUN6RCxPQUFPUCxRQUFRLENBQUNRLEVBQUUsQ0FBQztRQUFFQyxJQUFJLEVBQUU7VUFBRUMsVUFBVSxFQUFFO1FBQUs7TUFBRSxDQUFDLENBQUM7SUFDcEQsQ0FBQyxDQUFDLE9BQU9DLEtBQUssRUFBRTtNQUNkLE9BQU9YLFFBQVEsQ0FBQ1EsRUFBRSxDQUFDO1FBQUVDLElBQUksRUFBRTtVQUFFQyxVQUFVLEVBQUUsS0FBSztVQUFFQyxLQUFLLEVBQUVBLEtBQUssQ0FBQ0M7UUFBUTtNQUFFLENBQUMsQ0FBQztJQUMzRTtFQUNGLENBQ0YsQ0FBQztFQUVEcEIsTUFBTSxDQUFDcUIsSUFBSSxDQUNUO0lBQ0VuQixJQUFJLEVBQUVDLDhCQUFnQixDQUFDbUIsWUFBWTtJQUNuQ2pCLFFBQVEsRUFBRTtNQUNSWSxJQUFJLEVBQUVNLG9CQUFNLENBQUNDLE1BQU0sQ0FBQztRQUNsQkMsS0FBSyxFQUFFRixvQkFBTSxDQUFDRyxNQUFNLENBQUMsQ0FBQztRQUN0QkMsUUFBUSxFQUFFSixvQkFBTSxDQUFDRyxNQUFNLENBQUM7TUFDMUIsQ0FBQztJQUNIO0VBQ0YsQ0FBQyxFQUNELE9BQ0VwQixPQUFPLEVBQ1BDLE9BQU8sRUFDUEMsUUFBUSxLQUN3RDtJQUNoRSxNQUFNQyxNQUFNLEdBQUdILE9BQU8sQ0FBQ0ksSUFBSSxDQUFDQyxVQUFVLENBQUNGLE1BQU0sQ0FBQ0csYUFBYTtJQUMzRCxJQUFJO01BQ0YsTUFBTWdCLFVBQVUsR0FBRyxNQUFNLElBQUFDLG1DQUEyQixFQUFDO1FBQ25EcEIsTUFBTTtRQUNOcUIsVUFBVSxFQUFFaEIsd0JBQWEsQ0FBQ0MsU0FBUztRQUNuQ0UsSUFBSSxFQUFFO1VBQ0pjLFVBQVUsRUFBRTtZQUNWTixLQUFLLEVBQUVsQixPQUFPLENBQUNVLElBQUksQ0FBQ1EsS0FBSztZQUN6QkUsUUFBUSxFQUFFcEIsT0FBTyxDQUFDVSxJQUFJLENBQUNVO1VBQ3pCO1FBQ0Y7TUFDRixDQUFDLENBQUM7TUFDRixJQUFJLENBQUNDLFVBQVUsQ0FBQ1gsSUFBSSxDQUFDZSxpQkFBaUIsQ0FBQyxDQUFDLENBQUMsQ0FBQ0MsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDQyxNQUFNLEVBQ3hELE1BQU0sSUFBSUMsS0FBSyxDQUFDLGdDQUFnQyxDQUFDO01BQ25ELE1BQU1ELE1BQU0sR0FBR0UsSUFBSSxDQUFDQyxLQUFLLENBQUNULFVBQVUsQ0FBQ1gsSUFBSSxDQUFDZSxpQkFBaUIsQ0FBQyxDQUFDLENBQUMsQ0FBQ0MsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDQyxNQUFNLENBRzlFO01BQ0QsTUFBTUksR0FBRyxHQUFHSixNQUFNLENBQUNJLEdBQUcsQ0FDbkJDLE9BQU8sQ0FBQyxTQUFTLEVBQUUsR0FBRyxDQUFDLENBQ3ZCQyxJQUFJLENBQUMsQ0FBQyxDQUNORCxPQUFPLENBQUMsWUFBWSxFQUFFLFdBQVcsQ0FBQyxDQUFDO01BQUEsQ0FDbkNBLE9BQU8sQ0FBQyxJQUFJLEVBQUUsRUFBRSxDQUFDLENBQUM7TUFBQSxDQUNsQkEsT0FBTyxDQUFDLFdBQVcsRUFBRSxPQUFPLENBQUMsQ0FBQyxDQUFDO01BQ2xDLE9BQU8vQixRQUFRLENBQUNRLEVBQUUsQ0FBQztRQUFFQyxJQUFJLEVBQUVxQjtNQUFJLENBQUMsQ0FBQztJQUNuQyxDQUFDLENBQUMsT0FBT25CLEtBQUssRUFBRTtNQUNkLElBQ0UsSUFBQXNCLHVCQUFlLEVBQUN0QixLQUFLLENBQUMsSUFDdEJBLEtBQUssQ0FBQ3VCLFVBQVUsS0FBSyxHQUFHLElBQ3hCdkIsS0FBSyxDQUFDRixJQUFJLENBQUMwQixRQUFRLENBQUNDLDJCQUFhLENBQUNDLG9CQUFvQixDQUFDLEVBQ3ZEO1FBQ0EsT0FBT3JDLFFBQVEsQ0FBQ3NDLFVBQVUsQ0FBQztVQUFFN0IsSUFBSSxFQUFFMkIsMkJBQWEsQ0FBQ0M7UUFBcUIsQ0FBQyxDQUFDO01BQzFFO01BQ0EsT0FBT3JDLFFBQVEsQ0FBQ3VDLE1BQU0sQ0FBQztRQUFFTCxVQUFVLEVBQUV2QixLQUFLLENBQUN1QixVQUFVLElBQUksR0FBRztRQUFFekIsSUFBSSxFQUFFRSxLQUFLLENBQUNDO01BQVEsQ0FBQyxDQUFDO0lBQ3RGO0VBQ0YsQ0FDRixDQUFDO0VBRURwQixNQUFNLENBQUNxQixJQUFJLENBQ1Q7SUFDRW5CLElBQUksRUFBRUMsOEJBQWdCLENBQUM2QyxTQUFTO0lBQ2hDM0MsUUFBUSxFQUFFO01BQ1JZLElBQUksRUFBRU0sb0JBQU0sQ0FBQ0MsTUFBTSxDQUFDO1FBQ2xCQyxLQUFLLEVBQUVGLG9CQUFNLENBQUNHLE1BQU0sQ0FBQyxDQUFDO1FBQ3RCQyxRQUFRLEVBQUVKLG9CQUFNLENBQUNHLE1BQU0sQ0FBQyxDQUFDO1FBQ3pCdUIsS0FBSyxFQUFFMUIsb0JBQU0sQ0FBQzJCLEtBQUssQ0FBQzNCLG9CQUFNLENBQUNHLE1BQU0sQ0FBQyxDQUFDLENBQUM7UUFDcENsQixRQUFRLEVBQUVlLG9CQUFNLENBQUNHLE1BQU0sQ0FBQyxDQUFDO1FBQ3pCeUIsT0FBTyxFQUFFNUIsb0JBQU0sQ0FBQzZCLE9BQU8sQ0FBQztNQUMxQixDQUFDO0lBQ0g7RUFDRixDQUFDLEVBQ0QsT0FDRTlDLE9BQU8sRUFDUEMsT0FBTyxFQUNQQyxRQUFRLEtBQ3dEO0lBQ2hFLE1BQU1DLE1BQU0sR0FBR0gsT0FBTyxDQUFDSSxJQUFJLENBQUNDLFVBQVUsQ0FBQ0YsTUFBTSxDQUFDRyxhQUFhO0lBQzNELE1BQU07TUFBRWEsS0FBSztNQUFFRSxRQUFRO01BQUVzQixLQUFLO01BQUV6QyxRQUFRLEVBQUU2QyxTQUFTO01BQUVGO0lBQVEsQ0FBQyxHQUFHNUMsT0FBTyxDQUFDVSxJQUFJO0lBQzdFLE1BQU1xQyxhQUFhLEdBQUdsQixJQUFJLENBQUNtQixTQUFTLENBQUNGLFNBQVMsQ0FBQztJQUMvQyxJQUFJRyxjQUFjO0lBRWxCLElBQUk7TUFBQSxJQUFBQyxxQkFBQTtNQUNGLElBQUksQ0FBQ04sT0FBTyxFQUFFO1FBQ1pLLGNBQWMsR0FBRyxNQUFNLElBQUEzQixtQ0FBMkIsRUFBQztVQUNqRHBCLE1BQU07VUFDTnFCLFVBQVUsRUFBRWhCLHdCQUFhLENBQUM0QyxzQkFBc0I7VUFDaER6QyxJQUFJLEVBQUU7WUFDSmMsVUFBVSxFQUFFO2NBQUVOLEtBQUs7Y0FBRUUsUUFBUTtjQUFFc0IsS0FBSztjQUFFekMsUUFBUSxFQUFFOEM7WUFBYztVQUNoRTtRQUNGLENBQUMsQ0FBQztNQUNKLENBQUMsTUFBTTtRQUNMLE1BQU0sQ0FBQ0ssUUFBUSxFQUFFQyxTQUFTLENBQUMsR0FBRyxNQUFNQyxPQUFPLENBQUNDLEdBQUcsQ0FBQyxDQUM5Q3JELE1BQU0sQ0FBQ3NELE9BQU8sQ0FBQ0MsVUFBVSxDQUFDO1VBQUV2QztRQUFNLENBQUMsQ0FBQyxFQUNwQ2hCLE1BQU0sQ0FBQ3dELE1BQU0sQ0FBQztVQUFFeEMsS0FBSztVQUFFeUMsSUFBSSxFQUFFO1FBQUUsQ0FBQyxDQUFDLENBQ2xDLENBQUM7UUFDRixNQUFNQyxNQUFNLEdBQUcsSUFBQUMsNENBQW9CLEVBQUNULFFBQVEsRUFBRUMsU0FBUyxDQUFDO1FBQ3hESixjQUFjLEdBQUcsTUFBTSxJQUFBM0IsbUNBQTJCLEVBQUM7VUFDakRwQixNQUFNO1VBQ05xQixVQUFVLEVBQUVoQix3QkFBYSxDQUFDdUQsbUJBQW1CO1VBQzdDcEQsSUFBSSxFQUFFO1lBQ0pjLFVBQVUsRUFBRTtjQUFFTixLQUFLO2NBQUVFLFFBQVE7Y0FBRXNCLEtBQUs7Y0FBRXpDLFFBQVEsRUFBRThDLGFBQWE7Y0FBRWE7WUFBTztVQUN4RTtRQUNGLENBQUMsQ0FBQztNQUNKO01BQ0EsTUFBTUcsT0FBTyxHQUFHZCxjQUFjLENBQUN2QyxJQUFJLENBQUNlLGlCQUFpQixDQUFDLENBQUMsQ0FBQyxDQUFDQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUNDLE1BQU07TUFDekUsSUFBSSxDQUFDb0MsT0FBTyxFQUFFLE1BQU0sSUFBSW5DLEtBQUssQ0FBQyw4QkFBOEIsQ0FBQztNQUM3RCxNQUFNb0Msa0JBQWtCLEdBQUdDLEtBQUssQ0FBQ0MsSUFBSSxDQUNuQyxDQUFDLEVBQUFoQixxQkFBQSxHQUFBRCxjQUFjLENBQUN2QyxJQUFJLENBQUNlLGlCQUFpQixDQUFDLENBQUMsQ0FBQyxDQUFDQyxNQUFNLENBQUMsQ0FBQyxDQUFDLGNBQUF3QixxQkFBQSx1QkFBbERBLHFCQUFBLENBQW9EdkIsTUFBTSxLQUFJLEVBQUUsRUFBRXdDLFFBQVEsQ0FDekUsdUNBQ0YsQ0FDRixDQUFDLENBQUNDLEdBQUcsQ0FBRUMsQ0FBQyxJQUFNQSxDQUFDLENBQWUsQ0FBQyxDQUFDLENBQUM7TUFDakMsT0FBT3BFLFFBQVEsQ0FBQ1EsRUFBRSxDQUFDO1FBQ2pCQyxJQUFJLEVBQUU7VUFDSnFELE9BQU87VUFDUEM7UUFDRjtNQUNGLENBQUMsQ0FBQztJQUNKLENBQUMsQ0FBQyxPQUFPcEQsS0FBSyxFQUFFO01BQ2QsSUFDRSxJQUFBc0IsdUJBQWUsRUFBQ3RCLEtBQUssQ0FBQyxJQUN0QkEsS0FBSyxDQUFDdUIsVUFBVSxLQUFLLEdBQUcsSUFDeEJ2QixLQUFLLENBQUNGLElBQUksQ0FBQzBCLFFBQVEsQ0FBQ0MsMkJBQWEsQ0FBQ0Msb0JBQW9CLENBQUMsRUFDdkQ7UUFDQSxPQUFPckMsUUFBUSxDQUFDc0MsVUFBVSxDQUFDO1VBQUU3QixJQUFJLEVBQUUyQiwyQkFBYSxDQUFDQztRQUFxQixDQUFDLENBQUM7TUFDMUU7TUFDQSxPQUFPckMsUUFBUSxDQUFDdUMsTUFBTSxDQUFDO1FBQUVMLFVBQVUsRUFBRXZCLEtBQUssQ0FBQ3VCLFVBQVUsSUFBSSxHQUFHO1FBQUV6QixJQUFJLEVBQUVFLEtBQUssQ0FBQ0M7TUFBUSxDQUFDLENBQUM7SUFDdEY7RUFDRixDQUNGLENBQUM7QUFDSCJ9