"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.requestWithRetryAgentSearch = exports.getAgentIdByConfig = exports.agentIdMap = void 0;
var _errors = require("../../../../../../src/core/server/opensearch/client/errors");
var _query_assist = require("../../../../common/constants/query_assist");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const AGENT_REQUEST_OPTIONS = {
  /**
   * It is time-consuming for LLM to generate final answer
   * Give it a large timeout window
   */
  requestTimeout: 5 * 60 * 1000,
  /**
   * Do not retry
   */
  maxRetries: 0
};
const agentIdMap = exports.agentIdMap = {};
const getAgentIdByConfig = async (opensearchClient, configName) => {
  try {
    const response = await opensearchClient.transport.request({
      method: 'GET',
      path: `${_query_assist.ML_COMMONS_API_PREFIX}/config/${configName}`
    });
    if (!response || response.body.configuration.agent_id === undefined) {
      throw new Error('cannot find any agent by configuration: ' + configName);
    }
    return response.body.configuration.agent_id;
  } catch (error) {
    var _error$meta;
    const errorMessage = JSON.stringify((_error$meta = error.meta) === null || _error$meta === void 0 ? void 0 : _error$meta.body) || error;
    throw new Error(`Get agent '${configName}' failed, reason: ` + errorMessage);
  }
};
exports.getAgentIdByConfig = getAgentIdByConfig;
const requestWithRetryAgentSearch = async options => {
  const {
    client,
    configName,
    shouldRetryAgentSearch = true,
    body
  } = options;
  let retry = shouldRetryAgentSearch;
  if (!agentIdMap[configName]) {
    agentIdMap[configName] = await getAgentIdByConfig(client, configName);
    retry = false;
  }
  return client.transport.request({
    method: 'POST',
    path: `${_query_assist.ML_COMMONS_API_PREFIX}/agents/${agentIdMap[configName]}/_execute`,
    body
  }, AGENT_REQUEST_OPTIONS).catch(async error => {
    if (retry && (0, _errors.isResponseError)(error) && error.statusCode === 404) {
      agentIdMap[configName] = await getAgentIdByConfig(client, configName);
      return requestWithRetryAgentSearch({
        ...options,
        shouldRetryAgentSearch: false
      });
    }
    return Promise.reject(error);
  });
};
exports.requestWithRetryAgentSearch = requestWithRetryAgentSearch;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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