"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OpenIdAuthentication = void 0;
var fs = _interopRequireWildcard(require("fs"));
var _wreck = _interopRequireDefault(require("@hapi/wreck"));
var _http = _interopRequireDefault(require("http"));
var _https = _interopRequireDefault(require("https"));
var _security_cookie = require("../../../session/security_cookie");
var _routes = require("./routes");
var _authentication_type = require("../authentication_type");
var _helper = require("./helper");
var _object_properties_defined = require("../../../utils/object_properties_defined");
var _common = require("../../../../common");
var _cookie_splitter = require("../../../session/cookie_splitter");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && Object.prototype.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
class OpenIdAuthentication extends _authentication_type.AuthenticationType {
  constructor(config, sessionStorageFactory, router, esClient, core, logger) {
    var _this$config$openid, _this$config$openid2;
    super(config, sessionStorageFactory, router, esClient, core, logger);
    _defineProperty(this, "type", _common.AuthType.OPEN_ID);
    _defineProperty(this, "openIdAuthConfig", void 0);
    _defineProperty(this, "authHeaderName", void 0);
    _defineProperty(this, "openIdConnectUrl", void 0);
    _defineProperty(this, "wreckClient", void 0);
    _defineProperty(this, "wreckHttpsOption", {});
    _defineProperty(this, "redirectOIDCCapture", (request, toolkit) => {
      const nextUrl = this.generateNextUrl(request);
      const clearOldVersionCookie = (0, _security_cookie.clearOldVersionCookieValue)(this.config);
      return toolkit.redirected({
        location: `${this.coreSetup.http.basePath.serverBasePath}/auth/openid/captureUrlFragment?nextUrl=${nextUrl}`,
        'set-cookie': clearOldVersionCookie
      });
    });
    this.wreckClient = this.createWreckClient();
    this.openIdAuthConfig = {};
    this.authHeaderName = ((_this$config$openid = this.config.openid) === null || _this$config$openid === void 0 ? void 0 : _this$config$openid.header) || '';
    this.openIdAuthConfig.authHeaderName = this.authHeaderName;
    this.openIdConnectUrl = ((_this$config$openid2 = this.config.openid) === null || _this$config$openid2 === void 0 ? void 0 : _this$config$openid2.connect_url) || '';
    let scope = this.config.openid.scope;
    if (scope.indexOf('openid') < 0) {
      scope = `openid ${scope}`;
    }
    this.openIdAuthConfig.scope = scope;
  }
  async init() {
    try {
      const response = await this.wreckClient.get(this.openIdConnectUrl);
      const payload = JSON.parse(response.payload);
      this.openIdAuthConfig.authorizationEndpoint = payload.authorization_endpoint;
      this.openIdAuthConfig.tokenEndpoint = payload.token_endpoint;
      this.openIdAuthConfig.endSessionEndpoint = payload.end_session_endpoint || undefined;
      this.createExtraStorage();
      const routes = new _routes.OpenIdAuthRoutes(this.router, this.config, this.sessionStorageFactory, this.openIdAuthConfig, this.securityClient, this.coreSetup, this.wreckClient);
      routes.setupRoutes();
    } catch (error) {
      this.logger.error(error); // TODO: log more info
      throw new Error('Failed when trying to obtain the endpoints from your IdP');
    }
  }
  generateNextUrl(request) {
    const path = this.coreSetup.http.basePath.serverBasePath + (request.url.pathname || '/app/opensearch-dashboards');
    return escape(path);
  }
  createWreckClient() {
    var _this$config$openid3, _this$config$openid4, _this$config$openid5, _this$config$openid6, _this$config$openid7, _this$config$openid9;
    if ((_this$config$openid3 = this.config.openid) !== null && _this$config$openid3 !== void 0 && _this$config$openid3.root_ca) {
      this.wreckHttpsOption.ca = [fs.readFileSync(this.config.openid.root_ca)];
      this.logger.debug(`Using CA Cert: ${this.config.openid.root_ca}`);
    }
    if ((_this$config$openid4 = this.config.openid) !== null && _this$config$openid4 !== void 0 && _this$config$openid4.pfx) {
      // Use PFX or PKCS12 if provided
      this.logger.debug(`Using PFX or PKCS12: ${this.config.openid.pfx}`);
      this.wreckHttpsOption.pfx = [fs.readFileSync(this.config.openid.pfx)];
    } else if ((_this$config$openid5 = this.config.openid) !== null && _this$config$openid5 !== void 0 && _this$config$openid5.certificate && (_this$config$openid6 = this.config.openid) !== null && _this$config$openid6 !== void 0 && _this$config$openid6.private_key) {
      // Use 'certificate' and 'private_key' if provided
      this.logger.debug(`Using Certificate: ${this.config.openid.certificate}`);
      this.logger.debug(`Using Private Key: ${this.config.openid.private_key}`);
      this.wreckHttpsOption.cert = [fs.readFileSync(this.config.openid.certificate)];
      this.wreckHttpsOption.key = [fs.readFileSync(this.config.openid.private_key)];
    } else {
      this.logger.debug(`Client certificates not provided. Mutual TLS will not be used to obtain endpoints.`);
    }
    // Check if passphrase is provided, use it for 'pfx' and 'key'
    if (((_this$config$openid7 = this.config.openid) === null || _this$config$openid7 === void 0 ? void 0 : _this$config$openid7.passphrase) !== '') {
      var _this$config$openid8;
      this.logger.debug(`Passphrase not provided for private key and/or pfx.`);
      this.wreckHttpsOption.passphrase = (_this$config$openid8 = this.config.openid) === null || _this$config$openid8 === void 0 ? void 0 : _this$config$openid8.passphrase;
    }
    if (((_this$config$openid9 = this.config.openid) === null || _this$config$openid9 === void 0 ? void 0 : _this$config$openid9.verify_hostnames) === false) {
      this.logger.debug(`openId auth 'verify_hostnames' option is off.`);
      this.wreckHttpsOption.checkServerIdentity = (host, cert) => {
        return undefined;
      };
    }
    this.logger.info((0, _object_properties_defined.getObjectProperties)(this.wreckHttpsOption, 'WreckHttpsOptions'));
    if (Object.keys(this.wreckHttpsOption).length > 0) {
      return _wreck.default.defaults({
        agents: {
          http: new _http.default.Agent(),
          https: new _https.default.Agent(this.wreckHttpsOption),
          httpsAllowUnauthorized: new _https.default.Agent({
            rejectUnauthorized: false
          })
        }
      });
    } else {
      return _wreck.default;
    }
  }
  getWreckHttpsOptions() {
    return this.wreckHttpsOption;
  }
  createExtraStorage() {
    // @ts-ignore
    const hapiServer = this.sessionStorageFactory.asScoped({}).server;
    const extraCookiePrefix = this.config.openid.extra_storage.cookie_prefix;
    const extraCookieSettings = {
      isSecure: this.config.cookie.secure,
      isSameSite: this.config.cookie.isSameSite,
      password: this.config.cookie.password,
      domain: this.config.cookie.domain,
      path: this.coreSetup.http.basePath.serverBasePath || '/',
      clearInvalid: false,
      isHttpOnly: true,
      ignoreErrors: true,
      encoding: 'iron' // Same as hapi auth cookie
    };

    for (let i = 1; i <= this.config.openid.extra_storage.additional_cookies; i++) {
      hapiServer.states.add(extraCookiePrefix + i, extraCookieSettings);
    }
  }
  getExtraAuthStorageOptions() {
    // If we're here, we will always have the openid configuration
    return {
      cookiePrefix: this.config.openid.extra_storage.cookie_prefix,
      additionalCookies: this.config.openid.extra_storage.additional_cookies,
      logger: this.logger
    };
  }
  requestIncludesAuthInfo(request) {
    return request.headers.authorization ? true : false;
  }
  async getAdditionalAuthHeader(request) {
    return {};
  }
  getCookie(request, authInfo) {
    (0, _cookie_splitter.setExtraAuthStorage)(request, request.headers.authorization, this.getExtraAuthStorageOptions());
    return {
      username: authInfo.user_name,
      credentials: {
        authHeaderValueExtra: true
      },
      authType: this.type,
      expiryTime: Date.now() + this.config.session.ttl
    };
  }

  // OIDC expiry time is set by the IDP and refreshed via refreshTokens
  getKeepAliveExpiry(cookie, request) {
    return cookie.expiryTime;
  }

  // TODO: Add token expiration check here
  async isValidCookie(cookie, request) {
    var _cookie$credentials;
    if (cookie.authType !== this.type || !cookie.username || !cookie.expiryTime || !((_cookie$credentials = cookie.credentials) !== null && _cookie$credentials !== void 0 && _cookie$credentials.authHeaderValue) && !this.getExtraAuthStorageValue(request, cookie)) {
      return false;
    }
    if (cookie.expiryTime > Date.now()) {
      return true;
    }

    // need to renew id token
    if (cookie.credentials.refresh_token) {
      try {
        var _this$config$openid10, _this$config$openid11;
        const query = {
          grant_type: 'refresh_token',
          client_id: (_this$config$openid10 = this.config.openid) === null || _this$config$openid10 === void 0 ? void 0 : _this$config$openid10.client_id,
          client_secret: (_this$config$openid11 = this.config.openid) === null || _this$config$openid11 === void 0 ? void 0 : _this$config$openid11.client_secret,
          refresh_token: cookie.credentials.refresh_token
        };
        const refreshTokenResponse = await (0, _helper.callTokenEndpoint)(this.openIdAuthConfig.tokenEndpoint, query, this.wreckClient);

        // if no id_token from refresh token call, maybe the Idp doesn't allow refresh id_token
        if (refreshTokenResponse.idToken) {
          cookie.credentials = {
            authHeaderValueExtra: true,
            refresh_token: refreshTokenResponse.refreshToken
          };
          cookie.expiryTime = (0, _helper.getExpirationDate)(refreshTokenResponse);
          (0, _cookie_splitter.setExtraAuthStorage)(request, `Bearer ${refreshTokenResponse.idToken}`, this.getExtraAuthStorageOptions());
          return true;
        } else {
          return false;
        }
      } catch (error) {
        this.logger.error(error);
        return false;
      }
    } else {
      // no refresh token, and current token is expired
      return false;
    }
  }
  handleUnauthedRequest(request, response, toolkit) {
    if (this.isPageRequest(request)) {
      return this.redirectOIDCCapture(request, toolkit);
    } else {
      return response.unauthorized();
    }
  }
  getExtraAuthStorageValue(request, cookie) {
    var _cookie$credentials2;
    let extraValue = '';
    if (!((_cookie$credentials2 = cookie.credentials) !== null && _cookie$credentials2 !== void 0 && _cookie$credentials2.authHeaderValueExtra)) {
      return extraValue;
    }
    try {
      extraValue = (0, _cookie_splitter.getExtraAuthStorageValue)(request, this.getExtraAuthStorageOptions());
    } catch (error) {
      this.logger.info(error);
    }
    return extraValue;
  }
  buildAuthHeaderFromCookie(cookie, request) {
    var _cookie$credentials3;
    const header = {};
    if (cookie.credentials.authHeaderValueExtra) {
      try {
        const extraAuthStorageValue = this.getExtraAuthStorageValue(request, cookie);
        header.authorization = extraAuthStorageValue;
        return header;
      } catch (error) {
        this.logger.error(error);
        // TODO Re-throw?
        // throw error;
      }
    }

    const authHeaderValue = (_cookie$credentials3 = cookie.credentials) === null || _cookie$credentials3 === void 0 ? void 0 : _cookie$credentials3.authHeaderValue;
    if (authHeaderValue) {
      header.authorization = authHeaderValue;
    }
    return header;
  }
}
exports.OpenIdAuthentication = OpenIdAuthentication;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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