"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OpenIdAuthRoutes = void 0;
var _configSchema = require("@osd/config-schema");
var _cryptiles = require("@hapi/cryptiles");
var _querystring = require("querystring");
var _helper = require("./helper");
var _next_url = require("../../../utils/next_url");
var _common = require("../../../../common");
var _cookie_splitter = require("../../../session/cookie_splitter");
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
class OpenIdAuthRoutes {
  constructor(router, config, sessionStorageFactory, openIdAuthConfig, securityClient, core, wreckClient) {
    this.router = router;
    this.config = config;
    this.sessionStorageFactory = sessionStorageFactory;
    this.openIdAuthConfig = openIdAuthConfig;
    this.securityClient = securityClient;
    this.core = core;
    this.wreckClient = wreckClient;
  }
  redirectToLogin(request, response) {
    this.sessionStorageFactory.asScoped(request).clear();
    return response.redirected({
      headers: {
        location: `${this.core.http.basePath.serverBasePath}${_common.OPENID_AUTH_LOGIN}`
      }
    });
  }
  getExtraAuthStorageOptions(logger) {
    // If we're here, we will always have the openid configuration
    return {
      cookiePrefix: this.config.openid.extra_storage.cookie_prefix,
      additionalCookies: this.config.openid.extra_storage.additional_cookies,
      logger
    };
  }
  setupRoutes() {
    this.router.get({
      path: _common.OPENID_AUTH_LOGIN,
      validate: {
        query: _configSchema.schema.object({
          code: _configSchema.schema.maybe(_configSchema.schema.string()),
          nextUrl: _configSchema.schema.maybe(_configSchema.schema.string({
            validate: _next_url.validateNextUrl
          })),
          redirectHash: _configSchema.schema.maybe(_configSchema.schema.boolean()),
          state: _configSchema.schema.maybe(_configSchema.schema.string()),
          refresh: _configSchema.schema.maybe(_configSchema.schema.string())
        }, {
          unknowns: 'allow'
        })
      },
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      var _this$config$openid2, _this$config$openid3, _cookie$oidc2;
      // implementation refers to https://github.com/hapijs/bell/blob/master/lib/oauth.js
      // Sign-in initialization
      if (!request.query.code) {
        var _this$config$openid;
        const nonce = (0, _cryptiles.randomString)(OpenIdAuthRoutes.NONCE_LENGTH);
        const query = {
          client_id: (_this$config$openid = this.config.openid) === null || _this$config$openid === void 0 ? void 0 : _this$config$openid.client_id,
          response_type: _common.AUTH_RESPONSE_TYPE,
          redirect_uri: `${(0, _helper.getBaseRedirectUrl)(this.config, this.core, request)}${_common.OPENID_AUTH_LOGIN}`,
          state: nonce,
          scope: this.openIdAuthConfig.scope
        };
        (0, _helper.includeAdditionalParameters)(query, context, this.config);
        const queryString = (0, _querystring.stringify)(query);
        const location = `${this.openIdAuthConfig.authorizationEndpoint}?${queryString}`;
        const cookie = {
          oidc: {
            state: nonce,
            nextUrl: (0, _helper.getNextUrl)(this.config, this.core, request),
            redirectHash: request.query.redirectHash === 'true'
          },
          authType: _common.AuthType.OPEN_ID
        };
        this.sessionStorageFactory.asScoped(request).set(cookie);
        return response.redirected({
          headers: {
            location
          }
        });
      }

      // Authentication callback
      // validate state first
      let cookie;
      try {
        var _cookie$oidc;
        cookie = await this.sessionStorageFactory.asScoped(request).get();
        if (!cookie || !((_cookie$oidc = cookie.oidc) !== null && _cookie$oidc !== void 0 && _cookie$oidc.state) || cookie.oidc.state !== request.query.state) {
          return this.redirectToLogin(request, response);
        }
      } catch (error) {
        return this.redirectToLogin(request, response);
      }
      const nextUrl = cookie.oidc.nextUrl;
      const clientId = (_this$config$openid2 = this.config.openid) === null || _this$config$openid2 === void 0 ? void 0 : _this$config$openid2.client_id;
      const clientSecret = (_this$config$openid3 = this.config.openid) === null || _this$config$openid3 === void 0 ? void 0 : _this$config$openid3.client_secret;
      const redirectHash = ((_cookie$oidc2 = cookie.oidc) === null || _cookie$oidc2 === void 0 ? void 0 : _cookie$oidc2.redirectHash) || false;
      const query = {
        grant_type: _common.AUTH_GRANT_TYPE,
        code: request.query.code,
        redirect_uri: `${(0, _helper.getBaseRedirectUrl)(this.config, this.core, request)}${_common.OPENID_AUTH_LOGIN}`,
        client_id: clientId,
        client_secret: clientSecret
      };
      (0, _helper.includeAdditionalParameters)(query, context, this.config);
      try {
        var _this$config$openid4;
        const tokenResponse = await (0, _helper.callTokenEndpoint)(this.openIdAuthConfig.tokenEndpoint, query, this.wreckClient);
        const user = await this.securityClient.authenticateWithHeader(request, this.openIdAuthConfig.authHeaderName, `Bearer ${tokenResponse.idToken}`);

        // set to cookie
        const sessionStorage = {
          username: user.username,
          credentials: {
            authHeaderValueExtra: true
          },
          authType: _common.AuthType.OPEN_ID,
          expiryTime: (0, _helper.getExpirationDate)(tokenResponse)
        };
        if ((_this$config$openid4 = this.config.openid) !== null && _this$config$openid4 !== void 0 && _this$config$openid4.refresh_tokens && tokenResponse.refreshToken) {
          Object.assign(sessionStorage.credentials, {
            refresh_token: tokenResponse.refreshToken
          });
        }
        (0, _cookie_splitter.setExtraAuthStorage)(request, `Bearer ${tokenResponse.idToken}`, this.getExtraAuthStorageOptions(context.security_plugin.logger));
        this.sessionStorageFactory.asScoped(request).set(sessionStorage);
        if (redirectHash) {
          return response.redirected({
            headers: {
              location: `${this.core.http.basePath.serverBasePath}/auth/openid/redirectUrlFragment?nextUrl=${escape(nextUrl)}`
            }
          });
        } else {
          return response.redirected({
            headers: {
              location: nextUrl
            }
          });
        }
      } catch (error) {
        context.security_plugin.logger.error(`OpenId authentication failed: ${error}`);
        if (error.toString().toLowerCase().includes('authentication exception')) {
          return response.unauthorized();
        } else {
          return this.redirectToLogin(request, response);
        }
      }
    });
    this.router.get({
      path: _common.OPENID_AUTH_LOGOUT,
      validate: false
    }, async (context, request, response) => {
      var _cookie$credentials, _this$config$openid5;
      const cookie = await this.sessionStorageFactory.asScoped(request).get();
      let tokenFromExtraStorage = '';
      const extraAuthStorageOptions = this.getExtraAuthStorageOptions(context.security_plugin.logger);
      if (cookie !== null && cookie !== void 0 && (_cookie$credentials = cookie.credentials) !== null && _cookie$credentials !== void 0 && _cookie$credentials.authHeaderValueExtra) {
        tokenFromExtraStorage = (0, _cookie_splitter.getExtraAuthStorageValue)(request, extraAuthStorageOptions);
      }
      (0, _cookie_splitter.clearSplitCookies)(request, extraAuthStorageOptions);
      this.sessionStorageFactory.asScoped(request).clear();

      // authHeaderValue is the bearer header, e.g. "Bearer <auth_token>"
      const token = tokenFromExtraStorage.length ? tokenFromExtraStorage.split(' ')[1] : cookie === null || cookie === void 0 ? void 0 : cookie.credentials.authHeaderValue.split(' ')[1]; // get auth token
      const nextUrl = (0, _helper.getBaseRedirectUrl)(this.config, this.core, request);
      const logoutQueryParams = {
        post_logout_redirect_uri: `${nextUrl}`,
        id_token_hint: token
      };
      const endSessionUrl = (0, _helper.composeLogoutUrl)((_this$config$openid5 = this.config.openid) === null || _this$config$openid5 === void 0 ? void 0 : _this$config$openid5.logout_url, this.openIdAuthConfig.endSessionEndpoint, logoutQueryParams);
      return response.redirected({
        headers: {
          location: endSessionUrl
        }
      });
    });

    // captureUrlFragment is the first route that will be invoked in the SP initiated login.
    // This route will execute the captureUrlFragment.js script.
    this.core.http.resources.register({
      path: '/auth/openid/captureUrlFragment',
      validate: {
        query: _configSchema.schema.object({
          nextUrl: _configSchema.schema.maybe(_configSchema.schema.string({
            validate: _next_url.validateNextUrl
          }))
        })
      },
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      this.sessionStorageFactory.asScoped(request).clear();
      const serverBasePath = this.core.http.basePath.serverBasePath;
      return response.renderHtml({
        body: `
          <!DOCTYPE html>
          <title>OSD OIDC Capture</title>
          <link rel="icon" href="data:,">
          <script src="${serverBasePath}/auth/openid/captureUrlFragment.js"></script>
        `
      });
    });

    // This script will store the URL Hash in browser's local storage.
    this.core.http.resources.register({
      path: '/auth/openid/captureUrlFragment.js',
      validate: false,
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      this.sessionStorageFactory.asScoped(request).clear();
      return response.renderJs({
        body: `let oidcHash=window.location.hash.toString();
                let redirectHash = false;
                if (oidcHash !== "") {
                  window.localStorage.removeItem('oidcHash');
                  window.localStorage.setItem('oidcHash', oidcHash);
                    redirectHash = true;
                }
                let params = new URLSearchParams(window.location.search);
                let nextUrl = params.get("nextUrl");
                finalUrl = "login?nextUrl=" + encodeURIComponent(nextUrl);
                finalUrl += "&redirectHash=" + encodeURIComponent(redirectHash);
                window.location.replace(finalUrl);
              `
      });
    });

    //  Once the User is authenticated the browser will be redirected to '/auth/openid/redirectUrlFragment'
    //  route, which will execute the redirectUrlFragment.js.
    this.core.http.resources.register({
      path: '/auth/openid/redirectUrlFragment',
      validate: {
        query: _configSchema.schema.object({
          nextUrl: _configSchema.schema.any()
        })
      },
      options: {
        authRequired: true
      }
    }, async (context, request, response) => {
      const serverBasePath = this.core.http.basePath.serverBasePath;
      return response.renderHtml({
        body: `
          <!DOCTYPE html>
          <title>OSD OpenID Success</title>
          <link rel="icon" href="data:,">
          <script src="${serverBasePath}/auth/openid/redirectUrlFragment.js"></script>
        `
      });
    });

    // This script will pop the Hash from local storage if it exists.
    // And forward the browser to the next url.
    this.core.http.resources.register({
      path: '/auth/openid/redirectUrlFragment.js',
      validate: false,
      options: {
        authRequired: true
      }
    }, async (context, request, response) => {
      return response.renderJs({
        body: `let oidcHash=window.localStorage.getItem('oidcHash');
                window.localStorage.removeItem('oidcHash');
                let params = new URLSearchParams(window.location.search);
                let nextUrl = params.get("nextUrl");
                finalUrl = nextUrl + oidcHash;
                window.location.replace(finalUrl);
              `
      });
    });
  }
}
exports.OpenIdAuthRoutes = OpenIdAuthRoutes;
_defineProperty(OpenIdAuthRoutes, "NONCE_LENGTH", 22);
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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