"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SamlAuthRoutes = void 0;
var _configSchema = require("@osd/config-schema");
var _next_url = require("../../../utils/next_url");
var _common = require("../../../../common");
var _cookie_splitter = require("../../../session/cookie_splitter");
/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */

class SamlAuthRoutes {
  constructor(router,
  // @ts-ignore: unused variable
  config, sessionStorageFactory, securityClient, coreSetup) {
    this.router = router;
    this.config = config;
    this.sessionStorageFactory = sessionStorageFactory;
    this.securityClient = securityClient;
    this.coreSetup = coreSetup;
  }
  getExtraAuthStorageOptions(logger) {
    // If we're here, we will always have the openid configuration
    return {
      cookiePrefix: this.config.saml.extra_storage.cookie_prefix,
      additionalCookies: this.config.saml.extra_storage.additional_cookies,
      logger
    };
  }
  setupRoutes() {
    this.router.get({
      path: _common.SAML_AUTH_LOGIN,
      validate: {
        query: _configSchema.schema.object({
          nextUrl: _configSchema.schema.maybe(_configSchema.schema.string({
            validate: _next_url.validateNextUrl
          })),
          redirectHash: _configSchema.schema.string()
        })
      },
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      if (request.auth.isAuthenticated) {
        return response.redirected({
          headers: {
            location: `${this.coreSetup.http.basePath.serverBasePath}/app/opensearch-dashboards`
          }
        });
      }
      try {
        const samlHeader = await this.securityClient.getSamlHeader(request);
        // const { nextUrl = '/' } = request.query;
        const cookie = {
          saml: {
            nextUrl: request.query.nextUrl,
            requestId: samlHeader.requestId,
            redirectHash: request.query.redirectHash === 'true'
          }
        };
        this.sessionStorageFactory.asScoped(request).set(cookie);
        return response.redirected({
          headers: {
            location: samlHeader.location
          }
        });
      } catch (error) {
        context.security_plugin.logger.error(`Failed to get saml header: ${error}`);
        return response.internalError(); // TODO: redirect to error page?
      }
    });

    this.router.post({
      path: `/_opendistro/_security/saml/acs`,
      validate: {
        body: _configSchema.schema.any()
      },
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      let requestId = '';
      let nextUrl = '/';
      let redirectHash = false;
      try {
        const cookie = await this.sessionStorageFactory.asScoped(request).get();
        if (cookie) {
          var _cookie$saml, _cookie$saml2, _cookie$saml3;
          requestId = ((_cookie$saml = cookie.saml) === null || _cookie$saml === void 0 ? void 0 : _cookie$saml.requestId) || '';
          nextUrl = ((_cookie$saml2 = cookie.saml) === null || _cookie$saml2 === void 0 ? void 0 : _cookie$saml2.nextUrl) || `${this.coreSetup.http.basePath.serverBasePath}/app/opensearch-dashboards`;
          redirectHash = ((_cookie$saml3 = cookie.saml) === null || _cookie$saml3 === void 0 ? void 0 : _cookie$saml3.redirectHash) || false;
        }
        if (!requestId) {
          return response.badRequest({
            body: 'Invalid requestId'
          });
        }
      } catch (error) {
        context.security_plugin.logger.error(`Failed to parse cookie: ${error}`);
        return response.badRequest();
      }
      try {
        const credentials = await this.securityClient.authToken({
          requestId,
          samlResponse: request.body.SAMLResponse,
          acsEndpoint: undefined,
          authRequestType: _common.AuthType.SAML
        });
        const user = await this.securityClient.authenticateWithHeader(request, 'authorization', credentials.authorization);
        let expiryTime = Date.now() + this.config.session.ttl;
        const [headerEncoded, payloadEncoded, signature] = credentials.authorization.split('.');
        if (!payloadEncoded) {
          context.security_plugin.logger.error('JWT token payload not found');
        }
        const tokenPayload = JSON.parse(Buffer.from(payloadEncoded, 'base64').toString());
        if (tokenPayload.exp) {
          expiryTime = parseInt(tokenPayload.exp, 10) * 1000;
        }
        const cookie = {
          username: user.username,
          credentials: {
            authHeaderValueExtra: true
          },
          authType: _common.AuthType.SAML,
          // TODO: create constant
          expiryTime
        };
        (0, _cookie_splitter.setExtraAuthStorage)(request, credentials.authorization, this.getExtraAuthStorageOptions(context.security_plugin.logger));
        this.sessionStorageFactory.asScoped(request).set(cookie);
        if (redirectHash) {
          return response.redirected({
            headers: {
              location: `${this.coreSetup.http.basePath.serverBasePath}/auth/saml/redirectUrlFragment?nextUrl=${escape(nextUrl)}`
            }
          });
        } else {
          return response.redirected({
            headers: {
              location: nextUrl
            }
          });
        }
      } catch (error) {
        context.security_plugin.logger.error(`SAML SP initiated authentication workflow failed: ${error}`);
      }
      return response.internalError();
    });
    this.router.post({
      path: `/_opendistro/_security/saml/acs/idpinitiated`,
      validate: {
        body: _configSchema.schema.any()
      },
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      const acsEndpoint = `${this.coreSetup.http.basePath.serverBasePath}/_opendistro/_security/saml/acs/idpinitiated`;
      try {
        const credentials = await this.securityClient.authToken({
          requestId: undefined,
          samlResponse: request.body.SAMLResponse,
          acsEndpoint,
          authRequestType: _common.AuthType.SAML
        });
        const user = await this.securityClient.authenticateWithHeader(request, 'authorization', credentials.authorization);
        let expiryTime = Date.now() + this.config.session.ttl;
        const [headerEncoded, payloadEncoded, signature] = credentials.authorization.split('.');
        if (!payloadEncoded) {
          context.security_plugin.logger.error('JWT token payload not found');
        }
        const tokenPayload = JSON.parse(Buffer.from(payloadEncoded, 'base64').toString());
        if (tokenPayload.exp) {
          expiryTime = parseInt(tokenPayload.exp, 10) * 1000;
        }
        const cookie = {
          username: user.username,
          credentials: {
            authHeaderValueExtra: true
          },
          authType: _common.AuthType.SAML,
          // TODO: create constant
          expiryTime
        };
        (0, _cookie_splitter.setExtraAuthStorage)(request, credentials.authorization, this.getExtraAuthStorageOptions(context.security_plugin.logger));
        this.sessionStorageFactory.asScoped(request).set(cookie);
        return response.redirected({
          headers: {
            location: `${this.coreSetup.http.basePath.serverBasePath}/app/opensearch-dashboards`
          }
        });
      } catch (error) {
        context.security_plugin.logger.error(`SAML IDP initiated authentication workflow failed: ${error}`);
      }
      return response.internalError();
    });

    // captureUrlFragment is the first route that will be invoked in the SP initiated login.
    // This route will execute the captureUrlFragment.js script.
    this.coreSetup.http.resources.register({
      path: '/auth/saml/captureUrlFragment',
      validate: {
        query: _configSchema.schema.object({
          nextUrl: _configSchema.schema.maybe(_configSchema.schema.string({
            validate: _next_url.validateNextUrl
          }))
        })
      },
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      this.sessionStorageFactory.asScoped(request).clear();
      const serverBasePath = this.coreSetup.http.basePath.serverBasePath;
      return response.renderHtml({
        body: `
            <!DOCTYPE html>
            <title>OSD SAML Capture</title>
            <link rel="icon" href="data:,">
            <script src="${serverBasePath}/auth/saml/captureUrlFragment.js"></script>
          `
      });
    });

    // This script will store the URL Hash in browser's local storage.
    this.coreSetup.http.resources.register({
      path: '/auth/saml/captureUrlFragment.js',
      validate: false,
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      this.sessionStorageFactory.asScoped(request).clear();
      return response.renderJs({
        body: `let samlHash=window.location.hash.toString();
                 let redirectHash = false;
                 if (samlHash !== "") {
                    window.localStorage.removeItem('samlHash');
                    window.localStorage.setItem('samlHash', samlHash);
                     redirectHash = true;
                  }
                 let params = new URLSearchParams(window.location.search);
                 let nextUrl = params.get("nextUrl");
                 finalUrl = "login?nextUrl=" + encodeURIComponent(nextUrl);
                 finalUrl += "&redirectHash=" + encodeURIComponent(redirectHash);
                 window.location.replace(finalUrl);
                `
      });
    });

    //  Once the User is authenticated via the '_opendistro/_security/saml/acs' route,
    //  the browser will be redirected to '/auth/saml/redirectUrlFragment' route,
    //  which will execute the redirectUrlFragment.js.
    this.coreSetup.http.resources.register({
      path: '/auth/saml/redirectUrlFragment',
      validate: {
        query: _configSchema.schema.object({
          nextUrl: _configSchema.schema.any()
        })
      },
      options: {
        authRequired: true
      }
    }, async (context, request, response) => {
      const serverBasePath = this.coreSetup.http.basePath.serverBasePath;
      return response.renderHtml({
        body: `
            <!DOCTYPE html>
            <title>OSD SAML Success</title>
            <link rel="icon" href="data:,">
            <script src="${serverBasePath}/auth/saml/redirectUrlFragment.js"></script>
          `
      });
    });

    // This script will pop the Hash from local storage if it exists.
    // And forward the browser to the next url.
    this.coreSetup.http.resources.register({
      path: '/auth/saml/redirectUrlFragment.js',
      validate: false,
      options: {
        authRequired: true
      }
    }, async (context, request, response) => {
      return response.renderJs({
        body: `let samlHash=window.localStorage.getItem('samlHash');
                 window.localStorage.removeItem('samlHash');
                 let params = new URLSearchParams(window.location.search);
                 let nextUrl = params.get("nextUrl");
                 finalUrl = nextUrl + samlHash;
                 window.location.replace(finalUrl);
                `
      });
    });
    this.router.get({
      path: _common.SAML_AUTH_LOGOUT,
      validate: false
    }, async (context, request, response) => {
      try {
        const authInfo = await this.securityClient.authinfo(request);
        await (0, _cookie_splitter.clearSplitCookies)(request, this.getExtraAuthStorageOptions(context.security_plugin.logger));
        this.sessionStorageFactory.asScoped(request).clear();
        // TODO: need a default logout page
        const redirectUrl = authInfo.sso_logout_url || this.coreSetup.http.basePath.serverBasePath || '/';
        return response.redirected({
          headers: {
            location: redirectUrl
          }
        });
      } catch (error) {
        context.security_plugin.logger.error(`SAML logout failed: ${error}`);
        return response.badRequest();
      }
    });
  }
}
exports.SamlAuthRoutes = SamlAuthRoutes;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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