"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ObservabilityPlugin = void 0;
var _migrations = require("./adaptors/integrations/migrations");
var _opensearch_observability_plugin = require("./adaptors/opensearch_observability_plugin");
var _ppl_plugin = require("./adaptors/ppl_plugin");
var _ppl_parser = require("./parsers/ppl_parser");
var _index = require("./routes/index");
var _observability_saved_object = require("./saved_objects/observability_saved_object");
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
class ObservabilityPlugin {
  constructor(initializerContext) {
    this.initializerContext = initializerContext;
    _defineProperty(this, "logger", void 0);
    this.logger = initializerContext.logger.get();
  }
  async setup(core, deps) {
    const {
      assistantDashboards,
      dataSource
    } = deps;
    this.logger.debug('Observability: Setup');
    const router = core.http.createRouter();
    const dataSourceEnabled = !!dataSource;
    const openSearchObservabilityClient = core.opensearch.legacy.createClient('opensearch_observability', {
      plugins: [_ppl_plugin.PPLPlugin, _opensearch_observability_plugin.OpenSearchObservabilityPlugin]
    });
    if (dataSourceEnabled) {
      dataSource.registerCustomApiSchema(_ppl_plugin.PPLPlugin);
      dataSource.registerCustomApiSchema(_opensearch_observability_plugin.OpenSearchObservabilityPlugin);
    }
    // @ts-ignore
    core.http.registerRouteHandlerContext('observability_plugin', (_context, _request) => {
      return {
        logger: this.logger,
        observabilityClient: openSearchObservabilityClient
      };
    });
    const obsPanelType = {
      name: 'observability-panel',
      hidden: false,
      namespaceType: 'single',
      mappings: {
        dynamic: false,
        properties: {
          title: {
            type: 'text'
          },
          description: {
            type: 'text'
          }
        }
      },
      management: {
        importableAndExportable: true,
        getInAppUrl() {
          return {
            path: `/app/management/observability/settings`,
            uiCapabilitiesPath: 'advancedSettings.show'
          };
        },
        getTitle(obj) {
          return `Observability Settings [${obj.id}]`;
        }
      },
      migrations: {
        '3.0.0': doc => ({
          ...doc,
          description: ''
        }),
        '3.0.1': doc => ({
          ...doc,
          description: 'Some Description Text'
        }),
        '3.0.2': doc => ({
          ...doc,
          dateCreated: parseInt(doc.dateCreated || '0', 10)
        })
      }
    };
    const integrationInstanceType = {
      name: 'integration-instance',
      hidden: false,
      namespaceType: 'single',
      management: {
        importableAndExportable: true,
        getInAppUrl(obj) {
          return {
            path: `/app/integrations#/installed/${obj.id}`,
            uiCapabilitiesPath: 'advancedSettings.show'
          };
        },
        getTitle(obj) {
          return obj.attributes.name;
        }
      },
      mappings: {
        dynamic: false,
        properties: {
          name: {
            type: 'text'
          },
          templateName: {
            type: 'text'
          },
          dataSource: {
            type: 'text'
          },
          creationDate: {
            type: 'date'
          },
          assets: {
            type: 'nested'
          }
        }
      }
    };
    const integrationTemplateType = {
      name: 'integration-template',
      hidden: false,
      namespaceType: 'single',
      management: {
        importableAndExportable: true,
        getInAppUrl(obj) {
          return {
            path: `/app/integrations#/available/${obj.attributes.name}`,
            uiCapabilitiesPath: 'advancedSettings.show'
          };
        },
        getTitle(obj) {
          var _obj$attributes$displ;
          return (_obj$attributes$displ = obj.attributes.displayName) !== null && _obj$attributes$displ !== void 0 ? _obj$attributes$displ : obj.attributes.name;
        }
      },
      mappings: {
        dynamic: false,
        properties: {
          name: {
            type: 'text'
          },
          version: {
            type: 'text'
          },
          displayName: {
            type: 'text'
          },
          license: {
            type: 'text'
          },
          type: {
            type: 'text'
          },
          labels: {
            type: 'text'
          },
          author: {
            type: 'text'
          },
          description: {
            type: 'text'
          },
          sourceUrl: {
            type: 'text'
          },
          statics: {
            type: 'nested'
          },
          components: {
            type: 'nested'
          },
          assets: {
            type: 'nested'
          },
          sampleData: {
            type: 'nested'
          }
        }
      },
      migrations: {
        '3.0.0': _migrations.migrateV1IntegrationToV2Integration
      }
    };
    core.savedObjects.registerType(obsPanelType);
    core.savedObjects.registerType(integrationInstanceType);
    core.savedObjects.registerType(integrationTemplateType);

    // Register server side APIs
    (0, _index.setupRoutes)({
      router,
      client: openSearchObservabilityClient,
      dataSourceEnabled
    });
    core.savedObjects.registerType(_observability_saved_object.visualizationSavedObject);
    core.savedObjects.registerType(_observability_saved_object.searchSavedObject);
    core.capabilities.registerProvider(() => ({
      observability: {
        show: true
      }
    }));
    assistantDashboards === null || assistantDashboards === void 0 || assistantDashboards.registerMessageParser(_ppl_parser.PPLParsers);
    return {};
  }
  start(_core) {
    this.logger.debug('Observability: Started');
    return {};
  }
  stop() {}
}
exports.ObservabilityPlugin = ObservabilityPlugin;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJuYW1lcyI6WyJfbWlncmF0aW9ucyIsInJlcXVpcmUiLCJfb3BlbnNlYXJjaF9vYnNlcnZhYmlsaXR5X3BsdWdpbiIsIl9wcGxfcGx1Z2luIiwiX3BwbF9wYXJzZXIiLCJfaW5kZXgiLCJfb2JzZXJ2YWJpbGl0eV9zYXZlZF9vYmplY3QiLCJfZGVmaW5lUHJvcGVydHkiLCJvYmoiLCJrZXkiLCJ2YWx1ZSIsIl90b1Byb3BlcnR5S2V5IiwiT2JqZWN0IiwiZGVmaW5lUHJvcGVydHkiLCJlbnVtZXJhYmxlIiwiY29uZmlndXJhYmxlIiwid3JpdGFibGUiLCJhcmciLCJfdG9QcmltaXRpdmUiLCJTdHJpbmciLCJpbnB1dCIsImhpbnQiLCJwcmltIiwiU3ltYm9sIiwidG9QcmltaXRpdmUiLCJ1bmRlZmluZWQiLCJyZXMiLCJjYWxsIiwiVHlwZUVycm9yIiwiTnVtYmVyIiwiT2JzZXJ2YWJpbGl0eVBsdWdpbiIsImNvbnN0cnVjdG9yIiwiaW5pdGlhbGl6ZXJDb250ZXh0IiwibG9nZ2VyIiwiZ2V0Iiwic2V0dXAiLCJjb3JlIiwiZGVwcyIsImFzc2lzdGFudERhc2hib2FyZHMiLCJkYXRhU291cmNlIiwiZGVidWciLCJyb3V0ZXIiLCJodHRwIiwiY3JlYXRlUm91dGVyIiwiZGF0YVNvdXJjZUVuYWJsZWQiLCJvcGVuU2VhcmNoT2JzZXJ2YWJpbGl0eUNsaWVudCIsIm9wZW5zZWFyY2giLCJsZWdhY3kiLCJjcmVhdGVDbGllbnQiLCJwbHVnaW5zIiwiUFBMUGx1Z2luIiwiT3BlblNlYXJjaE9ic2VydmFiaWxpdHlQbHVnaW4iLCJyZWdpc3RlckN1c3RvbUFwaVNjaGVtYSIsInJlZ2lzdGVyUm91dGVIYW5kbGVyQ29udGV4dCIsIl9jb250ZXh0IiwiX3JlcXVlc3QiLCJvYnNlcnZhYmlsaXR5Q2xpZW50Iiwib2JzUGFuZWxUeXBlIiwibmFtZSIsImhpZGRlbiIsIm5hbWVzcGFjZVR5cGUiLCJtYXBwaW5ncyIsImR5bmFtaWMiLCJwcm9wZXJ0aWVzIiwidGl0bGUiLCJ0eXBlIiwiZGVzY3JpcHRpb24iLCJtYW5hZ2VtZW50IiwiaW1wb3J0YWJsZUFuZEV4cG9ydGFibGUiLCJnZXRJbkFwcFVybCIsInBhdGgiLCJ1aUNhcGFiaWxpdGllc1BhdGgiLCJnZXRUaXRsZSIsImlkIiwibWlncmF0aW9ucyIsImRvYyIsImRhdGVDcmVhdGVkIiwicGFyc2VJbnQiLCJpbnRlZ3JhdGlvbkluc3RhbmNlVHlwZSIsImF0dHJpYnV0ZXMiLCJ0ZW1wbGF0ZU5hbWUiLCJjcmVhdGlvbkRhdGUiLCJhc3NldHMiLCJpbnRlZ3JhdGlvblRlbXBsYXRlVHlwZSIsIl9vYmokYXR0cmlidXRlcyRkaXNwbCIsImRpc3BsYXlOYW1lIiwidmVyc2lvbiIsImxpY2Vuc2UiLCJsYWJlbHMiLCJhdXRob3IiLCJzb3VyY2VVcmwiLCJzdGF0aWNzIiwiY29tcG9uZW50cyIsInNhbXBsZURhdGEiLCJtaWdyYXRlVjFJbnRlZ3JhdGlvblRvVjJJbnRlZ3JhdGlvbiIsInNhdmVkT2JqZWN0cyIsInJlZ2lzdGVyVHlwZSIsInNldHVwUm91dGVzIiwiY2xpZW50IiwidmlzdWFsaXphdGlvblNhdmVkT2JqZWN0Iiwic2VhcmNoU2F2ZWRPYmplY3QiLCJjYXBhYmlsaXRpZXMiLCJyZWdpc3RlclByb3ZpZGVyIiwib2JzZXJ2YWJpbGl0eSIsInNob3ciLCJyZWdpc3Rlck1lc3NhZ2VQYXJzZXIiLCJQUExQYXJzZXJzIiwic3RhcnQiLCJfY29yZSIsInN0b3AiLCJleHBvcnRzIl0sInNvdXJjZXMiOlsicGx1Z2luLnRzIl0sInNvdXJjZXNDb250ZW50IjpbIi8qXG4gKiBDb3B5cmlnaHQgT3BlblNlYXJjaCBDb250cmlidXRvcnNcbiAqIFNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wXG4gKi9cblxuaW1wb3J0IHtcbiAgQ29yZVNldHVwLFxuICBDb3JlU3RhcnQsXG4gIElMZWdhY3lDbHVzdGVyQ2xpZW50LFxuICBMb2dnZXIsXG4gIFBsdWdpbixcbiAgUGx1Z2luSW5pdGlhbGl6ZXJDb250ZXh0LFxuICBTYXZlZE9iamVjdCxcbiAgU2F2ZWRPYmplY3RzVHlwZSxcbn0gZnJvbSAnLi4vLi4vLi4vc3JjL2NvcmUvc2VydmVyJztcbmltcG9ydCB7IERhdGFTb3VyY2VQbHVnaW5TZXR1cCB9IGZyb20gJy4uLy4uLy4uL3NyYy9wbHVnaW5zL2RhdGFfc291cmNlL3NlcnZlci90eXBlcyc7XG5pbXBvcnQgeyBEYXRhU291cmNlTWFuYWdlbWVudFBsdWdpbiB9IGZyb20gJy4uLy4uLy4uL3NyYy9wbHVnaW5zL2RhdGFfc291cmNlX21hbmFnZW1lbnQvcHVibGljL3BsdWdpbic7XG5pbXBvcnQgeyBtaWdyYXRlVjFJbnRlZ3JhdGlvblRvVjJJbnRlZ3JhdGlvbiB9IGZyb20gJy4vYWRhcHRvcnMvaW50ZWdyYXRpb25zL21pZ3JhdGlvbnMnO1xuaW1wb3J0IHsgT3BlblNlYXJjaE9ic2VydmFiaWxpdHlQbHVnaW4gfSBmcm9tICcuL2FkYXB0b3JzL29wZW5zZWFyY2hfb2JzZXJ2YWJpbGl0eV9wbHVnaW4nO1xuaW1wb3J0IHsgUFBMUGx1Z2luIH0gZnJvbSAnLi9hZGFwdG9ycy9wcGxfcGx1Z2luJztcbmltcG9ydCB7IFBQTFBhcnNlcnMgfSBmcm9tICcuL3BhcnNlcnMvcHBsX3BhcnNlcic7XG5pbXBvcnQgeyBzZXR1cFJvdXRlcyB9IGZyb20gJy4vcm91dGVzL2luZGV4JztcbmltcG9ydCB7XG4gIHNlYXJjaFNhdmVkT2JqZWN0LFxuICB2aXN1YWxpemF0aW9uU2F2ZWRPYmplY3QsXG59IGZyb20gJy4vc2F2ZWRfb2JqZWN0cy9vYnNlcnZhYmlsaXR5X3NhdmVkX29iamVjdCc7XG5pbXBvcnQgeyBBc3Npc3RhbnRQbHVnaW5TZXR1cCwgT2JzZXJ2YWJpbGl0eVBsdWdpblNldHVwLCBPYnNlcnZhYmlsaXR5UGx1Z2luU3RhcnQgfSBmcm9tICcuL3R5cGVzJztcblxuZXhwb3J0IGludGVyZmFjZSBPYnNlcnZhYmlsaXR5UGx1Z2luU2V0dXBEZXBlbmRlbmNpZXMge1xuICBkYXRhU291cmNlTWFuYWdlbWVudDogUmV0dXJuVHlwZTxEYXRhU291cmNlTWFuYWdlbWVudFBsdWdpblsnc2V0dXAnXT47XG4gIGRhdGFTb3VyY2U6IERhdGFTb3VyY2VQbHVnaW5TZXR1cDtcbn1cblxuZXhwb3J0IGNsYXNzIE9ic2VydmFiaWxpdHlQbHVnaW5cbiAgaW1wbGVtZW50cyBQbHVnaW48T2JzZXJ2YWJpbGl0eVBsdWdpblNldHVwLCBPYnNlcnZhYmlsaXR5UGx1Z2luU3RhcnQ+IHtcbiAgcHJpdmF0ZSByZWFkb25seSBsb2dnZXI6IExvZ2dlcjtcblxuICBjb25zdHJ1Y3Rvcihwcml2YXRlIHJlYWRvbmx5IGluaXRpYWxpemVyQ29udGV4dDogUGx1Z2luSW5pdGlhbGl6ZXJDb250ZXh0KSB7XG4gICAgdGhpcy5sb2dnZXIgPSBpbml0aWFsaXplckNvbnRleHQubG9nZ2VyLmdldCgpO1xuICB9XG5cbiAgcHVibGljIGFzeW5jIHNldHVwKFxuICAgIGNvcmU6IENvcmVTZXR1cCxcbiAgICBkZXBzOiB7XG4gICAgICBhc3Npc3RhbnREYXNoYm9hcmRzPzogQXNzaXN0YW50UGx1Z2luU2V0dXA7XG4gICAgICBkYXRhU291cmNlOiBPYnNlcnZhYmlsaXR5UGx1Z2luU2V0dXBEZXBlbmRlbmNpZXM7XG4gICAgfVxuICApIHtcbiAgICBjb25zdCB7IGFzc2lzdGFudERhc2hib2FyZHMsIGRhdGFTb3VyY2UgfSA9IGRlcHM7XG4gICAgdGhpcy5sb2dnZXIuZGVidWcoJ09ic2VydmFiaWxpdHk6IFNldHVwJyk7XG4gICAgY29uc3Qgcm91dGVyID0gY29yZS5odHRwLmNyZWF0ZVJvdXRlcigpO1xuXG4gICAgY29uc3QgZGF0YVNvdXJjZUVuYWJsZWQgPSAhIWRhdGFTb3VyY2U7XG4gICAgY29uc3Qgb3BlblNlYXJjaE9ic2VydmFiaWxpdHlDbGllbnQ6IElMZWdhY3lDbHVzdGVyQ2xpZW50ID0gY29yZS5vcGVuc2VhcmNoLmxlZ2FjeS5jcmVhdGVDbGllbnQoXG4gICAgICAnb3BlbnNlYXJjaF9vYnNlcnZhYmlsaXR5JyxcbiAgICAgIHtcbiAgICAgICAgcGx1Z2luczogW1BQTFBsdWdpbiwgT3BlblNlYXJjaE9ic2VydmFiaWxpdHlQbHVnaW5dLFxuICAgICAgfVxuICAgICk7XG4gICAgaWYgKGRhdGFTb3VyY2VFbmFibGVkKSB7XG4gICAgICBkYXRhU291cmNlLnJlZ2lzdGVyQ3VzdG9tQXBpU2NoZW1hKFBQTFBsdWdpbik7XG4gICAgICBkYXRhU291cmNlLnJlZ2lzdGVyQ3VzdG9tQXBpU2NoZW1hKE9wZW5TZWFyY2hPYnNlcnZhYmlsaXR5UGx1Z2luKTtcbiAgICB9XG4gICAgLy8gQHRzLWlnbm9yZVxuICAgIGNvcmUuaHR0cC5yZWdpc3RlclJvdXRlSGFuZGxlckNvbnRleHQoJ29ic2VydmFiaWxpdHlfcGx1Z2luJywgKF9jb250ZXh0LCBfcmVxdWVzdCkgPT4ge1xuICAgICAgcmV0dXJuIHtcbiAgICAgICAgbG9nZ2VyOiB0aGlzLmxvZ2dlcixcbiAgICAgICAgb2JzZXJ2YWJpbGl0eUNsaWVudDogb3BlblNlYXJjaE9ic2VydmFiaWxpdHlDbGllbnQsXG4gICAgICB9O1xuICAgIH0pO1xuXG4gICAgY29uc3Qgb2JzUGFuZWxUeXBlOiBTYXZlZE9iamVjdHNUeXBlID0ge1xuICAgICAgbmFtZTogJ29ic2VydmFiaWxpdHktcGFuZWwnLFxuICAgICAgaGlkZGVuOiBmYWxzZSxcbiAgICAgIG5hbWVzcGFjZVR5cGU6ICdzaW5nbGUnLFxuICAgICAgbWFwcGluZ3M6IHtcbiAgICAgICAgZHluYW1pYzogZmFsc2UsXG4gICAgICAgIHByb3BlcnRpZXM6IHtcbiAgICAgICAgICB0aXRsZToge1xuICAgICAgICAgICAgdHlwZTogJ3RleHQnLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgZGVzY3JpcHRpb246IHtcbiAgICAgICAgICAgIHR5cGU6ICd0ZXh0JyxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICAgIG1hbmFnZW1lbnQ6IHtcbiAgICAgICAgaW1wb3J0YWJsZUFuZEV4cG9ydGFibGU6IHRydWUsXG4gICAgICAgIGdldEluQXBwVXJsKCkge1xuICAgICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICBwYXRoOiBgL2FwcC9tYW5hZ2VtZW50L29ic2VydmFiaWxpdHkvc2V0dGluZ3NgLFxuICAgICAgICAgICAgdWlDYXBhYmlsaXRpZXNQYXRoOiAnYWR2YW5jZWRTZXR0aW5ncy5zaG93JyxcbiAgICAgICAgICB9O1xuICAgICAgICB9LFxuICAgICAgICBnZXRUaXRsZShvYmopIHtcbiAgICAgICAgICByZXR1cm4gYE9ic2VydmFiaWxpdHkgU2V0dGluZ3MgWyR7b2JqLmlkfV1gO1xuICAgICAgICB9LFxuICAgICAgfSxcbiAgICAgIG1pZ3JhdGlvbnM6IHtcbiAgICAgICAgJzMuMC4wJzogKGRvYykgPT4gKHsgLi4uZG9jLCBkZXNjcmlwdGlvbjogJycgfSksXG4gICAgICAgICczLjAuMSc6IChkb2MpID0+ICh7IC4uLmRvYywgZGVzY3JpcHRpb246ICdTb21lIERlc2NyaXB0aW9uIFRleHQnIH0pLFxuICAgICAgICAnMy4wLjInOiAoZG9jKSA9PiAoe1xuICAgICAgICAgIC4uLmRvYyxcbiAgICAgICAgICBkYXRlQ3JlYXRlZDogcGFyc2VJbnQoKGRvYyBhcyB7IGRhdGVDcmVhdGVkPzogc3RyaW5nIH0pLmRhdGVDcmVhdGVkIHx8ICcwJywgMTApLFxuICAgICAgICB9KSxcbiAgICAgIH0sXG4gICAgfTtcblxuICAgIGNvbnN0IGludGVncmF0aW9uSW5zdGFuY2VUeXBlOiBTYXZlZE9iamVjdHNUeXBlID0ge1xuICAgICAgbmFtZTogJ2ludGVncmF0aW9uLWluc3RhbmNlJyxcbiAgICAgIGhpZGRlbjogZmFsc2UsXG4gICAgICBuYW1lc3BhY2VUeXBlOiAnc2luZ2xlJyxcbiAgICAgIG1hbmFnZW1lbnQ6IHtcbiAgICAgICAgaW1wb3J0YWJsZUFuZEV4cG9ydGFibGU6IHRydWUsXG4gICAgICAgIGdldEluQXBwVXJsKG9iajogU2F2ZWRPYmplY3Q8SW50ZWdyYXRpb25JbnN0YW5jZT4pIHtcbiAgICAgICAgICByZXR1cm4ge1xuICAgICAgICAgICAgcGF0aDogYC9hcHAvaW50ZWdyYXRpb25zIy9pbnN0YWxsZWQvJHtvYmouaWR9YCxcbiAgICAgICAgICAgIHVpQ2FwYWJpbGl0aWVzUGF0aDogJ2FkdmFuY2VkU2V0dGluZ3Muc2hvdycsXG4gICAgICAgICAgfTtcbiAgICAgICAgfSxcbiAgICAgICAgZ2V0VGl0bGUob2JqOiBTYXZlZE9iamVjdDxJbnRlZ3JhdGlvbkluc3RhbmNlPikge1xuICAgICAgICAgIHJldHVybiBvYmouYXR0cmlidXRlcy5uYW1lO1xuICAgICAgICB9LFxuICAgICAgfSxcbiAgICAgIG1hcHBpbmdzOiB7XG4gICAgICAgIGR5bmFtaWM6IGZhbHNlLFxuICAgICAgICBwcm9wZXJ0aWVzOiB7XG4gICAgICAgICAgbmFtZToge1xuICAgICAgICAgICAgdHlwZTogJ3RleHQnLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgdGVtcGxhdGVOYW1lOiB7XG4gICAgICAgICAgICB0eXBlOiAndGV4dCcsXG4gICAgICAgICAgfSxcbiAgICAgICAgICBkYXRhU291cmNlOiB7XG4gICAgICAgICAgICB0eXBlOiAndGV4dCcsXG4gICAgICAgICAgfSxcbiAgICAgICAgICBjcmVhdGlvbkRhdGU6IHtcbiAgICAgICAgICAgIHR5cGU6ICdkYXRlJyxcbiAgICAgICAgICB9LFxuICAgICAgICAgIGFzc2V0czoge1xuICAgICAgICAgICAgdHlwZTogJ25lc3RlZCcsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfTtcblxuICAgIGNvbnN0IGludGVncmF0aW9uVGVtcGxhdGVUeXBlOiBTYXZlZE9iamVjdHNUeXBlID0ge1xuICAgICAgbmFtZTogJ2ludGVncmF0aW9uLXRlbXBsYXRlJyxcbiAgICAgIGhpZGRlbjogZmFsc2UsXG4gICAgICBuYW1lc3BhY2VUeXBlOiAnc2luZ2xlJyxcbiAgICAgIG1hbmFnZW1lbnQ6IHtcbiAgICAgICAgaW1wb3J0YWJsZUFuZEV4cG9ydGFibGU6IHRydWUsXG4gICAgICAgIGdldEluQXBwVXJsKG9iajogU2F2ZWRPYmplY3Q8U2VyaWFsaXplZEludGVncmF0aW9uPikge1xuICAgICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICBwYXRoOiBgL2FwcC9pbnRlZ3JhdGlvbnMjL2F2YWlsYWJsZS8ke29iai5hdHRyaWJ1dGVzLm5hbWV9YCxcbiAgICAgICAgICAgIHVpQ2FwYWJpbGl0aWVzUGF0aDogJ2FkdmFuY2VkU2V0dGluZ3Muc2hvdycsXG4gICAgICAgICAgfTtcbiAgICAgICAgfSxcbiAgICAgICAgZ2V0VGl0bGUob2JqOiBTYXZlZE9iamVjdDxTZXJpYWxpemVkSW50ZWdyYXRpb24+KSB7XG4gICAgICAgICAgcmV0dXJuIG9iai5hdHRyaWJ1dGVzLmRpc3BsYXlOYW1lID8/IG9iai5hdHRyaWJ1dGVzLm5hbWU7XG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgICAgbWFwcGluZ3M6IHtcbiAgICAgICAgZHluYW1pYzogZmFsc2UsXG4gICAgICAgIHByb3BlcnRpZXM6IHtcbiAgICAgICAgICBuYW1lOiB7XG4gICAgICAgICAgICB0eXBlOiAndGV4dCcsXG4gICAgICAgICAgfSxcbiAgICAgICAgICB2ZXJzaW9uOiB7XG4gICAgICAgICAgICB0eXBlOiAndGV4dCcsXG4gICAgICAgICAgfSxcbiAgICAgICAgICBkaXNwbGF5TmFtZToge1xuICAgICAgICAgICAgdHlwZTogJ3RleHQnLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgbGljZW5zZToge1xuICAgICAgICAgICAgdHlwZTogJ3RleHQnLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgdHlwZToge1xuICAgICAgICAgICAgdHlwZTogJ3RleHQnLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgbGFiZWxzOiB7XG4gICAgICAgICAgICB0eXBlOiAndGV4dCcsXG4gICAgICAgICAgfSxcbiAgICAgICAgICBhdXRob3I6IHtcbiAgICAgICAgICAgIHR5cGU6ICd0ZXh0JyxcbiAgICAgICAgICB9LFxuICAgICAgICAgIGRlc2NyaXB0aW9uOiB7XG4gICAgICAgICAgICB0eXBlOiAndGV4dCcsXG4gICAgICAgICAgfSxcbiAgICAgICAgICBzb3VyY2VVcmw6IHtcbiAgICAgICAgICAgIHR5cGU6ICd0ZXh0JyxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHN0YXRpY3M6IHtcbiAgICAgICAgICAgIHR5cGU6ICduZXN0ZWQnLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgY29tcG9uZW50czoge1xuICAgICAgICAgICAgdHlwZTogJ25lc3RlZCcsXG4gICAgICAgICAgfSxcbiAgICAgICAgICBhc3NldHM6IHtcbiAgICAgICAgICAgIHR5cGU6ICduZXN0ZWQnLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgc2FtcGxlRGF0YToge1xuICAgICAgICAgICAgdHlwZTogJ25lc3RlZCcsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgICBtaWdyYXRpb25zOiB7XG4gICAgICAgICczLjAuMCc6IG1pZ3JhdGVWMUludGVncmF0aW9uVG9WMkludGVncmF0aW9uLFxuICAgICAgfSxcbiAgICB9O1xuXG4gICAgY29yZS5zYXZlZE9iamVjdHMucmVnaXN0ZXJUeXBlKG9ic1BhbmVsVHlwZSk7XG4gICAgY29yZS5zYXZlZE9iamVjdHMucmVnaXN0ZXJUeXBlKGludGVncmF0aW9uSW5zdGFuY2VUeXBlKTtcbiAgICBjb3JlLnNhdmVkT2JqZWN0cy5yZWdpc3RlclR5cGUoaW50ZWdyYXRpb25UZW1wbGF0ZVR5cGUpO1xuXG4gICAgLy8gUmVnaXN0ZXIgc2VydmVyIHNpZGUgQVBJc1xuICAgIHNldHVwUm91dGVzKHsgcm91dGVyLCBjbGllbnQ6IG9wZW5TZWFyY2hPYnNlcnZhYmlsaXR5Q2xpZW50LCBkYXRhU291cmNlRW5hYmxlZCB9KTtcblxuICAgIGNvcmUuc2F2ZWRPYmplY3RzLnJlZ2lzdGVyVHlwZSh2aXN1YWxpemF0aW9uU2F2ZWRPYmplY3QpO1xuICAgIGNvcmUuc2F2ZWRPYmplY3RzLnJlZ2lzdGVyVHlwZShzZWFyY2hTYXZlZE9iamVjdCk7XG4gICAgY29yZS5jYXBhYmlsaXRpZXMucmVnaXN0ZXJQcm92aWRlcigoKSA9PiAoe1xuICAgICAgb2JzZXJ2YWJpbGl0eToge1xuICAgICAgICBzaG93OiB0cnVlLFxuICAgICAgfSxcbiAgICB9KSk7XG5cbiAgICBhc3Npc3RhbnREYXNoYm9hcmRzPy5yZWdpc3Rlck1lc3NhZ2VQYXJzZXIoUFBMUGFyc2Vycyk7XG5cbiAgICByZXR1cm4ge307XG4gIH1cblxuICBwdWJsaWMgc3RhcnQoX2NvcmU6IENvcmVTdGFydCkge1xuICAgIHRoaXMubG9nZ2VyLmRlYnVnKCdPYnNlcnZhYmlsaXR5OiBTdGFydGVkJyk7XG4gICAgcmV0dXJuIHt9O1xuICB9XG5cbiAgcHVibGljIHN0b3AoKSB7fVxufVxuIl0sIm1hcHBpbmdzIjoiOzs7Ozs7QUFpQkEsSUFBQUEsV0FBQSxHQUFBQyxPQUFBO0FBQ0EsSUFBQUMsZ0NBQUEsR0FBQUQsT0FBQTtBQUNBLElBQUFFLFdBQUEsR0FBQUYsT0FBQTtBQUNBLElBQUFHLFdBQUEsR0FBQUgsT0FBQTtBQUNBLElBQUFJLE1BQUEsR0FBQUosT0FBQTtBQUNBLElBQUFLLDJCQUFBLEdBQUFMLE9BQUE7QUFHb0QsU0FBQU0sZ0JBQUFDLEdBQUEsRUFBQUMsR0FBQSxFQUFBQyxLQUFBLElBQUFELEdBQUEsR0FBQUUsY0FBQSxDQUFBRixHQUFBLE9BQUFBLEdBQUEsSUFBQUQsR0FBQSxJQUFBSSxNQUFBLENBQUFDLGNBQUEsQ0FBQUwsR0FBQSxFQUFBQyxHQUFBLElBQUFDLEtBQUEsRUFBQUEsS0FBQSxFQUFBSSxVQUFBLFFBQUFDLFlBQUEsUUFBQUMsUUFBQSxvQkFBQVIsR0FBQSxDQUFBQyxHQUFBLElBQUFDLEtBQUEsV0FBQUYsR0FBQTtBQUFBLFNBQUFHLGVBQUFNLEdBQUEsUUFBQVIsR0FBQSxHQUFBUyxZQUFBLENBQUFELEdBQUEsMkJBQUFSLEdBQUEsZ0JBQUFBLEdBQUEsR0FBQVUsTUFBQSxDQUFBVixHQUFBO0FBQUEsU0FBQVMsYUFBQUUsS0FBQSxFQUFBQyxJQUFBLGVBQUFELEtBQUEsaUJBQUFBLEtBQUEsa0JBQUFBLEtBQUEsTUFBQUUsSUFBQSxHQUFBRixLQUFBLENBQUFHLE1BQUEsQ0FBQUMsV0FBQSxPQUFBRixJQUFBLEtBQUFHLFNBQUEsUUFBQUMsR0FBQSxHQUFBSixJQUFBLENBQUFLLElBQUEsQ0FBQVAsS0FBQSxFQUFBQyxJQUFBLDJCQUFBSyxHQUFBLHNCQUFBQSxHQUFBLFlBQUFFLFNBQUEsNERBQUFQLElBQUEsZ0JBQUFGLE1BQUEsR0FBQVUsTUFBQSxFQUFBVCxLQUFBLEtBekJwRDtBQUNBO0FBQ0E7QUFDQTtBQThCTyxNQUFNVSxtQkFBbUIsQ0FDd0M7RUFHdEVDLFdBQVdBLENBQWtCQyxrQkFBNEMsRUFBRTtJQUFBLEtBQTlDQSxrQkFBNEMsR0FBNUNBLGtCQUE0QztJQUFBekIsZUFBQTtJQUN2RSxJQUFJLENBQUMwQixNQUFNLEdBQUdELGtCQUFrQixDQUFDQyxNQUFNLENBQUNDLEdBQUcsQ0FBQyxDQUFDO0VBQy9DO0VBRUEsTUFBYUMsS0FBS0EsQ0FDaEJDLElBQWUsRUFDZkMsSUFHQyxFQUNEO0lBQ0EsTUFBTTtNQUFFQyxtQkFBbUI7TUFBRUM7SUFBVyxDQUFDLEdBQUdGLElBQUk7SUFDaEQsSUFBSSxDQUFDSixNQUFNLENBQUNPLEtBQUssQ0FBQyxzQkFBc0IsQ0FBQztJQUN6QyxNQUFNQyxNQUFNLEdBQUdMLElBQUksQ0FBQ00sSUFBSSxDQUFDQyxZQUFZLENBQUMsQ0FBQztJQUV2QyxNQUFNQyxpQkFBaUIsR0FBRyxDQUFDLENBQUNMLFVBQVU7SUFDdEMsTUFBTU0sNkJBQW1ELEdBQUdULElBQUksQ0FBQ1UsVUFBVSxDQUFDQyxNQUFNLENBQUNDLFlBQVksQ0FDN0YsMEJBQTBCLEVBQzFCO01BQ0VDLE9BQU8sRUFBRSxDQUFDQyxxQkFBUyxFQUFFQyw4REFBNkI7SUFDcEQsQ0FDRixDQUFDO0lBQ0QsSUFBSVAsaUJBQWlCLEVBQUU7TUFDckJMLFVBQVUsQ0FBQ2EsdUJBQXVCLENBQUNGLHFCQUFTLENBQUM7TUFDN0NYLFVBQVUsQ0FBQ2EsdUJBQXVCLENBQUNELDhEQUE2QixDQUFDO0lBQ25FO0lBQ0E7SUFDQWYsSUFBSSxDQUFDTSxJQUFJLENBQUNXLDJCQUEyQixDQUFDLHNCQUFzQixFQUFFLENBQUNDLFFBQVEsRUFBRUMsUUFBUSxLQUFLO01BQ3BGLE9BQU87UUFDTHRCLE1BQU0sRUFBRSxJQUFJLENBQUNBLE1BQU07UUFDbkJ1QixtQkFBbUIsRUFBRVg7TUFDdkIsQ0FBQztJQUNILENBQUMsQ0FBQztJQUVGLE1BQU1ZLFlBQThCLEdBQUc7TUFDckNDLElBQUksRUFBRSxxQkFBcUI7TUFDM0JDLE1BQU0sRUFBRSxLQUFLO01BQ2JDLGFBQWEsRUFBRSxRQUFRO01BQ3ZCQyxRQUFRLEVBQUU7UUFDUkMsT0FBTyxFQUFFLEtBQUs7UUFDZEMsVUFBVSxFQUFFO1VBQ1ZDLEtBQUssRUFBRTtZQUNMQyxJQUFJLEVBQUU7VUFDUixDQUFDO1VBQ0RDLFdBQVcsRUFBRTtZQUNYRCxJQUFJLEVBQUU7VUFDUjtRQUNGO01BQ0YsQ0FBQztNQUNERSxVQUFVLEVBQUU7UUFDVkMsdUJBQXVCLEVBQUUsSUFBSTtRQUM3QkMsV0FBV0EsQ0FBQSxFQUFHO1VBQ1osT0FBTztZQUNMQyxJQUFJLEVBQUcsd0NBQXVDO1lBQzlDQyxrQkFBa0IsRUFBRTtVQUN0QixDQUFDO1FBQ0gsQ0FBQztRQUNEQyxRQUFRQSxDQUFDaEUsR0FBRyxFQUFFO1VBQ1osT0FBUSwyQkFBMEJBLEdBQUcsQ0FBQ2lFLEVBQUcsR0FBRTtRQUM3QztNQUNGLENBQUM7TUFDREMsVUFBVSxFQUFFO1FBQ1YsT0FBTyxFQUFHQyxHQUFHLEtBQU07VUFBRSxHQUFHQSxHQUFHO1VBQUVULFdBQVcsRUFBRTtRQUFHLENBQUMsQ0FBQztRQUMvQyxPQUFPLEVBQUdTLEdBQUcsS0FBTTtVQUFFLEdBQUdBLEdBQUc7VUFBRVQsV0FBVyxFQUFFO1FBQXdCLENBQUMsQ0FBQztRQUNwRSxPQUFPLEVBQUdTLEdBQUcsS0FBTTtVQUNqQixHQUFHQSxHQUFHO1VBQ05DLFdBQVcsRUFBRUMsUUFBUSxDQUFFRixHQUFHLENBQThCQyxXQUFXLElBQUksR0FBRyxFQUFFLEVBQUU7UUFDaEYsQ0FBQztNQUNIO0lBQ0YsQ0FBQztJQUVELE1BQU1FLHVCQUF5QyxHQUFHO01BQ2hEcEIsSUFBSSxFQUFFLHNCQUFzQjtNQUM1QkMsTUFBTSxFQUFFLEtBQUs7TUFDYkMsYUFBYSxFQUFFLFFBQVE7TUFDdkJPLFVBQVUsRUFBRTtRQUNWQyx1QkFBdUIsRUFBRSxJQUFJO1FBQzdCQyxXQUFXQSxDQUFDN0QsR0FBcUMsRUFBRTtVQUNqRCxPQUFPO1lBQ0w4RCxJQUFJLEVBQUcsZ0NBQStCOUQsR0FBRyxDQUFDaUUsRUFBRyxFQUFDO1lBQzlDRixrQkFBa0IsRUFBRTtVQUN0QixDQUFDO1FBQ0gsQ0FBQztRQUNEQyxRQUFRQSxDQUFDaEUsR0FBcUMsRUFBRTtVQUM5QyxPQUFPQSxHQUFHLENBQUN1RSxVQUFVLENBQUNyQixJQUFJO1FBQzVCO01BQ0YsQ0FBQztNQUNERyxRQUFRLEVBQUU7UUFDUkMsT0FBTyxFQUFFLEtBQUs7UUFDZEMsVUFBVSxFQUFFO1VBQ1ZMLElBQUksRUFBRTtZQUNKTyxJQUFJLEVBQUU7VUFDUixDQUFDO1VBQ0RlLFlBQVksRUFBRTtZQUNaZixJQUFJLEVBQUU7VUFDUixDQUFDO1VBQ0QxQixVQUFVLEVBQUU7WUFDVjBCLElBQUksRUFBRTtVQUNSLENBQUM7VUFDRGdCLFlBQVksRUFBRTtZQUNaaEIsSUFBSSxFQUFFO1VBQ1IsQ0FBQztVQUNEaUIsTUFBTSxFQUFFO1lBQ05qQixJQUFJLEVBQUU7VUFDUjtRQUNGO01BQ0Y7SUFDRixDQUFDO0lBRUQsTUFBTWtCLHVCQUF5QyxHQUFHO01BQ2hEekIsSUFBSSxFQUFFLHNCQUFzQjtNQUM1QkMsTUFBTSxFQUFFLEtBQUs7TUFDYkMsYUFBYSxFQUFFLFFBQVE7TUFDdkJPLFVBQVUsRUFBRTtRQUNWQyx1QkFBdUIsRUFBRSxJQUFJO1FBQzdCQyxXQUFXQSxDQUFDN0QsR0FBdUMsRUFBRTtVQUNuRCxPQUFPO1lBQ0w4RCxJQUFJLEVBQUcsZ0NBQStCOUQsR0FBRyxDQUFDdUUsVUFBVSxDQUFDckIsSUFBSyxFQUFDO1lBQzNEYSxrQkFBa0IsRUFBRTtVQUN0QixDQUFDO1FBQ0gsQ0FBQztRQUNEQyxRQUFRQSxDQUFDaEUsR0FBdUMsRUFBRTtVQUFBLElBQUE0RSxxQkFBQTtVQUNoRCxRQUFBQSxxQkFBQSxHQUFPNUUsR0FBRyxDQUFDdUUsVUFBVSxDQUFDTSxXQUFXLGNBQUFELHFCQUFBLGNBQUFBLHFCQUFBLEdBQUk1RSxHQUFHLENBQUN1RSxVQUFVLENBQUNyQixJQUFJO1FBQzFEO01BQ0YsQ0FBQztNQUNERyxRQUFRLEVBQUU7UUFDUkMsT0FBTyxFQUFFLEtBQUs7UUFDZEMsVUFBVSxFQUFFO1VBQ1ZMLElBQUksRUFBRTtZQUNKTyxJQUFJLEVBQUU7VUFDUixDQUFDO1VBQ0RxQixPQUFPLEVBQUU7WUFDUHJCLElBQUksRUFBRTtVQUNSLENBQUM7VUFDRG9CLFdBQVcsRUFBRTtZQUNYcEIsSUFBSSxFQUFFO1VBQ1IsQ0FBQztVQUNEc0IsT0FBTyxFQUFFO1lBQ1B0QixJQUFJLEVBQUU7VUFDUixDQUFDO1VBQ0RBLElBQUksRUFBRTtZQUNKQSxJQUFJLEVBQUU7VUFDUixDQUFDO1VBQ0R1QixNQUFNLEVBQUU7WUFDTnZCLElBQUksRUFBRTtVQUNSLENBQUM7VUFDRHdCLE1BQU0sRUFBRTtZQUNOeEIsSUFBSSxFQUFFO1VBQ1IsQ0FBQztVQUNEQyxXQUFXLEVBQUU7WUFDWEQsSUFBSSxFQUFFO1VBQ1IsQ0FBQztVQUNEeUIsU0FBUyxFQUFFO1lBQ1R6QixJQUFJLEVBQUU7VUFDUixDQUFDO1VBQ0QwQixPQUFPLEVBQUU7WUFDUDFCLElBQUksRUFBRTtVQUNSLENBQUM7VUFDRDJCLFVBQVUsRUFBRTtZQUNWM0IsSUFBSSxFQUFFO1VBQ1IsQ0FBQztVQUNEaUIsTUFBTSxFQUFFO1lBQ05qQixJQUFJLEVBQUU7VUFDUixDQUFDO1VBQ0Q0QixVQUFVLEVBQUU7WUFDVjVCLElBQUksRUFBRTtVQUNSO1FBQ0Y7TUFDRixDQUFDO01BQ0RTLFVBQVUsRUFBRTtRQUNWLE9BQU8sRUFBRW9CO01BQ1g7SUFDRixDQUFDO0lBRUQxRCxJQUFJLENBQUMyRCxZQUFZLENBQUNDLFlBQVksQ0FBQ3ZDLFlBQVksQ0FBQztJQUM1Q3JCLElBQUksQ0FBQzJELFlBQVksQ0FBQ0MsWUFBWSxDQUFDbEIsdUJBQXVCLENBQUM7SUFDdkQxQyxJQUFJLENBQUMyRCxZQUFZLENBQUNDLFlBQVksQ0FBQ2IsdUJBQXVCLENBQUM7O0lBRXZEO0lBQ0EsSUFBQWMsa0JBQVcsRUFBQztNQUFFeEQsTUFBTTtNQUFFeUQsTUFBTSxFQUFFckQsNkJBQTZCO01BQUVEO0lBQWtCLENBQUMsQ0FBQztJQUVqRlIsSUFBSSxDQUFDMkQsWUFBWSxDQUFDQyxZQUFZLENBQUNHLG9EQUF3QixDQUFDO0lBQ3hEL0QsSUFBSSxDQUFDMkQsWUFBWSxDQUFDQyxZQUFZLENBQUNJLDZDQUFpQixDQUFDO0lBQ2pEaEUsSUFBSSxDQUFDaUUsWUFBWSxDQUFDQyxnQkFBZ0IsQ0FBQyxPQUFPO01BQ3hDQyxhQUFhLEVBQUU7UUFDYkMsSUFBSSxFQUFFO01BQ1I7SUFDRixDQUFDLENBQUMsQ0FBQztJQUVIbEUsbUJBQW1CLGFBQW5CQSxtQkFBbUIsZUFBbkJBLG1CQUFtQixDQUFFbUUscUJBQXFCLENBQUNDLHNCQUFVLENBQUM7SUFFdEQsT0FBTyxDQUFDLENBQUM7RUFDWDtFQUVPQyxLQUFLQSxDQUFDQyxLQUFnQixFQUFFO0lBQzdCLElBQUksQ0FBQzNFLE1BQU0sQ0FBQ08sS0FBSyxDQUFDLHdCQUF3QixDQUFDO0lBQzNDLE9BQU8sQ0FBQyxDQUFDO0VBQ1g7RUFFT3FFLElBQUlBLENBQUEsRUFBRyxDQUFDO0FBQ2pCO0FBQUNDLE9BQUEsQ0FBQWhGLG1CQUFBLEdBQUFBLG1CQUFBIn0=