#!/bin/sh

# Copyright (c) 2024 Intel Corporation                                          
# SPDX-License-Identifier: MIT                                                  
# shellcheck shell=sh
# shellcheck source=/dev/null
# shellcheck disable=SC1091

# FPGA environment setup for oneAPI

# ############################################################################

# Detect host environment (fail fast)
ARCH=$(uname -m)
arch_type=
case $ARCH in
x86_64) # Linux x86-64
	arch_type=linux64
	;;
armv7l) # Linux SoC
	arch_type=arm32
	;;
*)
	echo "Platform '$ARCH' is not supported by the Intel(R) FPGA SDK for OpenCL(TM)"
	exit 1
	;;
esac

# FPGA_VARS_DIR is an exported variable from the compiler/<version>/env/vars.sh script
# It is the directory where this script exists
SCRIPTDIR="${FPGA_VARS_DIR}"

# Prepend path segment(s) to path-like env vars (PATH, CPATH, etc.).

# prepend_path() avoids dangling ":" that affects some env vars (PATH and CPATH)
# PATH > https://www.gnu.org/software/libc/manual/html_node/Standard-Environment.html

# Usage:
#   env_var=$(prepend_path "$prepend_to_var" "$existing_env_var")
#   export env_var
#
# Inputs:
#   $1 == path segment to be prepended to $2
#   $2 == value of existing path-like environment variable

prepend_path() (
	path_to_add="$1"
	path_is_now="$2"

	if [ "" = "${path_is_now}" ] ; then   # avoid dangling ":"
		printf "%s" "${path_to_add}"
	else
		printf "%s" "${path_to_add}:${path_is_now}"
	fi
)

# Set up the Installable Client Driver (ICD) file for the sim/HW flow
OCL_ICD_FILENAMES=$(prepend_path "libalteracl.so" "${OCL_ICD_FILENAMES:-}")
# Set up the ICD file for the emulator
OCL_ICD_FILENAMES=$(prepend_path "libintelocl_emu.so" "${OCL_ICD_FILENAMES:-}")
export OCL_ICD_FILENAMES

# The following line is meant for the runtime:
# Applications need to know where the FPGA Client Drivers (FCD)
# for the FPGA boards are
export ACL_BOARD_VENDOR_PATH=/opt/Intel/OpenCLFPGA/oneAPI/Boards

# Set this so VTune can find aocl (it uses this variable to search)
export INTELFPGAOCLSDKROOT="${SCRIPTDIR}"

# Update PATH variables so the compiler can find information:
# PATH           : executables  (aoc, aocl)
# LD_LIBRARY_PATH: libraries    (libOpenCL.so, ...)
# CPATH          : header files (acl_types.h, ...)
PATH=$(prepend_path "${INTELFPGAOCLSDKROOT}/bin" "${PATH:-}")
export PATH

LD_LIBRARY_PATH=$(prepend_path "${INTELFPGAOCLSDKROOT}/host/$arch_type/lib" "${LD_LIBRARY_PATH:-}")
export LD_LIBRARY_PATH

CPATH=$(prepend_path "${INTELFPGAOCLSDKROOT}/include" "${CPATH:-}")
export CPATH
