module Colors
  module ColormapRegistry
    register_listed_colormap("magma")
    register_listed_colormap("inferno")
    register_listed_colormap("plasma")
    register_listed_colormap("viridis")
    register_listed_colormap("cividis")

    twilight_data = load_colormap_data("twilight")
    register_listed_colormap("twilight", twilight_data)

    twilight_shifted_data = [*twilight_data[(twilight_data.length/2) .. -1],
                             *twilight_data[0 ... (twilight_data.length/2)]]
    twilight_shifted_data.reverse!
    register_listed_colormap("twilight_shifted", twilight_shifted_data)

    register_listed_colormap("turbo")

    data = {}

    data[:binary] = {
      red:    [ [0r, 1r, 1r], [1r, 0r, 0r] ],
      green:  [ [0r, 1r, 1r], [1r, 0r, 0r] ],
      blue:   [ [0r, 1r, 1r], [1r, 0r, 0r] ]
    }

    data[:autumn] = {
      red:   [ [0r, 1r, 1r], [1r, 1r, 1r] ],
      green: [ [0r, 0r, 0r], [1r, 1r, 1r] ],
      blue:  [ [0r, 0r, 0r], [1r, 0r, 0r ] ]
    }

    data[:bone] = {
      red:   [ [0r, 0r, 0r],
               [0.746032, 0.652778, 0.652778],
               [1.0, 1.0, 1.0] ],
      green: [ [0r, 0r, 0r],
               [0.365079, 0.319444, 0.319444],
               [0.746032, 0.777778, 0.777778],
               [1.0, 1.0, 1.0] ],
      blue:  [ [0r, 0r, 0r],
               [0.365079, 0.444444, 0.444444],
               [1.0, 1.0, 1.0] ]
    }

    data[:cool] = {
      red:   [[0r, 0r, 0r], [1r, 1r, 1r]],
      green: [[0r, 1r, 1r], [1r, 0r, 0r]],
      blue:  [[0r, 1r, 1r], [1r, 1r, 1r]]
    }

    data[:copper] = {
      red:   [ [0r, 0r, 0r],
               [0.809524, 1.000000, 1.000000],
               [1.0, 1.0, 1.0]],
      green: [ [0r, 0r, 0r],
               [1.0, 0.7812, 0.7812] ],
      blue:  [ [0r, 0r, 0r],
               [1.0, 0.4975, 0.4975] ]
    }

    data[:flag] = {
      red:   ->(x) { 0.75 * Math.sin((x * 31.5 + 0.25) * Math::PI) + 0.5 },
      green: ->(x) { Math.sin(x * 31.5 * Math::PI) },
      blue:  ->(x) { 0.75 * Math.sin((x * 31.5 - 0.25) * Math::PI) + 0.5 }
    }

    data[:prism] = {
      red:   ->(x) { 0.75 * Math.sin((x * 20.9 + 0.25) * Math::PI) + 0.67 },
      green: ->(x) { 0.75 * Math.sin((x * 20.9 - 0.25) * Math::PI) + 0.33 },
      blue:  ->(x) { -1.1 * Math.sin((x * 20.9) * Math::PI) }
    }

    cubehelix_color = ->(gamma, s, r, h, p0, p1, x) do
      xg = x ** gamma
      a = h * xg * (1r - xg) / 2r
      phi = 2r * Math::PI * (s / 3r + r * x)
      xg + a * (p0 * Math.cos(phi) + p1 * Math.sin(phi))
    end

    data[:cubehelix] = {
      red:   cubehelix_color.curry[1.0, 0.5, -1.5, 1.0, -0.14861, 1.78277], 
      green: cubehelix_color.curry[1.0, 0.5, -1.5, 1.0, -0.29227, -0.90649],
      blue:  cubehelix_color.curry[1.0, 0.5, -1.5, 1.0, 1.97294, 0.0]
    }

    data[:bwr] = [[0.0, 0.0, 1.0], [1.0, 1.0, 1.0], [1.0, 0.0, 0.0]]
    data[:brg] = [[0.0, 0.0, 1.0], [1.0, 0.0, 0.0], [0.0, 1.0, 0.0]]

    # Gnuplot palette functions
    gnuplot_functions = {
      0  => ->(x) { 0r },
      1  => ->(x) { 0.5r },
      2  => ->(x) { 1r },
      3  => ->(x) { x },
      4  => ->(x) { x ** 2 },
      5  => ->(x) { x ** 3 },
      6  => ->(x) { x ** 4 },
      7  => ->(x) { Math.sqrt(x) },
      8  => ->(x) { Math.sqrt(Math.sqrt(x)) },
      9  => ->(x) { Math.sin(x * Math::PI / 2) },
      10 => ->(x) { Math.cos(x * Math::PI / 2) },
      11 => ->(x) { (x - 0.5).abs },
      12 => ->(x) { (2 * x - 1) ** 2 },
      13 => ->(x) { Math.sin(x * Math::PI) },
      14 => ->(x) { Math.cos(x * Math::PI).abs },
      15 => ->(x) { Math.sin(x * 2 * Math::PI) },
      16 => ->(x) { Math.cos(x * 2 * Math::PI) },
      17 => ->(x) { Math.sin(x * 2 * Math::PI).abs },
      18 => ->(x) { Math.cos(x * 2 * Math::PI).abs },
      19 => ->(x) { Math.sin(x * 4 * Math::PI).abs },
      20 => ->(x) { Math.cos(x * 4 * Math::PI).abs },
      21 => ->(x) { 3 * x },
      22 => ->(x) { 3 * x - 1 },
      23 => ->(x) { 3 * x - 2 },
      24 => ->(x) { (3 * x - 1).abs },
      25 => ->(x) { (3 * x - 2).abs },
      26 => ->(x) { (3 * x - 1) / 2 },
      27 => ->(x) { (3 * x - 2) / 2 },
      28 => ->(x) { ((3 * x - 1) / 2).abs },
      29 => ->(x) { ((3 * x - 2) / 2).abs },
      30 => ->(x) { x / 0.32 - 0.78125 },
      31 => ->(x) { 2 * x - 0.84 },
      32 => ->(x) {
        case
        when x < 0.25
          4 * x
        when 0.25 <= x && x < 0.92
          -2 * x + 1.84
        when 0.92 <= x
          x / 0.08 - 11.5
        end
      },
      33 => ->(x) { (2 * x - 0.5).abs },
      34 => ->(x) { 2 * x },
      35 => ->(x) { 2 * x - 0.5 },
      36 => ->(x) { 2 * x - 1 },
    }

    data[:gnuplot] = {
      red:   gnuplot_functions[7],
      green: gnuplot_functions[5],
      blue:  gnuplot_functions[15]
    }

    data[:gnuplot2] = {
      red:   gnuplot_functions[30],
      green: gnuplot_functions[31],
      blue:  gnuplot_functions[32]
    }

    data[:ocean] = {
      red:   gnuplot_functions[23],
      green: gnuplot_functions[28],
      blue:  gnuplot_functions[3]
    }

    data[:afmhot] = {
      red:   gnuplot_functions[34],
      green: gnuplot_functions[35],
      blue:  gnuplot_functions[36]
    }

    data[:rainbow] = {
      red:   gnuplot_functions[33],
      green: gnuplot_functions[13],
      blue:  gnuplot_functions[10]
    }

    data[:seismic] = [
      [0.0, 0.0, 0.3],
      [0.0, 0.0, 1.0],
      [1.0, 1.0, 1.0],
      [1.0, 0.0, 0.0],
      [0.5, 0.0, 0.0]
    ]

    data[:terrain] = [
      [0.00, [0.2, 0.2,  0.6]],
      [0.15, [0.0, 0.6,  1.0]],
      [0.25, [0.0, 0.8,  0.4]],
      [0.50, [1.0, 1.0,  0.6]],
      [0.75, [0.5, 0.36, 0.33]],
      [1.00, [1.0, 1.0,  1.0]]
    ]

    data[:gray] = {
      red:   [[0r, 0r, 0r], [1r, 1r, 1r]],
      green: [[0r, 0r, 0r], [1r, 1r, 1r]],
      blue:  [[0r, 0r, 0r], [1r, 1r, 1r]]
    }

    data[:hot] = {
      red:   [ [0r, 0.0416, 0.0416],
               [0.365079, 1.000000, 1.000000],
               [1.0, 1.0, 1.0] ],
      green: [ [0r, 0r, 0r],
               [0.365079, 0.000000, 0.000000],
               [0.746032, 1.000000, 1.000000],
               [1.0, 1.0, 1.0] ],
      blue:  [ [0r, 0r, 0r],
               [0.746032, 0.000000, 0.000000],
               [1.0, 1.0, 1.0] ]
    }

    data[:hsv] = {
      red:   [ [0r, 1r, 1r],
               [0.158730, 1.000000, 1.000000],
               [0.174603, 0.968750, 0.968750],
               [0.333333, 0.031250, 0.031250],
               [0.349206, 0.000000, 0.000000],
               [0.666667, 0.000000, 0.000000],
               [0.682540, 0.031250, 0.031250],
               [0.841270, 0.968750, 0.968750],
               [0.857143, 1.000000, 1.000000],
               [1.0, 1.0, 1.0] ],
      green: [ [0r, 0r, 0r],
               [0.158730, 0.937500, 0.937500],
               [0.174603, 1.000000, 1.000000],
               [0.507937, 1.000000, 1.000000],
               [0.666667, 0.062500, 0.062500],
               [0.682540, 0.000000, 0.000000],
               [1r, 0r, 0r] ],
      blue:  [ [0r, 0r, 0r],
               [0.333333, 0.000000, 0.000000],
               [0.349206, 0.062500, 0.062500],
               [0.507937, 1.000000, 1.000000],
               [0.841270, 1.000000, 1.000000],
               [0.857143, 0.937500, 0.937500],
               [1.0, 0.09375, 0.09375] ]
    }

    data[:jet] = {
      red:   [ [0.00, 0r, 0r],
               [0.35, 0r, 0r],
               [0.66, 1r, 1r],
               [0.89, 1r, 1r],
               [1.00, 0.5, 0.5] ],
      green: [ [0.000, 0r, 0r],
               [0.125, 0r, 0r],
               [0.375, 1r, 1r],
               [0.640, 1r, 1r],
               [0.910, 0r, 0r],
               [1.000, 0r, 0r] ],
      blue:  [ [0.00, 0.5, 0.5],
               [0.11, 1r, 1r],
               [0.34, 1r, 1r],
               [0.65, 0r, 0r],
               [1.00, 0r, 0r] ]
    }

    data[:pink] = load_colormap_data("pink")

    data[:spring] = {
      red:   [[0r, 1r, 1r], [1.0, 1.0, 1.0]],
      green: [[0r, 0r, 0r], [1.0, 1.0, 1.0]],
      blue:  [[0r, 1r, 1r], [1.0, 0.0, 0.0]]
    }

    data[:summer] = {
      red:   [[0r, 0r,  0r ], [1.0, 1.0, 1.0]],
      green: [[0r, 0.5, 0.5], [1.0, 1.0, 1.0]],
      blue:  [[0r, 0.4, 0.4], [1.0, 0.4, 0.4]]
    }

    data[:winter] = {
      red:   [[0r, 0r, 0r], [1.0, 0.0, 0.0]],
      green: [[0r, 0r, 0r], [1.0, 1.0, 1.0]],
      blue:  [[0r, 1r, 1r], [1.0, 0.5, 0.5]]
    }

    data[:nipy_spectral] = load_colormap_data("nipy_spectral")

    # 34 colormaps based on color specifications and designs
    # developed by Cynthia Brewer (http://colorbrewer.org).
    # The ColorBrewer palettes have been included under the terms
    # of an Apache-stype license (for details, see the file
    # LICENSE_COLORBREWER in the license directory of the matplotlib
    # source distribution).

    # RGB values taken from Brewer's Excel sheet, divided by 255

    data[:Blues] = [
      [0.96862745098039216,  0.98431372549019602,  1.0                ],
      [0.87058823529411766,  0.92156862745098034,  0.96862745098039216],
      [0.77647058823529413,  0.85882352941176465,  0.93725490196078431],
      [0.61960784313725492,  0.792156862745098  ,  0.88235294117647056],
      [0.41960784313725491,  0.68235294117647061,  0.83921568627450982],
      [0.25882352941176473,  0.5725490196078431 ,  0.77647058823529413],
      [0.12941176470588237,  0.44313725490196076,  0.70980392156862748],
      [0.03137254901960784,  0.31764705882352939,  0.61176470588235299],
      [0.03137254901960784,  0.18823529411764706,  0.41960784313725491]
    ]

    data[:BrBG] = [
      [0.32941176470588235,  0.18823529411764706,  0.0196078431372549 ],
      [0.5490196078431373 ,  0.31764705882352939,  0.0392156862745098 ],
      [0.74901960784313726,  0.50588235294117645,  0.17647058823529413],
      [0.87450980392156863,  0.76078431372549016,  0.49019607843137253],
      [0.96470588235294119,  0.90980392156862744,  0.76470588235294112],
      [0.96078431372549022,  0.96078431372549022,  0.96078431372549022],
      [0.7803921568627451 ,  0.91764705882352937,  0.89803921568627454],
      [0.50196078431372548,  0.80392156862745101,  0.75686274509803919],
      [0.20784313725490197,  0.59215686274509804,  0.5607843137254902 ],
      [0.00392156862745098,  0.4                ,  0.36862745098039218],
      [0.0                ,  0.23529411764705882,  0.18823529411764706]
    ]

    data[:BuGn] = [
      [0.96862745098039216,  0.9882352941176471 ,  0.99215686274509807],
      [0.89803921568627454,  0.96078431372549022,  0.97647058823529409],
      [0.8                ,  0.92549019607843142,  0.90196078431372551],
      [0.6                ,  0.84705882352941175,  0.78823529411764703],
      [0.4                ,  0.76078431372549016,  0.64313725490196083],
      [0.25490196078431371,  0.68235294117647061,  0.46274509803921571],
      [0.13725490196078433,  0.54509803921568623,  0.27058823529411763],
      [0.0                ,  0.42745098039215684,  0.17254901960784313],
      [0.0                ,  0.26666666666666666,  0.10588235294117647]
    ]

    data[:BuPu] = [
      [0.96862745098039216,  0.9882352941176471 ,  0.99215686274509807],
      [0.8784313725490196 ,  0.92549019607843142,  0.95686274509803926],
      [0.74901960784313726,  0.82745098039215681,  0.90196078431372551],
      [0.61960784313725492,  0.73725490196078436,  0.85490196078431369],
      [0.5490196078431373 ,  0.58823529411764708,  0.77647058823529413],
      [0.5490196078431373 ,  0.41960784313725491,  0.69411764705882351],
      [0.53333333333333333,  0.25490196078431371,  0.61568627450980395],
      [0.50588235294117645,  0.05882352941176471,  0.48627450980392156],
      [0.30196078431372547,  0.0                ,  0.29411764705882354]
    ]

    data[:GnBu] = [
      [0.96862745098039216,  0.9882352941176471 ,  0.94117647058823528],
      [0.8784313725490196 ,  0.95294117647058818,  0.85882352941176465],
      [0.8                ,  0.92156862745098034,  0.77254901960784317],
      [0.6588235294117647 ,  0.8666666666666667 ,  0.70980392156862748],
      [0.4823529411764706 ,  0.8                ,  0.7686274509803922 ],
      [0.30588235294117649,  0.70196078431372544,  0.82745098039215681],
      [0.16862745098039217,  0.5490196078431373 ,  0.74509803921568629],
      [0.03137254901960784,  0.40784313725490196,  0.67450980392156867],
      [0.03137254901960784,  0.25098039215686274,  0.50588235294117645]
    ]

    data[:Greens] = [
      [0.96862745098039216,  0.9882352941176471 ,  0.96078431372549022],
      [0.89803921568627454,  0.96078431372549022,  0.8784313725490196 ],
      [0.7803921568627451 ,  0.9137254901960784 ,  0.75294117647058822],
      [0.63137254901960782,  0.85098039215686272,  0.60784313725490191],
      [0.45490196078431372,  0.7686274509803922 ,  0.46274509803921571],
      [0.25490196078431371,  0.6705882352941176 ,  0.36470588235294116],
      [0.13725490196078433,  0.54509803921568623,  0.27058823529411763],
      [0.0                ,  0.42745098039215684,  0.17254901960784313],
      [0.0                ,  0.26666666666666666,  0.10588235294117647]
    ]

    data[:Greys] = [
      [1.0                ,  1.0                ,  1.0                ],
      [0.94117647058823528,  0.94117647058823528,  0.94117647058823528],
      [0.85098039215686272,  0.85098039215686272,  0.85098039215686272],
      [0.74117647058823533,  0.74117647058823533,  0.74117647058823533],
      [0.58823529411764708,  0.58823529411764708,  0.58823529411764708],
      [0.45098039215686275,  0.45098039215686275,  0.45098039215686275],
      [0.32156862745098042,  0.32156862745098042,  0.32156862745098042],
      [0.14509803921568629,  0.14509803921568629,  0.14509803921568629],
      [0.0                ,  0.0                ,  0.0                ]
    ]

    data[:OrRd] = [
      [1.0                ,  0.96862745098039216,  0.92549019607843142],
      [0.99607843137254903,  0.90980392156862744,  0.78431372549019607],
      [0.99215686274509807,  0.83137254901960789,  0.61960784313725492],
      [0.99215686274509807,  0.73333333333333328,  0.51764705882352946],
      [0.9882352941176471 ,  0.55294117647058827,  0.34901960784313724],
      [0.93725490196078431,  0.396078431372549  ,  0.28235294117647058],
      [0.84313725490196079,  0.18823529411764706,  0.12156862745098039],
      [0.70196078431372544,  0.0                ,  0.0                ],
      [0.49803921568627452,  0.0                ,  0.0                ]
    ]

    data[:Oranges] = [
      [1.0                ,  0.96078431372549022,  0.92156862745098034],
      [0.99607843137254903,  0.90196078431372551,  0.80784313725490198],
      [0.99215686274509807,  0.81568627450980391,  0.63529411764705879],
      [0.99215686274509807,  0.68235294117647061,  0.41960784313725491],
      [0.99215686274509807,  0.55294117647058827,  0.23529411764705882],
      [0.94509803921568625,  0.41176470588235292,  0.07450980392156863],
      [0.85098039215686272,  0.28235294117647058,  0.00392156862745098],
      [0.65098039215686276,  0.21176470588235294,  0.01176470588235294],
      [0.49803921568627452,  0.15294117647058825,  0.01568627450980392]
    ]

    data[:PRGn] = [
      [0.25098039215686274,  0.0                ,  0.29411764705882354],
      [0.46274509803921571,  0.16470588235294117,  0.51372549019607838],
      [0.6                ,  0.4392156862745098 ,  0.6705882352941176 ],
      [0.76078431372549016,  0.6470588235294118 ,  0.81176470588235294],
      [0.90588235294117647,  0.83137254901960789,  0.90980392156862744],
      [0.96862745098039216,  0.96862745098039216,  0.96862745098039216],
      [0.85098039215686272,  0.94117647058823528,  0.82745098039215681],
      [0.65098039215686276,  0.85882352941176465,  0.62745098039215685],
      [0.35294117647058826,  0.68235294117647061,  0.38039215686274508],
      [0.10588235294117647,  0.47058823529411764,  0.21568627450980393],
      [0.0                ,  0.26666666666666666,  0.10588235294117647]
    ]

    data[:PiYG] = [
      [0.55686274509803924,  0.00392156862745098,  0.32156862745098042],
      [0.77254901960784317,  0.10588235294117647,  0.49019607843137253],
      [0.87058823529411766,  0.46666666666666667,  0.68235294117647061],
      [0.94509803921568625,  0.71372549019607845,  0.85490196078431369],
      [0.99215686274509807,  0.8784313725490196 ,  0.93725490196078431],
      [0.96862745098039216,  0.96862745098039216,  0.96862745098039216],
      [0.90196078431372551,  0.96078431372549022,  0.81568627450980391],
      [0.72156862745098038,  0.88235294117647056,  0.52549019607843139],
      [0.49803921568627452,  0.73725490196078436,  0.25490196078431371],
      [0.30196078431372547,  0.5725490196078431 ,  0.12941176470588237],
      [0.15294117647058825,  0.39215686274509803,  0.09803921568627451]
    ]

    data[:PuBu] = [
      [1.0                ,  0.96862745098039216,  0.98431372549019602],
      [0.92549019607843142,  0.90588235294117647,  0.94901960784313721],
      [0.81568627450980391,  0.81960784313725488,  0.90196078431372551],
      [0.65098039215686276,  0.74117647058823533,  0.85882352941176465],
      [0.45490196078431372,  0.66274509803921566,  0.81176470588235294],
      [0.21176470588235294,  0.56470588235294117,  0.75294117647058822],
      [0.0196078431372549 ,  0.4392156862745098 ,  0.69019607843137254],
      [0.01568627450980392,  0.35294117647058826,  0.55294117647058827],
      [0.00784313725490196,  0.2196078431372549 ,  0.34509803921568627]
    ]

    data[:PuBuGn] = [
      [1.0                ,  0.96862745098039216,  0.98431372549019602],
      [0.92549019607843142,  0.88627450980392153,  0.94117647058823528],
      [0.81568627450980391,  0.81960784313725488,  0.90196078431372551],
      [0.65098039215686276,  0.74117647058823533,  0.85882352941176465],
      [0.40392156862745099,  0.66274509803921566,  0.81176470588235294],
      [0.21176470588235294,  0.56470588235294117,  0.75294117647058822],
      [0.00784313725490196,  0.50588235294117645,  0.54117647058823526],
      [0.00392156862745098,  0.42352941176470588,  0.34901960784313724],
      [0.00392156862745098,  0.27450980392156865,  0.21176470588235294]
    ]

    data[:PuOr] = [
      [0.49803921568627452,  0.23137254901960785,  0.03137254901960784],
      [0.70196078431372544,  0.34509803921568627,  0.02352941176470588],
      [0.8784313725490196 ,  0.50980392156862742,  0.07843137254901961],
      [0.99215686274509807,  0.72156862745098038,  0.38823529411764707],
      [0.99607843137254903,  0.8784313725490196 ,  0.71372549019607845],
      [0.96862745098039216,  0.96862745098039216,  0.96862745098039216],
      [0.84705882352941175,  0.85490196078431369,  0.92156862745098034],
      [0.69803921568627447,  0.6705882352941176 ,  0.82352941176470584],
      [0.50196078431372548,  0.45098039215686275,  0.67450980392156867],
      [0.32941176470588235,  0.15294117647058825,  0.53333333333333333],
      [0.17647058823529413,  0.0                ,  0.29411764705882354]
    ]

    data[:PuRd] = [
      [0.96862745098039216,  0.95686274509803926,  0.97647058823529409],
      [0.90588235294117647,  0.88235294117647056,  0.93725490196078431],
      [0.83137254901960789,  0.72549019607843135,  0.85490196078431369],
      [0.78823529411764703,  0.58039215686274515,  0.7803921568627451 ],
      [0.87450980392156863,  0.396078431372549  ,  0.69019607843137254],
      [0.90588235294117647,  0.16078431372549021,  0.54117647058823526],
      [0.80784313725490198,  0.07058823529411765,  0.33725490196078434],
      [0.59607843137254901,  0.0                ,  0.2627450980392157 ],
      [0.40392156862745099,  0.0                ,  0.12156862745098039]
    ]

    data[:Purples] = [
      [0.9882352941176471 ,  0.98431372549019602,  0.99215686274509807],
      [0.93725490196078431,  0.92941176470588238,  0.96078431372549022],
      [0.85490196078431369,  0.85490196078431369,  0.92156862745098034],
      [0.73725490196078436,  0.74117647058823533,  0.86274509803921573],
      [0.61960784313725492,  0.60392156862745094,  0.78431372549019607],
      [0.50196078431372548,  0.49019607843137253,  0.72941176470588232],
      [0.41568627450980394,  0.31764705882352939,  0.63921568627450975],
      [0.32941176470588235,  0.15294117647058825,  0.5607843137254902 ],
      [0.24705882352941178,  0.0                ,  0.49019607843137253]
    ]

    data[:RdBu] = [
      [0.40392156862745099,  0.0                ,  0.12156862745098039],
      [0.69803921568627447,  0.09411764705882353,  0.16862745098039217],
      [0.83921568627450982,  0.37647058823529411,  0.30196078431372547],
      [0.95686274509803926,  0.6470588235294118 ,  0.50980392156862742],
      [0.99215686274509807,  0.85882352941176465,  0.7803921568627451 ],
      [0.96862745098039216,  0.96862745098039216,  0.96862745098039216],
      [0.81960784313725488,  0.89803921568627454,  0.94117647058823528],
      [0.5725490196078431 ,  0.77254901960784317,  0.87058823529411766],
      [0.2627450980392157 ,  0.57647058823529407,  0.76470588235294112],
      [0.12941176470588237,  0.4                ,  0.67450980392156867],
      [0.0196078431372549 ,  0.18823529411764706,  0.38039215686274508]
    ]

    data[:RdGy] = [
      [0.40392156862745099,  0.0                ,  0.12156862745098039],
      [0.69803921568627447,  0.09411764705882353,  0.16862745098039217],
      [0.83921568627450982,  0.37647058823529411,  0.30196078431372547],
      [0.95686274509803926,  0.6470588235294118 ,  0.50980392156862742],
      [0.99215686274509807,  0.85882352941176465,  0.7803921568627451 ],
      [1.0                ,  1.0                ,  1.0                ],
      [0.8784313725490196 ,  0.8784313725490196 ,  0.8784313725490196 ],
      [0.72941176470588232,  0.72941176470588232,  0.72941176470588232],
      [0.52941176470588236,  0.52941176470588236,  0.52941176470588236],
      [0.30196078431372547,  0.30196078431372547,  0.30196078431372547],
      [0.10196078431372549,  0.10196078431372549,  0.10196078431372549]
    ]

    data[:RdPu] = [
      [1.0                ,  0.96862745098039216,  0.95294117647058818],
      [0.99215686274509807,  0.8784313725490196 ,  0.86666666666666667],
      [0.9882352941176471 ,  0.77254901960784317,  0.75294117647058822],
      [0.98039215686274506,  0.62352941176470589,  0.70980392156862748],
      [0.96862745098039216,  0.40784313725490196,  0.63137254901960782],
      [0.86666666666666667,  0.20392156862745098,  0.59215686274509804],
      [0.68235294117647061,  0.00392156862745098,  0.49411764705882355],
      [0.47843137254901963,  0.00392156862745098,  0.46666666666666667],
      [0.28627450980392155,  0.0                ,  0.41568627450980394]
    ]

    data[:RdYlBu] = [
      [0.6470588235294118 , 0.0                 , 0.14901960784313725],
      [0.84313725490196079, 0.18823529411764706 , 0.15294117647058825],
      [0.95686274509803926, 0.42745098039215684 , 0.2627450980392157 ],
      [0.99215686274509807, 0.68235294117647061 , 0.38039215686274508],
      [0.99607843137254903, 0.8784313725490196  , 0.56470588235294117],
      [1.0                , 1.0                 , 0.74901960784313726],
      [0.8784313725490196 , 0.95294117647058818 , 0.97254901960784312],
      [0.6705882352941176 , 0.85098039215686272 , 0.9137254901960784 ],
      [0.45490196078431372, 0.67843137254901964 , 0.81960784313725488],
      [0.27058823529411763, 0.45882352941176469 , 0.70588235294117652],
      [0.19215686274509805, 0.21176470588235294 , 0.58431372549019611]
    ]

    data[:RdYlGn] = [
      [0.6470588235294118 , 0.0                 , 0.14901960784313725],
      [0.84313725490196079, 0.18823529411764706 , 0.15294117647058825],
      [0.95686274509803926, 0.42745098039215684 , 0.2627450980392157 ],
      [0.99215686274509807, 0.68235294117647061 , 0.38039215686274508],
      [0.99607843137254903, 0.8784313725490196  , 0.54509803921568623],
      [1.0                , 1.0                 , 0.74901960784313726],
      [0.85098039215686272, 0.93725490196078431 , 0.54509803921568623],
      [0.65098039215686276, 0.85098039215686272 , 0.41568627450980394],
      [0.4                , 0.74117647058823533 , 0.38823529411764707],
      [0.10196078431372549, 0.59607843137254901 , 0.31372549019607843],
      [0.0                , 0.40784313725490196 , 0.21568627450980393]
    ]

    data[:Reds] = [
      [1.0                , 0.96078431372549022 , 0.94117647058823528],
      [0.99607843137254903, 0.8784313725490196  , 0.82352941176470584],
      [0.9882352941176471 , 0.73333333333333328 , 0.63137254901960782],
      [0.9882352941176471 , 0.5725490196078431  , 0.44705882352941179],
      [0.98431372549019602, 0.41568627450980394 , 0.29019607843137257],
      [0.93725490196078431, 0.23137254901960785 , 0.17254901960784313],
      [0.79607843137254897, 0.094117647058823528, 0.11372549019607843],
      [0.6470588235294118 , 0.058823529411764705, 0.08235294117647058],
      [0.40392156862745099, 0.0                 , 0.05098039215686274]
    ]

    data[:Spectral] = [
      [0.61960784313725492, 0.003921568627450980, 0.25882352941176473],
      [0.83529411764705885, 0.24313725490196078 , 0.30980392156862746],
      [0.95686274509803926, 0.42745098039215684 , 0.2627450980392157 ],
      [0.99215686274509807, 0.68235294117647061 , 0.38039215686274508],
      [0.99607843137254903, 0.8784313725490196  , 0.54509803921568623],
      [1.0                , 1.0                 , 0.74901960784313726],
      [0.90196078431372551, 0.96078431372549022 , 0.59607843137254901],
      [0.6705882352941176 , 0.8666666666666667  , 0.64313725490196083],
      [0.4                , 0.76078431372549016 , 0.6470588235294118 ],
      [0.19607843137254902, 0.53333333333333333 , 0.74117647058823533],
      [0.36862745098039218, 0.30980392156862746 , 0.63529411764705879]
    ]

    data[:YlGn] = [
      [1.0                , 1.0                 , 0.89803921568627454],
      [0.96862745098039216, 0.9882352941176471  , 0.72549019607843135],
      [0.85098039215686272, 0.94117647058823528 , 0.63921568627450975],
      [0.67843137254901964, 0.8666666666666667  , 0.55686274509803924],
      [0.47058823529411764, 0.77647058823529413 , 0.47450980392156861],
      [0.25490196078431371, 0.6705882352941176  , 0.36470588235294116],
      [0.13725490196078433, 0.51764705882352946 , 0.2627450980392157 ],
      [0.0                , 0.40784313725490196 , 0.21568627450980393],
      [0.0                , 0.27058823529411763 , 0.16078431372549021]
    ]

    data[:YlGnBu] = [
      [1.0                , 1.0                 , 0.85098039215686272],
      [0.92941176470588238, 0.97254901960784312 , 0.69411764705882351],
      [0.7803921568627451 , 0.9137254901960784  , 0.70588235294117652],
      [0.49803921568627452, 0.80392156862745101 , 0.73333333333333328],
      [0.25490196078431371, 0.71372549019607845 , 0.7686274509803922 ],
      [0.11372549019607843, 0.56862745098039214 , 0.75294117647058822],
      [0.13333333333333333, 0.36862745098039218 , 0.6588235294117647 ],
      [0.14509803921568629, 0.20392156862745098 , 0.58039215686274515],
      [0.03137254901960784, 0.11372549019607843 , 0.34509803921568627]
    ]

    data[:YlOrBr] = [
      [1.0                , 1.0                 , 0.89803921568627454],
      [1.0                , 0.96862745098039216 , 0.73725490196078436],
      [0.99607843137254903, 0.8901960784313725  , 0.56862745098039214],
      [0.99607843137254903, 0.7686274509803922  , 0.30980392156862746],
      [0.99607843137254903, 0.6                 , 0.16078431372549021],
      [0.92549019607843142, 0.4392156862745098  , 0.07843137254901961],
      [0.8                , 0.29803921568627451 , 0.00784313725490196],
      [0.6                , 0.20392156862745098 , 0.01568627450980392],
      [0.4                , 0.14509803921568629 , 0.02352941176470588]
    ]

    data[:YlOrRd] = [
      [1.0                , 1.0                 , 0.8                ],
      [1.0                , 0.92941176470588238 , 0.62745098039215685],
      [0.99607843137254903, 0.85098039215686272 , 0.46274509803921571],
      [0.99607843137254903, 0.69803921568627447 , 0.29803921568627451],
      [0.99215686274509807, 0.55294117647058827 , 0.23529411764705882],
      [0.9882352941176471 , 0.30588235294117649 , 0.16470588235294117],
      [0.8901960784313725 , 0.10196078431372549 , 0.10980392156862745],
      [0.74117647058823533, 0.0                 , 0.14901960784313725],
      [0.50196078431372548, 0.0                 , 0.14901960784313725]
    ]

    # ColorBrewer's qualitative maps, implemented using ListedColormap
    # for use with mpl.colors.NoNorm

    data[:Accent] = {listed: [
      [0.49803921568627452, 0.78823529411764703, 0.49803921568627452],
      [0.74509803921568629, 0.68235294117647061, 0.83137254901960789],
      [0.99215686274509807, 0.75294117647058822, 0.52549019607843139],
      [1.0,                 1.0,                 0.6                ],
      [0.2196078431372549,  0.42352941176470588, 0.69019607843137254],
      [0.94117647058823528, 0.00784313725490196, 0.49803921568627452],
      [0.74901960784313726, 0.35686274509803922, 0.09019607843137254],
      [0.4,                 0.4,                 0.4                ]
    ]}

    data[:Dark2] = {listed: [
      [0.10588235294117647, 0.61960784313725492, 0.46666666666666667],
      [0.85098039215686272, 0.37254901960784315, 0.00784313725490196],
      [0.45882352941176469, 0.4392156862745098,  0.70196078431372544],
      [0.90588235294117647, 0.16078431372549021, 0.54117647058823526],
      [0.4,                 0.65098039215686276, 0.11764705882352941],
      [0.90196078431372551, 0.6705882352941176,  0.00784313725490196],
      [0.65098039215686276, 0.46274509803921571, 0.11372549019607843],
      [0.4,                 0.4,                 0.4                ]
    ]}

    data[:Paired] = {listed: [
      [0.65098039215686276, 0.80784313725490198, 0.8901960784313725 ],
      [0.12156862745098039, 0.47058823529411764, 0.70588235294117652],
      [0.69803921568627447, 0.87450980392156863, 0.54117647058823526],
      [0.2,                 0.62745098039215685, 0.17254901960784313],
      [0.98431372549019602, 0.60392156862745094, 0.6                ],
      [0.8901960784313725,  0.10196078431372549, 0.10980392156862745],
      [0.99215686274509807, 0.74901960784313726, 0.43529411764705883],
      [1.0,                 0.49803921568627452, 0.0                ],
      [0.792156862745098,   0.69803921568627447, 0.83921568627450982],
      [0.41568627450980394, 0.23921568627450981, 0.60392156862745094],
      [1.0,                 1.0,                 0.6                ],
      [0.69411764705882351, 0.34901960784313724, 0.15686274509803921]
    ]}

    data[:Pastel1] = {listed: [
      [0.98431372549019602, 0.70588235294117652, 0.68235294117647061],
      [0.70196078431372544, 0.80392156862745101, 0.8901960784313725 ],
      [0.8,                 0.92156862745098034, 0.77254901960784317],
      [0.87058823529411766, 0.79607843137254897, 0.89411764705882357],
      [0.99607843137254903, 0.85098039215686272, 0.65098039215686276],
      [1.0,                 1.0,                 0.8                ],
      [0.89803921568627454, 0.84705882352941175, 0.74117647058823533],
      [0.99215686274509807, 0.85490196078431369, 0.92549019607843142],
      [0.94901960784313721, 0.94901960784313721, 0.94901960784313721]
    ]}

    data[:Pastel2] = {listed: [
      [0.70196078431372544, 0.88627450980392153, 0.80392156862745101],
      [0.99215686274509807, 0.80392156862745101, 0.67450980392156867],
      [0.79607843137254897, 0.83529411764705885, 0.90980392156862744],
      [0.95686274509803926, 0.792156862745098,   0.89411764705882357],
      [0.90196078431372551, 0.96078431372549022, 0.78823529411764703],
      [1.0,                 0.94901960784313721, 0.68235294117647061],
      [0.94509803921568625, 0.88627450980392153, 0.8                ],
      [0.8,                 0.8,                 0.8                ]
    ]}

    data[:Set1] = {listed: [
      [0.89411764705882357, 0.10196078431372549, 0.10980392156862745],
      [0.21568627450980393, 0.49411764705882355, 0.72156862745098038],
      [0.30196078431372547, 0.68627450980392157, 0.29019607843137257],
      [0.59607843137254901, 0.30588235294117649, 0.63921568627450975],
      [1.0,                 0.49803921568627452, 0.0                ],
      [1.0,                 1.0,                 0.2                ],
      [0.65098039215686276, 0.33725490196078434, 0.15686274509803921],
      [0.96862745098039216, 0.50588235294117645, 0.74901960784313726],
      [0.6,                 0.6,                 0.6                ]
    ]}

    data[:Set2] = {listed: [
      [0.4,                 0.76078431372549016, 0.6470588235294118 ],
      [0.9882352941176471,  0.55294117647058827, 0.3843137254901961 ],
      [0.55294117647058827, 0.62745098039215685, 0.79607843137254897],
      [0.90588235294117647, 0.54117647058823526, 0.76470588235294112],
      [0.65098039215686276, 0.84705882352941175, 0.32941176470588235],
      [1.0,                 0.85098039215686272, 0.18431372549019609],
      [0.89803921568627454, 0.7686274509803922,  0.58039215686274515],
      [0.70196078431372544, 0.70196078431372544, 0.70196078431372544]
    ]}

    data[:Set3] = {listed: [
      [0.55294117647058827, 0.82745098039215681, 0.7803921568627451 ],
      [1.0,                 1.0,                 0.70196078431372544],
      [0.74509803921568629, 0.72941176470588232, 0.85490196078431369],
      [0.98431372549019602, 0.50196078431372548, 0.44705882352941179],
      [0.50196078431372548, 0.69411764705882351, 0.82745098039215681],
      [0.99215686274509807, 0.70588235294117652, 0.3843137254901961 ],
      [0.70196078431372544, 0.87058823529411766, 0.41176470588235292],
      [0.9882352941176471,  0.80392156862745101, 0.89803921568627454],
      [0.85098039215686272, 0.85098039215686272, 0.85098039215686272],
      [0.73725490196078436, 0.50196078431372548, 0.74117647058823533],
      [0.8,                 0.92156862745098034, 0.77254901960784317],
      [1.0,                 0.92941176470588238, 0.43529411764705883]
    ]}

    # The next 7 palettes are from the Yorick scientific visualization package,
    # an evolution of the GIST package, both by David H. Munro.
    # They are released under a BSD-like license (see LICENSE_YORICK in
    # the license directory of the matplotlib source distribution).
    #
    # Most palette functions have been reduced to simple function descriptions
    # by Reinier Heeres, since the rgb components were mostly straight lines.
    # gist_earth_data and gist_ncar_data were simplified by a script and some
    # manual effort.

    data[:gist_earth] = load_colormap_data("gist_earth")

    data[:gist_gray] = {
      red:   gnuplot_functions[3],
      green: gnuplot_functions[3],
      blue:  gnuplot_functions[3]
    }

    data[:gist_heat] = {
      red:   ->(x) { 1.5 * x },
      green: ->(x) { 2 * x - 1 },
      blue:  ->(x) { 4 * x - 3 }
    }

    data[:gist_ncar] = load_colormap_data("gist_ncar")

    data[:gist_rainbow] = [
      [0.000, [1.00, 0.00, 0.16]],
      [0.030, [1.00, 0.00, 0.00]],
      [0.215, [1.00, 1.00, 0.00]],
      [0.400, [0.00, 1.00, 0.00]],
      [0.586, [0.00, 1.00, 1.00]],
      [0.770, [0.00, 0.00, 1.00]],
      [0.954, [1.00, 0.00, 1.00]],
      [1.000, [1.00, 0.00, 0.75]]
    ]

    data[:gist_stern] = {
      'red': [
        [0.000, 0.000, 0.000],
        [0.0547, 1.000, 1.000],
        [0.250, 0.027, 0.250],
        # [0.2500, 0.250, 0.250],
        [1.000, 1.000, 1.000]
      ],
      'green': [
        [0r, 0r, 0r],
        [1r, 1r, 1r]
      ],
      'blue': [
        [0.000, 0.000, 0.000],
        [0.500, 1.000, 1.000],
        [0.735, 0.000, 0.000],
        [1.000, 1.000, 1.000]
      ]
    }

    data[:gist_yarg] = {
      red:   ->(x) { 1r - x },
      green: ->(x) { 1r - x },
      blue:  ->(x) { 1r - x }
    }

    # This bipolar colormap was generated from CoolWarmFloat33.csv of
    # "Diverging Color Maps for Scientific Visualization" by Kenneth Moreland.
    # <http://www.kennethmoreland.com/color-maps/>
    data[:coolwarm] = load_colormap_data("coolwarm")

    # Implementation of Carey Rappaport's CMRmap.
    # See `A Color Map for Effective Black-and-White Rendering of Color-Scale
    # Images' by Carey Rappaport
    # http://www.mathworks.com/matlabcentral/fileexchange/2662-cmrmap-m
    data[:CMRmap] = {
      'red': [
        [0.000, 0.00, 0.00],
        [0.125, 0.15, 0.15],
        [0.250, 0.30, 0.30],
        [0.375, 0.60, 0.60],
        [0.500, 1.00, 1.00],
        [0.625, 0.90, 0.90],
        [0.750, 0.90, 0.90],
        [0.875, 0.90, 0.90],
        [1.000, 1.00, 1.00]
      ],
      'green': [
        [0.000, 0.00, 0.00],
        [0.125, 0.15, 0.15],
        [0.250, 0.15, 0.15],
        [0.375, 0.20, 0.20],
        [0.500, 0.25, 0.25],
        [0.625, 0.50, 0.50],
        [0.750, 0.75, 0.75],
        [0.875, 0.90, 0.90],
        [1.000, 1.00, 1.00]
      ],
      'blue': [
        [0.000, 0.00, 0.00],
        [0.125, 0.50, 0.50],
        [0.250, 0.75, 0.75],
        [0.375, 0.50, 0.50],
        [0.500, 0.15, 0.15],
        [0.625, 0.00, 0.00],
        [0.750, 0.10, 0.10],
        [0.875, 0.50, 0.50],
        [1.000, 1.00, 1.00]
      ]
    }

    # An MIT licensed, colorblind-friendly heatmap from Wistia:
    #   https://github.com/wistia/heatmap-palette
    #   http://wistia.com/blog/heatmaps-for-colorblindness
    #
    # >>> import matplotlib.colors as c
    # >>> colors = ["#e4ff7a", "#ffe81a", "#ffbd00", "#ffa000", "#fc7f00"]
    # >>> cm = c.LinearSegmentedColormap.from_list('wistia', colors)
    # >>> _wistia_data = cm._segmentdata
    # >>> del _wistia_data['alpha']
    #
    data[:Wistia] = {
      red: [
        [0.0, 0.8941176470588236, 0.8941176470588236],
        [0.25, 1.0, 1.0],
        [0.5, 1.0, 1.0],
        [0.75, 1.0, 1.0],
        [1.0, 0.9882352941176471, 0.9882352941176471]
      ],
      green: [
        [0.0, 1.0, 1.0],
        [0.25, 0.9098039215686274, 0.9098039215686274],
        [0.5, 0.7411764705882353, 0.7411764705882353],
        [0.75, 0.6274509803921569, 0.6274509803921569],
        [1.0, 0.4980392156862745, 0.4980392156862745]
      ],
      blue: [
        [0.0, 0.47843137254901963, 0.47843137254901963],
        [0.25, 0.10196078431372549, 0.10196078431372549],
        [0.5, 0.0, 0.0],
        [0.75, 0.0, 0.0],
        [1.0, 0.0, 0.0]
      ],
    }

    # Categorical palettes from Vega:
    # https://github.com/vega/vega/wiki/Scales
    # (divided by 255)
    #

    data[:tab10] = {listed:[
      [0.12156862745098039, 0.4666666666666667,  0.7058823529411765  ],  # 1f77b4
      [1.0,                 0.4980392156862745,  0.054901960784313725],  # ff7f0e
      [0.17254901960784313, 0.6274509803921569,  0.17254901960784313 ],  # 2ca02c
      [0.8392156862745098,  0.15294117647058825, 0.1568627450980392  ],  # d62728
      [0.5803921568627451,  0.403921568627451,   0.7411764705882353  ],  # 9467bd
      [0.5490196078431373,  0.33725490196078434, 0.29411764705882354 ],  # 8c564b
      [0.8901960784313725,  0.4666666666666667,  0.7607843137254902  ],  # e377c2
      [0.4980392156862745,  0.4980392156862745,  0.4980392156862745  ],  # 7f7f7f
      [0.7372549019607844,  0.7411764705882353,  0.13333333333333333 ],  # bcbd22
      [0.09019607843137255, 0.7450980392156863,  0.8117647058823529  ]   # 17becf
    ]}

    data[:tab20] = {listed:[
      [0.12156862745098039, 0.4666666666666667,  0.7058823529411765  ],  # 1f77b4
      [0.6823529411764706,  0.7803921568627451,  0.9098039215686274  ],  # aec7e8
      [1.0,                 0.4980392156862745,  0.054901960784313725],  # ff7f0e
      [1.0,                 0.7333333333333333,  0.47058823529411764 ],  # ffbb78
      [0.17254901960784313, 0.6274509803921569,  0.17254901960784313 ],  # 2ca02c
      [0.596078431372549,   0.8745098039215686,  0.5411764705882353  ],  # 98df8a
      [0.8392156862745098,  0.15294117647058825, 0.1568627450980392  ],  # d62728
      [1.0,                 0.596078431372549,   0.5882352941176471  ],  # ff9896
      [0.5803921568627451,  0.403921568627451,   0.7411764705882353  ],  # 9467bd
      [0.7725490196078432,  0.6901960784313725,  0.8352941176470589  ],  # c5b0d5
      [0.5490196078431373,  0.33725490196078434, 0.29411764705882354 ],  # 8c564b
      [0.7686274509803922,  0.611764705882353,   0.5803921568627451  ],  # c49c94
      [0.8901960784313725,  0.4666666666666667,  0.7607843137254902  ],  # e377c2
      [0.9686274509803922,  0.7137254901960784,  0.8235294117647058  ],  # f7b6d2
      [0.4980392156862745,  0.4980392156862745,  0.4980392156862745  ],  # 7f7f7f
      [0.7803921568627451,  0.7803921568627451,  0.7803921568627451  ],  # c7c7c7
      [0.7372549019607844,  0.7411764705882353,  0.13333333333333333 ],  # bcbd22
      [0.8588235294117647,  0.8588235294117647,  0.5529411764705883  ],  # dbdb8d
      [0.09019607843137255, 0.7450980392156863,  0.8117647058823529  ],  # 17becf
      [0.6196078431372549,  0.8549019607843137,  0.8980392156862745  ]   # 9edae5
    ]}

    data[:tab20b] = {listed:[
      [0.2235294117647059,  0.23137254901960785, 0.4745098039215686 ],  # 393b79
      [0.3215686274509804,  0.32941176470588235, 0.6392156862745098 ],  # 5254a3
      [0.4196078431372549,  0.43137254901960786, 0.8117647058823529 ],  # 6b6ecf
      [0.611764705882353,   0.6196078431372549,  0.8705882352941177 ],  # 9c9ede
      [0.38823529411764707, 0.4745098039215686,  0.2235294117647059 ],  # 637939
      [0.5490196078431373,  0.6352941176470588,  0.3215686274509804 ],  # 8ca252
      [0.7098039215686275,  0.8117647058823529,  0.4196078431372549 ],  # b5cf6b
      [0.807843137254902,   0.8588235294117647,  0.611764705882353  ],  # cedb9c
      [0.5490196078431373,  0.42745098039215684, 0.19215686274509805],  # 8c6d31
      [0.7411764705882353,  0.6196078431372549,  0.2235294117647059 ],  # bd9e39
      [0.9058823529411765,  0.7294117647058823,  0.3215686274509804 ],  # e7ba52
      [0.9058823529411765,  0.796078431372549,   0.5803921568627451 ],  # e7cb94
      [0.5176470588235295,  0.23529411764705882, 0.2235294117647059 ],  # 843c39
      [0.6784313725490196,  0.28627450980392155, 0.2901960784313726 ],  # ad494a
      [0.8392156862745098,  0.3803921568627451,  0.4196078431372549 ],  # d6616b
      [0.9058823529411765,  0.5882352941176471,  0.611764705882353  ],  # e7969c
      [0.4823529411764706,  0.2549019607843137,  0.45098039215686275],  # 7b4173
      [0.6470588235294118,  0.3176470588235294,  0.5803921568627451 ],  # a55194
      [0.807843137254902,   0.42745098039215684, 0.7411764705882353 ],  # ce6dbd
      [0.8705882352941177,  0.6196078431372549,  0.8392156862745098 ]   # de9ed6
    ]}

    data[:tab20c] = {listed:[
      [0.19215686274509805, 0.5098039215686274,  0.7411764705882353  ],  # 3182bd
      [0.4196078431372549,  0.6823529411764706,  0.8392156862745098  ],  # 6baed6
      [0.6196078431372549,  0.792156862745098,   0.8823529411764706  ],  # 9ecae1
      [0.7764705882352941,  0.8588235294117647,  0.9372549019607843  ],  # c6dbef
      [0.9019607843137255,  0.3333333333333333,  0.050980392156862744],  # e6550d
      [0.9921568627450981,  0.5529411764705883,  0.23529411764705882 ],  # fd8d3c
      [0.9921568627450981,  0.6823529411764706,  0.4196078431372549  ],  # fdae6b
      [0.9921568627450981,  0.8156862745098039,  0.6352941176470588  ],  # fdd0a2
      [0.19215686274509805, 0.6392156862745098,  0.32941176470588235 ],  # 31a354
      [0.4549019607843137,  0.7686274509803922,  0.4627450980392157  ],  # 74c476
      [0.6313725490196078,  0.8509803921568627,  0.6078431372549019  ],  # a1d99b
      [0.7803921568627451,  0.9137254901960784,  0.7529411764705882  ],  # c7e9c0
      [0.4588235294117647,  0.4196078431372549,  0.6941176470588235  ],  # 756bb1
      [0.6196078431372549,  0.6039215686274509,  0.7843137254901961  ],  # 9e9ac8
      [0.7372549019607844,  0.7411764705882353,  0.8627450980392157  ],  # bcbddc
      [0.8549019607843137,  0.8549019607843137,  0.9215686274509803  ],  # dadaeb
      [0.38823529411764707, 0.38823529411764707, 0.38823529411764707 ],  # 636363
      [0.5882352941176471,  0.5882352941176471,  0.5882352941176471  ],  # 969696
      [0.7411764705882353,  0.7411764705882353,  0.7411764705882353  ],  # bdbdbd
      [0.8509803921568627,  0.8509803921568627,  0.8509803921568627  ]   # d9d9d9
    ]}

    sorted_names = data.each_key.sort do |a_key, b_key|
      a_data = data[a_key]
      b_data = data[b_key]
      a_listed = a_data.is_a?(Hash) && a_data.key?(:listed)
      b_listed = a_data.is_a?(Hash) && a_data.key?(:listed)
      case
      when !a_listed && b_listed
        -1
      when a_listed && !b_listed
        1
      else
        a_key <=> b_key
      end
    end

    sorted_names.each do |name|
      spec = data[name]
      cmap = case
             when spec.is_a?(Hash) && spec.key?(:red)
               LinearSegmentedColormap.new(name.to_s, spec, n_colors: LUT_SIZE)
             when spec.is_a?(Hash) && spec.key?(:listed)
               colors = spec[:listed].map { |rgb| RGB.new(*rgb) }
               ListedColormap.new(colors, name: name.to_s)
             else
               case spec[0][1]
               when Array
                 colors = spec.map {|v, rgb| [v, RGB.new(*rgb)] }
               else
                 colors = spec.map {|rgb| RGB.new(*rgb) }
               end
               LinearSegmentedColormap.new_from_list(name.to_s, colors, n_colors: LUT_SIZE)
             end
      BUILTIN_COLORMAPS[cmap.name] = cmap
    end
  end
end
