"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CustomPanelsAdaptor = void 0;
var _uuid = require("uuid");
var _custom_panels = require("../../../common/constants/custom_panels");
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
class CustomPanelsAdaptor {
  constructor() {
    // index a panel
    _defineProperty(this, "indexPanel", async function (client, panelBody) {
      try {
        const response = await client.callAsCurrentUser('observability.createObject', {
          body: {
            operationalPanel: panelBody
          }
        });
        return response;
      } catch (error) {
        throw new Error('Index Panel Error:' + error);
      }
    });
    // update a panel
    _defineProperty(this, "updatePanel", async function (client, panelId, updatePanelBody) {
      try {
        const response = await client.callAsCurrentUser('observability.updateObjectById', {
          objectId: panelId,
          body: {
            operationalPanel: updatePanelBody
          }
        });
        return response;
      } catch (error) {
        throw new Error('Update Panel Error:' + error);
      }
    });
    // fetch a panel by id
    _defineProperty(this, "getPanel", async function (client, panelId) {
      try {
        const response = await client.callAsCurrentUser('observability.getObjectById', {
          objectId: panelId
        });
        return response.observabilityObjectList[0];
      } catch (error) {
        throw new Error('Get Panel Error:' + error);
      }
    });
    // gets list of panels stored in index
    _defineProperty(this, "viewPanelList", async function (client) {
      try {
        const response = await client.callAsCurrentUser('observability.getObject', {
          objectType: 'operationalPanel',
          maxItems: 10000
        });
        return response.observabilityObjectList.filter(panel => !panel.operationalPanel.applicationId).map(panel => ({
          name: panel.operationalPanel.name,
          id: panel.objectId,
          dateCreated: panel.createdTimeMs,
          dateModified: panel.lastUpdatedTimeMs
        }));
      } catch (error) {
        throw new Error('View Panel List Error:' + error);
      }
    });
    // Delete a panel by Id
    _defineProperty(this, "deletePanel", async function (client, panelId) {
      try {
        const response = await client.callAsCurrentUser('observability.deleteObjectById', {
          objectId: panelId
        });
        return {
          status: 'OK',
          message: response
        };
      } catch (error) {
        throw new Error('Delete Panel Error:' + error);
      }
    });
    // Delete a panel by Id
    _defineProperty(this, "deletePanelList", async function (client, panelIdList) {
      try {
        const response = await client.callAsCurrentUser('observability.deleteObjectByIdList', {
          objectIdList: panelIdList
        });
        return {
          status: 'OK',
          message: response
        };
      } catch (error) {
        throw new Error('Delete Panel List Error:' + error);
      }
    });
    // Create a new Panel
    _defineProperty(this, "createNewPanel", async (client, panelName, appId) => {
      const panelBody = {
        name: panelName,
        visualizations: [],
        timeRange: {
          to: 'now',
          from: 'now-1d'
        },
        queryFilter: {
          query: '',
          language: 'ppl'
        }
      };
      if (appId) {
        panelBody.applicationId = appId;
        panelBody.timeRange = {
          to: 'now',
          from: 'now-24h'
        };
      }
      try {
        const response = await this.indexPanel(client, panelBody);
        return response.objectId;
      } catch (error) {
        throw new Error('Create Panel Error:' + error);
      }
    });
    // Rename an existing panel
    _defineProperty(this, "renamePanel", async (client, panelId, panelName) => {
      const updatePanelBody = {
        name: panelName
      };
      try {
        const response = await this.updatePanel(client, panelId, updatePanelBody);
        return response.objectId;
      } catch (error) {
        throw new Error('Rename Panel Error:' + error);
      }
    });
    // Clone an existing panel
    _defineProperty(this, "clonePanel", async (client, panelId, panelName) => {
      const updatePanelBody = {
        name: panelName
      };
      try {
        const getPanel = await this.getPanel(client, panelId);
        const clonePanelBody = {
          name: panelName,
          visualizations: getPanel.operationalPanel.visualizations,
          timeRange: getPanel.operationalPanel.timeRange,
          queryFilter: getPanel.operationalPanel.queryFilter
        };
        const indexResponse = await this.indexPanel(client, clonePanelBody);
        const getClonedPanel = await this.getPanel(client, indexResponse.objectId);
        return {
          clonePanelId: getClonedPanel.objectId,
          dateCreated: getClonedPanel.createdTimeMs,
          dateModified: getClonedPanel.lastUpdatedTimeMs
        };
      } catch (error) {
        throw new Error('Clone Panel Error:' + error);
      }
    });
    // Add filters to an existing panel
    _defineProperty(this, "addPanelFilter", async (client, panelId, query, language, to, from) => {
      const updatePanelBody = {
        timeRange: {
          to,
          from
        },
        queryFilter: {
          query,
          language
        }
      };
      try {
        const response = await this.updatePanel(client, panelId, updatePanelBody);
        return response.objectId;
      } catch (error) {
        throw new Error('Add Panel Filter Error:' + error);
      }
    });
    // parses fetched saved visualization
    _defineProperty(this, "parseSavedVisualizations", visualization => {
      return {
        id: visualization.objectId,
        name: visualization.savedVisualization.name,
        query: visualization.savedVisualization.query,
        type: visualization.savedVisualization.type,
        timeField: visualization.savedVisualization.selected_timestamp.name,
        selected_date_range: visualization.savedVisualization.selected_date_range,
        selected_fields: visualization.savedVisualization.selected_fields,
        userConfigs: visualization.savedVisualization.hasOwnProperty('userConfigs') || visualization.savedVisualization.hasOwnProperty('user_configs') ? JSON.parse(visualization.savedVisualization.user_configs || visualization.savedVisualization.userConfigs) : {},
        subType: visualization.savedVisualization.hasOwnProperty('subType') ? visualization.savedVisualization.subType : '',
        units_of_measure: visualization.savedVisualization.hasOwnProperty('units_of_measure') ? visualization.savedVisualization.units_of_measure : '',
        ...(visualization.savedVisualization.application_id ? {
          application_id: visualization.savedVisualization.application_id
        } : {})
      };
    });
    // gets all saved visualizations
    _defineProperty(this, "getAllSavedVisualizations", async client => {
      try {
        const response = await client.callAsCurrentUser('observability.getObject', {
          objectType: 'savedVisualization'
        });
        return response.observabilityObjectList.map(visualization => this.parseSavedVisualizations(visualization));
      } catch (error) {
        throw new Error('View Saved Visualizations Error:' + error);
      }
    });
    // gets list of savedVisualizations by Id
    _defineProperty(this, "getSavedVisualizationById", async (client, savedVisualizationId) => {
      try {
        const response = await client.callAsCurrentUser('observability.getObjectById', {
          objectId: savedVisualizationId
        });
        const visualization = response.observabilityObjectList[0];
        return this.parseSavedVisualizations(visualization);
      } catch (error) {
        throw new Error('Fetch Saved Visualizations By Id Error:' + error);
      }
    });
    // Get All Visualizations from a Panel
    _defineProperty(this, "getVisualizations", async (client, panelId) => {
      try {
        const response = await client.callAsCurrentUser('observability.getObjectById', {
          objectId: panelId
        });
        return response.observabilityObjectList[0].operationalPanel.visualizations;
      } catch (error) {
        throw new Error('Get Visualizations Error:' + error);
      }
    });
    _defineProperty(this, "calculatOverlapArea", (bb1, bb2) => {
      const xLeft = Math.max(bb1.x1, bb2.x1);
      const yTop = Math.max(bb1.y1, bb2.y1);
      const xRight = Math.min(bb1.x2, bb2.x2);
      const yBottom = Math.min(bb1.y2, bb2.y2);
      if (xRight < xLeft || yBottom < yTop) return 0;
      return (xRight - xLeft) * (yBottom - yTop);
    });
    _defineProperty(this, "getTotalOverlapArea", panelVisualizations => {
      const newVizBox = {
        x1: 0,
        y1: 0,
        x2: 6,
        y2: 4
      };
      const currentVizBoxes = panelVisualizations.map(visualization => {
        return {
          x1: visualization.x,
          y1: visualization.y,
          x2: visualization.x + visualization.w,
          y2: visualization.y + visualization.h
        };
      });
      let isOverlapping = 0;
      currentVizBoxes.map(viz => {
        isOverlapping += this.calculatOverlapArea(viz, newVizBox);
      });
      return isOverlapping;
    });
    // We want to check if the new visualization being added, can be placed at { x: 0, y: 0, w: 6, h: 4 };
    // To check this we try to calculate overlap between all the current visualizations and new visualization
    // if there is no overalap (i.e Total Overlap Area is 0), we place the new viz. in default position
    // else, we add it to the bottom of the panel
    _defineProperty(this, "getNewVizDimensions", panelVisualizations => {
      let maxY = 0;
      let maxYH = 0;

      // check if we can place the new visualization at default location
      if (this.getTotalOverlapArea(panelVisualizations) === 0) {
        return {
          x: 0,
          y: 0,
          w: 6,
          h: 4
        };
      }

      // else place the new visualization at the bottom of the panel
      panelVisualizations.map(panelVisualization => {
        if (panelVisualization.y >= maxY) {
          maxY = panelVisualization.y;
          maxYH = panelVisualization.h;
        }
      });
      return {
        x: 0,
        y: maxY + maxYH,
        w: 6,
        h: 4
      };
    });
    // Add Visualization in the  Panel
    _defineProperty(this, "addVisualization", async (client, panelId, savedVisualizationId, oldVisualizationId) => {
      try {
        const allPanelVisualizations = await this.getVisualizations(client, panelId);
        let newDimensions;
        let visualizationsList = [];
        if (oldVisualizationId === undefined) {
          newDimensions = this.getNewVizDimensions(allPanelVisualizations);
          visualizationsList = allPanelVisualizations;
        } else {
          allPanelVisualizations.map(visualization => {
            if (visualization.id !== oldVisualizationId) {
              visualizationsList.push(visualization);
            } else {
              newDimensions = {
                x: visualization.x,
                y: visualization.y,
                w: visualization.w,
                h: visualization.h
              };
            }
          });
        }
        const newPanelVisualizations = [...visualizationsList, {
          id: 'panel_viz_' + (0, _uuid.v4)(),
          savedVisualizationId,
          ...newDimensions
        }];
        const updatePanelResponse = await this.updatePanel(client, panelId, {
          visualizations: newPanelVisualizations
        });
        return newPanelVisualizations;
      } catch (error) {
        throw new Error('Add/Replace Visualization Error:' + error);
      }
    });
    // Add Multiple visualizations in a Panel
    _defineProperty(this, "addMultipleVisualizations", async (client, panelId, savedVisualizationIds) => {
      try {
        const allPanelVisualizations = await this.getVisualizations(client, panelId);
        let newDimensions;
        let visualizationsList = [...allPanelVisualizations];
        savedVisualizationIds.map(savedVisualizationId => {
          newDimensions = this.getNewVizDimensions(visualizationsList);
          visualizationsList = [...visualizationsList, {
            id: 'panel_viz_' + (0, _uuid.v4)(),
            savedVisualizationId,
            ...newDimensions
          }];
        });
        const updatePanelResponse = await this.updatePanel(client, panelId, {
          visualizations: visualizationsList
        });
        return visualizationsList;
      } catch (error) {
        throw new Error('Add/Replace Visualization Error:' + error);
      }
    });
    // Edits all Visualizations in the Panel
    _defineProperty(this, "editVisualization", async (client, panelId, visualizationParams) => {
      try {
        const allPanelVisualizations = await this.getVisualizations(client, panelId);
        const filteredPanelVisualizations = [];
        for (let i = 0; i < allPanelVisualizations.length; i++) {
          for (let j = 0; j < visualizationParams.length; j++) {
            if (allPanelVisualizations[i].id === visualizationParams[j].i) {
              filteredPanelVisualizations.push({
                ...allPanelVisualizations[i],
                x: visualizationParams[j].x,
                y: visualizationParams[j].y,
                w: visualizationParams[j].w,
                h: visualizationParams[j].h
              });
            }
          }
        }
        const updatePanelResponse = await this.updatePanel(client, panelId, {
          visualizations: filteredPanelVisualizations
        });
        return filteredPanelVisualizations;
      } catch (error) {
        throw new Error('Edit Visualizations Error:' + error);
      }
    });
    // Create Sample Panels
    _defineProperty(this, "addSamplePanels", async (client, savedVisualizationIds) => {
      try {
        const panelBody = (0, _custom_panels.createDemoPanel)(savedVisualizationIds);
        const indexResponse = await this.indexPanel(client, panelBody);
        const fetchPanel = await this.getPanel(client, indexResponse.objectId);
        const fetchResponse = {
          name: fetchPanel.operationalPanel.name,
          id: fetchPanel.objectId,
          dateCreated: fetchPanel.createdTimeMs,
          dateModified: fetchPanel.lastUpdatedTimeMs
        };
        return [fetchResponse];
      } catch (error) {
        throw new Error('Create Panel Error:' + error);
      }
    });
  }
}
exports.CustomPanelsAdaptor = CustomPanelsAdaptor;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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