"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerDataConnectionsRoute = registerDataConnectionsRoute;
var _configSchema = require("@osd/config-schema");
var _shared = require("../../../common/constants/shared");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function registerDataConnectionsRoute(router, dataSourceEnabled) {
  router.get({
    path: `${_shared.DATACONNECTIONS_BASE}/{name}`,
    validate: {
      params: _configSchema.schema.object({
        name: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    try {
      const dataConnectionsresponse = await context.observability_plugin.observabilityClient.asScoped(request).callAsCurrentUser('ppl.getDataConnectionById', {
        dataconnection: request.params.name
      });
      return response.ok({
        body: dataConnectionsresponse
      });
    } catch (error) {
      console.error('Issue in fetching data connection:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.delete({
    path: `${_shared.DATACONNECTIONS_BASE}/{name}`,
    validate: {
      params: _configSchema.schema.object({
        name: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    try {
      const dataConnectionsresponse = await context.observability_plugin.observabilityClient.asScoped(request).callAsCurrentUser('ppl.deleteDataConnection', {
        dataconnection: request.params.name
      });
      return response.ok({
        body: dataConnectionsresponse
      });
    } catch (error) {
      console.error('Issue in deleting data connection:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: `${_shared.DATACONNECTIONS_BASE}${_shared.EDIT}`,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        allowedRoles: _configSchema.schema.arrayOf(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      const dataConnectionsresponse = await context.observability_plugin.observabilityClient.asScoped(request).callAsCurrentUser('ppl.modifyDataConnection', {
        body: {
          name: request.body.name,
          allowedRoles: request.body.allowedRoles
        }
      });
      return response.ok({
        body: dataConnectionsresponse
      });
    } catch (error) {
      console.error('Issue in modifying data connection:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: `${_shared.DATACONNECTIONS_BASE}${_shared.EDIT}${_shared.DATACONNECTIONS_UPDATE_STATUS}`,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        status: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    try {
      const dataConnectionsresponse = await context.observability_plugin.observabilityClient.asScoped(request).callAsCurrentUser('ppl.modifyDataConnection', {
        body: {
          name: request.body.name,
          status: request.body.status
        }
      });
      return response.ok({
        body: dataConnectionsresponse
      });
    } catch (error) {
      console.error('Issue in modifying data connection:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: `${_shared.DATACONNECTIONS_BASE}`,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        connector: _configSchema.schema.string(),
        allowedRoles: _configSchema.schema.arrayOf(_configSchema.schema.string()),
        properties: _configSchema.schema.any()
      })
    }
  }, async (context, request, response) => {
    try {
      const dataConnectionsresponse = await context.observability_plugin.observabilityClient.asScoped(request).callAsCurrentUser('ppl.createDataSource', {
        body: {
          name: request.body.name,
          connector: request.body.connector,
          allowedRoles: request.body.allowedRoles,
          properties: request.body.properties
        }
      });
      return response.ok({
        body: dataConnectionsresponse
      });
    } catch (error) {
      console.error('Issue in creating data source:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.response
      });
    }
  });
  router.get({
    path: `${_shared.DATACONNECTIONS_BASE}`,
    validate: false
  }, async (context, request, response) => {
    try {
      const dataConnectionsresponse = await context.observability_plugin.observabilityClient.asScoped(request).callAsCurrentUser('ppl.getDataConnections');
      return response.ok({
        body: dataConnectionsresponse
      });
    } catch (error) {
      console.error('Issue in fetching data sources:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.response
      });
    }
  });
  router.get({
    path: `${_shared.DATACONNECTIONS_BASE}/dataSourceMDSId={dataSourceMDSId?}`,
    validate: {
      params: _configSchema.schema.object({
        dataSourceMDSId: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        }))
      })
    }
  }, async (context, request, response) => {
    const dataSourceMDSId = request.params.dataSourceMDSId;
    try {
      let dataConnectionsresponse;
      if (dataSourceEnabled && dataSourceMDSId) {
        const client = await context.dataSource.opensearch.legacy.getClient(dataSourceMDSId);
        dataConnectionsresponse = await client.callAPI('ppl.getDataConnections');
      } else {
        dataConnectionsresponse = await context.observability_plugin.observabilityClient.asScoped(request).callAsCurrentUser('ppl.getDataConnections');
      }
      return response.ok({
        body: dataConnectionsresponse
      });
    } catch (error) {
      console.error('Issue in fetching data sources:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.response
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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