"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.handleWithCallback = void 0;
exports.registerIntegrationsRoute = registerIntegrationsRoute;
var _configSchema = require("@osd/config-schema");
var mime = _interopRequireWildcard(require("mime"));
var _sanitizeFilename = _interopRequireDefault(require("sanitize-filename"));
var _shared = require("../../../common/constants/shared");
var _integrations_manager = require("../../adaptors/integrations/integrations_manager");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && Object.prototype.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

/**
 * Handle an `OpenSearchDashboardsRequest` using the provided `callback` function.
 * This is a convenience method that handles common error handling and response formatting.
 * The callback must accept a `IntegrationsAdaptor` as its first argument.
 *
 * If the callback throws an error,
 * the `OpenSearchDashboardsResponse` will be formatted using the error's `statusCode` and `message` properties.
 * Otherwise, the callback's return value will be formatted in a JSON object under the `data` field.
 *
 * @param {IntegrationsAdaptor} adaptor The adaptor instance to use for making requests.
 * @param {OpenSearchDashboardsResponseFactory} response The factory to use for creating responses.
 * @callback callback A callback that will invoke a request on a provided adaptor.
 * @returns {Promise<OpenSearchDashboardsResponse>} An `OpenSearchDashboardsResponse` with the return data from the callback.
 */
const handleWithCallback = async (adaptor, response, callback) => {
  try {
    const data = await callback(adaptor);
    return response.ok({
      body: {
        data
      }
    });
  } catch (err) {
    console.error(`handleWithCallback: callback failed with error "${err.message}"`);
    return response.custom({
      statusCode: err.statusCode || 500,
      body: err.message
    });
  }
};
exports.handleWithCallback = handleWithCallback;
const getAdaptor = (context, _request) => {
  return new _integrations_manager.IntegrationsManager(context.core.savedObjects.client);
};
function registerIntegrationsRoute(router) {
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/repository`,
    validate: false
  }, async (context, request, response) => {
    const adaptor = getAdaptor(context, request);
    return handleWithCallback(adaptor, response, async a => a.getIntegrationTemplates());
  });
  router.post({
    path: `${_shared.INTEGRATIONS_BASE}/store/{templateName}`,
    validate: {
      params: _configSchema.schema.object({
        templateName: _configSchema.schema.string()
      }),
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        indexPattern: _configSchema.schema.string(),
        workflows: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
        dataSource: _configSchema.schema.maybe(_configSchema.schema.string()),
        tableName: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const adaptor = getAdaptor(context, request);
    return handleWithCallback(adaptor, response, async a => {
      return a.loadIntegrationInstance(request.params.templateName, request.body.name, request.body.indexPattern, request.body.workflows, request.body.dataSource, request.body.tableName);
    });
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/repository/{name}`,
    validate: {
      params: _configSchema.schema.object({
        name: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = getAdaptor(context, request);
    return handleWithCallback(adaptor, response, async a => (await a.getIntegrationTemplates({
      name: request.params.name
    })).hits[0]);
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/repository/{id}/static/{path}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string(),
        path: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = getAdaptor(context, request);
    try {
      const requestPath = (0, _sanitizeFilename.default)(request.params.path);
      const result = await adaptor.getStatic(request.params.id, requestPath);
      return response.ok({
        headers: {
          'Content-Type': mime.getType(request.params.path)
        },
        body: result
      });
    } catch (err) {
      return response.custom({
        statusCode: err.statusCode ? err.statusCode : 500,
        body: err.message
      });
    }
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/repository/{id}/schema`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = getAdaptor(context, request);
    return handleWithCallback(adaptor, response, async a => a.getSchemas(request.params.id));
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/repository/{id}/assets`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = getAdaptor(context, request);
    return handleWithCallback(adaptor, response, async a => a.getAssets(request.params.id));
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/repository/{id}/data`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = getAdaptor(context, request);
    return handleWithCallback(adaptor, response, async a => a.getSampleData(request.params.id));
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/store`,
    validate: false
  }, async (context, request, response) => {
    const adaptor = getAdaptor(context, request);
    return handleWithCallback(adaptor, response, async a => a.getIntegrationInstances({}));
  });
  router.delete({
    path: `${_shared.INTEGRATIONS_BASE}/store/{id}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = getAdaptor(context, request);
    return handleWithCallback(adaptor, response, async a => a.deleteIntegrationInstance(request.params.id));
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/store/{id}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = getAdaptor(context, request);
    return handleWithCallback(adaptor, response, async a => a.getIntegrationInstance({
      id: request.params.id
    }));
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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