'use strict';
Object.defineProperty(exports, "__esModule", { value: true });
exports.plugin = exports.ExpressPlugin = exports.kMiddlewareStack = void 0;
const core_1 = require("@opencensus/core");
const shimmer = require("shimmer");
const semver = require("semver");
exports.kMiddlewareStack = Symbol('express-middleware-stack');
class ExpressPlugin extends core_1.BasePlugin {
    constructor(moduleName) {
        super(moduleName);
        this.kPatched = Symbol('express-layer-patched');
    }
    applyPatch() {
        this.logger.debug('Patched express');
        if (!semver.satisfies(this.version, '^4.0.0')) {
            this.logger.debug('express version %s not supported - aborting...', this.version);
            return this.moduleExports;
        }
        if (this.moduleExports) {
            const routerProto = semver.satisfies(this.version, '^5')
                ? (this.moduleExports.Router && this.moduleExports.Router.prototype)
                : this.moduleExports.Router;
            const plugin = this;
            this.logger.debug('patching express.Router.prototype.route');
            shimmer.wrap(routerProto, 'route', (original) => {
                return function route_trace(path) {
                    const route = original.apply(this, arguments);
                    const layer = this.stack[this.stack.length - 1];
                    plugin.applyLayerPatch(layer, path);
                    return route;
                };
            });
            this.logger.debug('patching express.Router.prototype.use');
            shimmer.wrap(routerProto, 'use', (original) => {
                return function use(path) {
                    const route = original.apply(this, arguments);
                    const layer = this.stack[this.stack.length - 1];
                    plugin.applyLayerPatch(layer, path);
                    return route;
                };
            });
            this.logger.debug('patching express.Application.use');
            shimmer.wrap(this.moduleExports.application, 'use', (original) => {
                return function use(path) {
                    const route = original.apply(this, arguments);
                    const layer = this._router.stack[this._router.stack.length - 1];
                    plugin.applyLayerPatch(layer, path);
                    return route;
                };
            });
        }
        return this.moduleExports;
    }
    applyUnpatch() {
        if (!semver.satisfies(this.version, '^4.0.0')) {
            return this.moduleExports;
        }
        const routerProto = semver.satisfies(this.version, '^5')
            ? (this.moduleExports.Router && this.moduleExports.Router.prototype)
            : this.moduleExports.Router;
        shimmer.unwrap(routerProto, 'use');
        shimmer.unwrap(routerProto, 'route');
        shimmer.unwrap(this.moduleExports.application, 'use');
    }
    applyLayerPatch(layer, layerPath) {
        const plugin = this;
        if (layer[this.kPatched] === true)
            return;
        layer[this.kPatched] = true;
        plugin.logger.debug('patching express.Router.Layer.handle');
        shimmer.wrap(layer, 'handle', function (original) {
            if (original.length !== 4) {
                return function (req, res, next) {
                    plugin.safePush(req, exports.kMiddlewareStack, layerPath);
                    let spanName = `Middleware - ${layer.name}`;
                    if (layer.route) {
                        spanName = `Route - ${layer.route.path}`;
                    }
                    else if (layer.name === 'router') {
                        spanName = `Router - ${layerPath}`;
                    }
                    const span = plugin.tracer.startChildSpan(spanName, core_1.SpanKind.CLIENT);
                    if (span === null)
                        return original.apply(this, arguments);
                    arguments[2] = function () {
                        if (!(req.route && arguments[0] instanceof Error)) {
                            req[exports.kMiddlewareStack].pop();
                        }
                        return plugin.patchEnd(span, next)();
                    };
                    return original.apply(this, arguments);
                };
            }
            return function (_err, req, res, next) {
                return original.apply(this, arguments);
            };
        });
    }
    safePush(obj, prop, value) {
        if (!obj[prop])
            obj[prop] = [];
        obj[prop].push(value);
    }
    patchEnd(span, callback) {
        const plugin = this;
        const patchedEnd = function (err) {
            if (plugin.options.detailedCommands === true && err instanceof Error) {
                span.addAttribute('error', err.message);
            }
            if (span.ended === false) {
                span.end();
            }
            return callback.apply(this, arguments);
        };
        return this.tracer.wrap(patchedEnd);
    }
}
exports.ExpressPlugin = ExpressPlugin;
const plugin = new ExpressPlugin('express');
exports.plugin = plugin;
//# sourceMappingURL=data:application/json;base64,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