'use strict';
Object.defineProperty(exports, "__esModule", { value: true });
exports.plugin = exports.RedisPlugin = void 0;
const core_1 = require("@opencensus/core");
const shimmer = require("shimmer");
const semver = require("semver");
class RedisPlugin extends core_1.BasePlugin {
    constructor(moduleName) {
        super(moduleName);
    }
    applyPatch() {
        this.logger.debug('Patched redis');
        if (semver.lt(this.version, '2.4.0')) {
            this.logger.info('disabling redis plugin because version isnt supported');
            return this.moduleExports;
        }
        if (this.moduleExports.RedisClient) {
            this.logger.debug('patching redis.RedisClient.prototype.create_stream');
            shimmer.wrap(this.moduleExports.RedisClient.prototype, 'create_stream', this.getPatchCreateStream());
            this.logger.debug('patching redis.RedisClient.prototype.internal_send');
            shimmer.wrap(this.moduleExports.RedisClient.prototype, 'internal_send_command', this.getPatchSendCommand());
            this.logger.debug('patching redis.RedisClient.prototype.createClient');
            shimmer.wrap(this.moduleExports, 'createClient', this.getPatchCreateClient());
        }
        return this.moduleExports;
    }
    applyUnpatch() {
        if (semver.lt(this.version, '2.4.0'))
            return;
        shimmer.unwrap(this.moduleExports.RedisClient.prototype, 'internal_send_command');
        shimmer.unwrap(this.moduleExports, 'createClient');
        shimmer.unwrap(this.moduleExports.RedisClient.prototype, 'create_stream');
    }
    getPatchCreateStream() {
        const plugin = this;
        return function createStreamWrap(original) {
            return function create_stream_trace() {
                if (!this.stream) {
                    Object.defineProperty(this, 'stream', {
                        get: function () { return this._patched_redis_stream; },
                        set: function (val) {
                            plugin.tracer.wrapEmitter(val);
                            this._patched_redis_stream = val;
                        }
                    });
                }
                return original.apply(this, arguments);
            };
        };
    }
    getPatchCreateClient() {
        const plugin = this;
        return function createClientWrap(original) {
            return function createClientTrace() {
                const client = original.apply(this, arguments);
                plugin.tracer.wrapEmitter(client);
                return client;
            };
        };
    }
    getPatchSendCommand() {
        const plugin = this;
        const addArguments = typeof this.options === 'object'
            && this.options.detailedCommands === true;
        return function internalSendCommandWrap(original) {
            return function internal_send_command_trace(cmd, args, cb) {
                if (!plugin.tracer.currentRootSpan) {
                    return original.apply(this, arguments);
                }
                if (arguments.length === 1 && typeof cmd === 'object') {
                    const span = plugin.tracer.startChildSpan(`redis-${cmd.command}`, core_1.SpanKind.CLIENT);
                    if (span === null)
                        return original.apply(this, arguments);
                    span.addAttribute('command', cmd.command);
                    if (addArguments) {
                        span.addAttribute('arguments', JSON.stringify(cmd.args || []));
                    }
                    cmd.callback = plugin.patchEnd(span, cmd.callback);
                    return original.apply(this, arguments);
                }
                if (typeof cmd === 'string' && Array.isArray(args) && typeof cb === 'function') {
                    const span = plugin.tracer.startChildSpan(`redis-${cmd}`, core_1.SpanKind.CLIENT);
                    if (span === null)
                        return original.apply(this, arguments);
                    span.addAttribute('command', cmd);
                    if (addArguments) {
                        span.addAttribute('arguments', JSON.stringify(args));
                    }
                    cb = plugin.patchEnd(span, cb);
                    return original.apply(this, arguments);
                }
                return original.apply(this, arguments);
            };
        };
    }
    patchEnd(span, resultHandler) {
        const plugin = this;
        const patchedEnd = function (err) {
            if (plugin.options.detailedCommands === true && err instanceof Error) {
                span.addAttribute('error', err.message);
            }
            if (span.ended === false) {
                span.end();
            }
            if (typeof resultHandler === 'function') {
                return resultHandler.apply(this, arguments);
            }
        };
        return this.tracer.wrap(patchedEnd);
    }
}
exports.RedisPlugin = RedisPlugin;
const plugin = new RedisPlugin('redis');
exports.plugin = plugin;
//# sourceMappingURL=data:application/json;base64,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