"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WORKSPACES_API_BASE_URL = void 0;
exports.registerRoutes = registerRoutes;
var _configSchema = require("@osd/config-schema");
var _server = require("../../../../core/server");
var _constants = require("../../common/constants");
var _duplicate = require("./duplicate");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const WORKSPACES_API_BASE_URL = exports.WORKSPACES_API_BASE_URL = '/api/workspaces';
const workspacePermissionMode = _configSchema.schema.oneOf([_configSchema.schema.literal(_constants.WorkspacePermissionMode.Read), _configSchema.schema.literal(_constants.WorkspacePermissionMode.Write), _configSchema.schema.literal(_constants.WorkspacePermissionMode.LibraryRead), _configSchema.schema.literal(_constants.WorkspacePermissionMode.LibraryWrite)]);
const principalType = _configSchema.schema.oneOf([_configSchema.schema.literal(_server.PrincipalType.Users), _configSchema.schema.literal(_server.PrincipalType.Groups)]);
const workspacePermissions = _configSchema.schema.recordOf(workspacePermissionMode, _configSchema.schema.recordOf(principalType, _configSchema.schema.arrayOf(_configSchema.schema.string()), {}));
const workspaceOptionalAttributesSchema = {
  description: _configSchema.schema.maybe(_configSchema.schema.string()),
  features: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
  color: _configSchema.schema.maybe(_configSchema.schema.string()),
  icon: _configSchema.schema.maybe(_configSchema.schema.string()),
  defaultVISTheme: _configSchema.schema.maybe(_configSchema.schema.string()),
  reserved: _configSchema.schema.maybe(_configSchema.schema.boolean())
};
const createWorkspaceAttributesSchema = _configSchema.schema.object({
  name: _configSchema.schema.string(),
  ...workspaceOptionalAttributesSchema
});
const updateWorkspaceAttributesSchema = _configSchema.schema.object({
  name: _configSchema.schema.maybe(_configSchema.schema.string()),
  ...workspaceOptionalAttributesSchema
});
function registerRoutes({
  client,
  logger,
  http,
  maxImportExportSize,
  permissionControlClient,
  isPermissionControlEnabled
}) {
  const router = http.createRouter();
  router.post({
    path: `${WORKSPACES_API_BASE_URL}/_list`,
    validate: {
      body: _configSchema.schema.object({
        search: _configSchema.schema.maybe(_configSchema.schema.string()),
        sortOrder: _configSchema.schema.maybe(_configSchema.schema.string()),
        perPage: _configSchema.schema.number({
          min: 0,
          defaultValue: 20
        }),
        page: _configSchema.schema.number({
          min: 0,
          defaultValue: 1
        }),
        sortField: _configSchema.schema.maybe(_configSchema.schema.string()),
        searchFields: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
        permissionModes: _configSchema.schema.maybe(_configSchema.schema.arrayOf(workspacePermissionMode))
      })
    }
  }, router.handleLegacyErrors(async (context, req, res) => {
    const result = await client.list({
      context,
      request: req,
      logger
    }, req.body);
    if (!result.success) {
      return res.ok({
        body: result
      });
    }
    return res.ok({
      body: result
    });
  }));
  router.get({
    path: `${WORKSPACES_API_BASE_URL}/{id}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, router.handleLegacyErrors(async (context, req, res) => {
    const {
      id
    } = req.params;
    const result = await client.get({
      context,
      request: req,
      logger
    }, id);
    return res.ok({
      body: result
    });
  }));
  router.post({
    path: `${WORKSPACES_API_BASE_URL}`,
    validate: {
      body: _configSchema.schema.object({
        attributes: createWorkspaceAttributesSchema,
        permissions: _configSchema.schema.maybe(workspacePermissions)
      })
    }
  }, router.handleLegacyErrors(async (context, req, res) => {
    const {
      attributes,
      permissions
    } = req.body;
    const principals = permissionControlClient === null || permissionControlClient === void 0 ? void 0 : permissionControlClient.getPrincipalsFromRequest(req);
    const createPayload = attributes;
    if (isPermissionControlEnabled) {
      var _principals$users;
      createPayload.permissions = permissions;
      // Assign workspace owner to current user
      if (!!(principals !== null && principals !== void 0 && (_principals$users = principals.users) !== null && _principals$users !== void 0 && _principals$users.length)) {
        const acl = new _server.ACL(permissions);
        const currentUserId = principals.users[0];
        [_constants.WorkspacePermissionMode.Write, _constants.WorkspacePermissionMode.LibraryWrite].forEach(permissionMode => {
          if (!acl.hasPermission([permissionMode], {
            users: [currentUserId]
          })) {
            acl.addPermission([permissionMode], {
              users: [currentUserId]
            });
          }
        });
        createPayload.permissions = acl.getPermissions();
      }
    }
    const result = await client.create({
      context,
      request: req,
      logger
    }, createPayload);
    return res.ok({
      body: result
    });
  }));
  router.put({
    path: `${WORKSPACES_API_BASE_URL}/{id?}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      }),
      body: _configSchema.schema.object({
        attributes: updateWorkspaceAttributesSchema,
        permissions: _configSchema.schema.maybe(workspacePermissions)
      })
    }
  }, router.handleLegacyErrors(async (context, req, res) => {
    const {
      id
    } = req.params;
    const {
      attributes,
      permissions
    } = req.body;
    const result = await client.update({
      context,
      request: req,
      logger
    }, id, {
      ...attributes,
      ...(isPermissionControlEnabled ? {
        permissions
      } : {})
    });
    return res.ok({
      body: result
    });
  }));
  router.delete({
    path: `${WORKSPACES_API_BASE_URL}/{id?}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, router.handleLegacyErrors(async (context, req, res) => {
    const {
      id
    } = req.params;
    const result = await client.delete({
      context,
      request: req,
      logger
    }, id);
    return res.ok({
      body: result
    });
  }));

  // duplicate saved objects among workspaces
  (0, _duplicate.registerDuplicateRoute)(router, logger, client, maxImportExportSize);
}