"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.IntegrationsManager = void 0;
var _path = _interopRequireDefault(require("path"));
var _metrics_helper = require("../../common/metrics/metrics_helper");
var _integrations_builder = require("./integrations_builder");
var _repository = require("./repository/repository");
var _fs_data_adaptor = require("./repository/fs_data_adaptor");
var _index_data_adaptor = require("./repository/index_data_adaptor");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
class IntegrationsManager {
  constructor(client, repository) {
    _defineProperty(this, "client", void 0);
    _defineProperty(this, "instanceBuilder", void 0);
    _defineProperty(this, "repository", void 0);
    _defineProperty(this, "deleteIntegrationInstance", async id => {
      let children;
      try {
        children = await this.client.get('integration-instance', id);
      } catch (err) {
        var _err$output;
        return ((_err$output = err.output) === null || _err$output === void 0 ? void 0 : _err$output.statusCode) === 404 ? Promise.resolve([id]) : Promise.reject(err);
      }
      const toDelete = children.attributes.assets.filter(i => i.assetId).map(i => {
        return {
          id: i.assetId,
          type: i.assetType
        };
      });
      toDelete.push({
        id,
        type: 'integration-instance'
      });
      const result = Promise.all(toDelete.map(async asset => {
        try {
          await this.client.delete(asset.type, asset.id);
          return Promise.resolve(asset.id);
        } catch (err) {
          var _err$output2;
          (0, _metrics_helper.addRequestToMetric)('integrations', 'delete', err);
          return ((_err$output2 = err.output) === null || _err$output2 === void 0 ? void 0 : _err$output2.statusCode) === 404 ? Promise.resolve(asset.id) : Promise.reject(err);
        }
      }));
      (0, _metrics_helper.addRequestToMetric)('integrations', 'delete', 'count');
      return result;
    });
    // Internal; use getIntegrationTemplates.
    _defineProperty(this, "_getAllIntegrationTemplates", async () => {
      const integrationList = await this.repository.getIntegrationList();
      const configResults = await Promise.all(integrationList.map(x => x.getConfig()));
      const configs = configResults.filter(cfg => cfg.ok);
      return Promise.resolve({
        hits: configs.map(cfg => cfg.value)
      });
    });
    // Internal; use getIntegrationTemplates.
    _defineProperty(this, "_getIntegrationTemplatesByName", async name => {
      const integration = await this.repository.getIntegration(name);
      const config = await (integration === null || integration === void 0 ? void 0 : integration.getConfig());
      if (!config || !config.ok) {
        return Promise.resolve({
          hits: []
        });
      }
      return Promise.resolve({
        hits: [config.value]
      });
    });
    _defineProperty(this, "getIntegrationTemplates", async query => {
      if (query !== null && query !== void 0 && query.name) {
        return this._getIntegrationTemplatesByName(query.name);
      }
      return this._getAllIntegrationTemplates();
    });
    _defineProperty(this, "getIntegrationInstances", async _query => {
      var _result$saved_objects;
      (0, _metrics_helper.addRequestToMetric)('integrations', 'get', 'count');
      const result = await this.client.find({
        type: 'integration-instance',
        perPage: 1000
      });
      return Promise.resolve({
        total: result.total,
        hits: (_result$saved_objects = result.saved_objects) === null || _result$saved_objects === void 0 ? void 0 : _result$saved_objects.map(x => ({
          ...x.attributes,
          id: x.id
        }))
      });
    });
    _defineProperty(this, "getIntegrationInstance", async query => {
      (0, _metrics_helper.addRequestToMetric)('integrations', 'get', 'count');
      const result = await this.client.get('integration-instance', `${query.id}`);
      return Promise.resolve(this.buildInstanceResponse(result));
    });
    _defineProperty(this, "buildInstanceResponse", async savedObj => {
      const assets = savedObj.attributes.assets;
      const status = assets ? await this.getAssetStatus(assets) : 'available';
      return {
        id: savedObj.id,
        status,
        ...savedObj.attributes
      };
    });
    _defineProperty(this, "getAssetStatus", async assets => {
      const statuses = await Promise.all(assets.map(async asset => {
        try {
          await this.client.get(asset.assetType, asset.assetId);
          return {
            id: asset.assetId,
            status: 'available'
          };
        } catch (err) {
          var _err$output3;
          const statusCode = (_err$output3 = err.output) === null || _err$output3 === void 0 ? void 0 : _err$output3.statusCode;
          if (statusCode && 400 <= statusCode && statusCode < 500) {
            return {
              id: asset.assetId,
              status: 'unavailable'
            };
          }
          console.error('Failed to get asset status', err);
          return {
            id: asset.assetId,
            status: 'unknown'
          };
        }
      }));
      const [available, unavailable, unknown] = [statuses.filter(x => x.status === 'available').length, statuses.filter(x => x.status === 'unavailable').length, statuses.filter(x => x.status === 'unknown').length];
      if (unknown > 0) return 'unknown';
      if (unavailable > 0 && available > 0) return 'partially-available';
      if (unavailable > 0) return 'unavailable';
      return 'available';
    });
    _defineProperty(this, "loadIntegrationInstance", async (templateName, name, indexPattern, workflows, dataSource, tableName) => {
      const template = await this.repository.getIntegration(templateName);
      if (template === null) {
        return Promise.reject({
          message: `Template ${templateName} not found`,
          statusCode: 404
        });
      }
      try {
        (0, _metrics_helper.addRequestToMetric)('integrations', 'create', 'count');
        const result = await this.instanceBuilder.build(template, {
          name,
          indexPattern,
          workflows,
          dataSource,
          tableName
        });
        const test = await this.client.create('integration-instance', result);
        return Promise.resolve({
          ...result,
          id: test.id
        });
      } catch (err) {
        (0, _metrics_helper.addRequestToMetric)('integrations', 'create', err);
        return Promise.reject({
          message: err.message,
          statusCode: 500
        });
      }
    });
    _defineProperty(this, "getStatic", async (templateName, staticPath) => {
      const integration = await this.repository.getIntegration(templateName);
      if (integration === null) {
        return Promise.reject({
          message: `Template ${templateName} not found`,
          statusCode: 404
        });
      }
      const data = await integration.getStatic(staticPath);
      if (data.ok) {
        return data.value;
      }
      const is404 = data.error.code === 'ENOENT';
      return Promise.reject({
        message: data.error.message,
        statusCode: is404 ? 404 : 500
      });
    });
    _defineProperty(this, "getSchemas", async templateName => {
      const integration = await this.repository.getIntegration(templateName);
      if (integration === null) {
        return Promise.reject({
          message: `Template ${templateName} not found`,
          statusCode: 404
        });
      }
      const result = await integration.getSchemas();
      if (result.ok) {
        return result.value;
      }
      const is404 = result.error.code === 'ENOENT';
      return Promise.reject({
        message: result.error.message,
        statusCode: is404 ? 404 : 500
      });
    });
    _defineProperty(this, "getAssets", async templateName => {
      const integration = await this.repository.getIntegration(templateName);
      if (integration === null) {
        return Promise.reject({
          message: `Template ${templateName} not found`,
          statusCode: 404
        });
      }
      const assets = await integration.getAssets();
      if (assets.ok) {
        return assets.value;
      }
      const is404 = assets.error.code === 'ENOENT';
      return Promise.reject({
        message: assets.error.message,
        statusCode: is404 ? 404 : 500
      });
    });
    _defineProperty(this, "getSampleData", async templateName => {
      const integration = await this.repository.getIntegration(templateName);
      if (integration === null) {
        return Promise.reject({
          message: `Template ${templateName} not found`,
          statusCode: 404
        });
      }
      const sampleData = await integration.getSampleData();
      if (sampleData.ok) {
        return sampleData.value;
      }
      const is404 = sampleData.error.code === 'ENOENT';
      return Promise.reject({
        message: sampleData.error.message,
        statusCode: is404 ? 404 : 500
      });
    });
    this.client = client;
    this.repository = repository !== null && repository !== void 0 ? repository : new _repository.TemplateManager([new _index_data_adaptor.IndexDataAdaptor(this.client), new _fs_data_adaptor.FileSystemDataAdaptor(_path.default.join(__dirname, '__data__/repository'))]);
    this.instanceBuilder = new _integrations_builder.IntegrationInstanceBuilder(this.client);
  }
}
exports.IntegrationsManager = IntegrationsManager;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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