"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomCensusExporter = void 0;
const serviceManager_1 = require("../serviceManager");
const core_1 = require("@opencensus/core");
const default_config_1 = require("./config/default-config");
const constants_1 = require("./constants");
var CanonicalCodeString;
(function (CanonicalCodeString) {
    CanonicalCodeString["OK"] = "OK";
    CanonicalCodeString["CANCELLED"] = "CANCELLED";
    CanonicalCodeString["UNKNOWN"] = "UNKNOWN";
    CanonicalCodeString["INVALID_ARGUMENT"] = "INVALID_ARGUMENT";
    CanonicalCodeString["DEADLINE_EXCEEDED"] = "DEADLINE_EXCEEDED";
    CanonicalCodeString["NOT_FOUND"] = "NOT_FOUND";
    CanonicalCodeString["ALREADY_EXISTS"] = "ALREADY_EXISTS";
    CanonicalCodeString["PERMISSION_DENIED"] = "PERMISSION_DENIED";
    CanonicalCodeString["RESOURCE_EXHAUSTED"] = "RESOURCE_EXHAUSTED";
    CanonicalCodeString["FAILED_PRECONDITION"] = "FAILED_PRECONDITION";
    CanonicalCodeString["ABORTED"] = "ABORTED";
    CanonicalCodeString["OUT_OF_RANGE"] = "OUT_OF_RANGE";
    CanonicalCodeString["UNIMPLEMENTED"] = "UNIMPLEMENTED";
    CanonicalCodeString["INTERNAL"] = "INTERNAL";
    CanonicalCodeString["UNAVAILABLE"] = "UNAVAILABLE";
    CanonicalCodeString["DATA_LOSS"] = "DATA_LOSS";
    CanonicalCodeString["UNAUTHENTICATED"] = "UNAUTHENTICATED";
})(CanonicalCodeString || (CanonicalCodeString = {}));
class CustomCensusExporter {
    constructor(config) {
        this.transport = serviceManager_1.ServiceManager.get('transport');
        this.config = config;
        this.buffer = new core_1.ExporterBuffer(this, default_config_1.defaultConfig);
    }
    onEndSpan(root) {
        this.buffer.addToBuffer(root);
    }
    onStartSpan(root) { }
    sendTraces(zipkinTraces) {
        return new Promise((resolve, reject) => {
            zipkinTraces.forEach(span => {
                const isRootClient = span.kind === 'CLIENT' && !span.parentId;
                if (isRootClient && this.config.outbound === false)
                    return;
                if (process.env.NODE_ENV === 'test' || (span.duration > constants_1.Constants.MINIMUM_TRACE_DURATION)) {
                    this.transport.send('trace-span', span);
                }
            });
            resolve();
        });
    }
    mountSpanList(rootSpans) {
        const spanList = [];
        for (const root of rootSpans) {
            spanList.push(this.translateSpan(root));
            for (const span of root.spans) {
                spanList.push(this.translateSpan(span));
            }
        }
        return spanList;
    }
    translateSpan(span) {
        const spanTranslated = {
            traceId: span.traceId,
            name: span.name,
            id: span.id,
            parentId: span.parentSpanId,
            kind: this.getSpanKind(span.kind),
            timestamp: span.startTime.getTime() * 1000,
            duration: Math.round(span.duration * 1000),
            debug: false,
            shared: false,
            localEndpoint: { serviceName: this.config.serviceName },
            tags: span.attributes
        };
        if (typeof spanTranslated.tags['result.code'] !== 'string') {
            spanTranslated.tags['result.code'] = CanonicalCodeString[span.status.code];
        }
        if (typeof span.status.message === 'string') {
            spanTranslated.tags['result.message'] = span.status.message;
        }
        return spanTranslated;
    }
    publish(rootSpans) {
        const spanList = this.mountSpanList(rootSpans);
        return this.sendTraces(spanList).catch((err) => {
            return err;
        });
    }
    getSpanKind(kind) {
        switch (kind) {
            case core_1.SpanKind.CLIENT: {
                return 'CLIENT';
            }
            case core_1.SpanKind.SERVER: {
                return 'SERVER';
            }
            default: {
                return 'UNKNOWN';
            }
        }
    }
}
exports.CustomCensusExporter = CustomCensusExporter;
//# sourceMappingURL=data:application/json;base64,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