"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.plugin = exports.NetPlugin = void 0;
const core_1 = require("@opencensus/core");
const shimmer = require("shimmer");
class NetPlugin extends core_1.BasePlugin {
    constructor(moduleName) {
        super(moduleName);
    }
    applyPatch() {
        this.logger.debug('applying patch to %s@%s', this.moduleName, this.version);
        if (this.moduleExports && this.moduleExports.Server &&
            this.moduleExports.Server.prototype) {
            shimmer.wrap(this.moduleExports.Server.prototype, 'emit', this.getPatchIncomingRequestFunction());
        }
        else {
            this.logger.error('Could not apply patch to %s.emit. Interface is not as expected.', this.moduleName);
        }
        return this.moduleExports;
    }
    applyUnpatch() {
        if (this.moduleExports && this.moduleExports.Server &&
            this.moduleExports.Server.prototype) {
            shimmer.unwrap(this.moduleExports.Server.prototype, 'emit');
        }
        else {
            this.logger.error('Could not unapply patch to %s.emit. Interface is not as expected.', this.moduleName);
        }
    }
    getPatchIncomingRequestFunction() {
        return (original) => {
            const plugin = this;
            return function incomingRequest(event, ...args) {
                if (event !== 'connection') {
                    return original.apply(this, arguments);
                }
                const socket = args[0];
                plugin.logger.debug('%s plugin incomingRequest', plugin.moduleName);
                const traceOptions = {
                    name: 'socket',
                    kind: core_1.SpanKind.SERVER,
                    spanContext: undefined
                };
                return plugin.tracer.startRootSpan(traceOptions, rootSpan => {
                    if (!rootSpan)
                        return original.apply(this, arguments);
                    plugin.tracer.wrapEmitter(socket);
                    const address = socket.address();
                    if (typeof address === 'string') {
                        rootSpan.addAttribute('net.address', address);
                    }
                    else {
                        rootSpan.addAttribute('net.host', address.address);
                        rootSpan.addAttribute('net.port', address.port);
                        rootSpan.addAttribute('net.family', address.family);
                    }
                    socket.on('error', (err) => {
                        rootSpan.addAttribute('net.error', err.message);
                    });
                    const originalEnd = socket.end;
                    socket.end = function () {
                        if (rootSpan.ended === false) {
                            rootSpan.end();
                        }
                        return originalEnd.apply(this, arguments);
                    };
                    socket.on('close', () => {
                        if (rootSpan.ended === false) {
                            rootSpan.end();
                        }
                    });
                    return original.apply(this, arguments);
                });
            };
        };
    }
}
exports.NetPlugin = NetPlugin;
exports.plugin = new NetPlugin('net');
//# sourceMappingURL=data:application/json;base64,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