'use strict';
Object.defineProperty(exports, "__esModule", { value: true });
exports.plugin = exports.VuePlugin = void 0;
const core_1 = require("@opencensus/core");
const shimmer = require("shimmer");
var RendererType;
(function (RendererType) {
    RendererType["NORMAL"] = "normal";
    RendererType["BUNDLE"] = "bundle";
})(RendererType || (RendererType = {}));
class VuePlugin extends core_1.BasePlugin {
    constructor(moduleName) {
        super(moduleName);
        this.rendererResults = [];
    }
    applyPatch() {
        this.logger.debug('Patched vue-server-renderer');
        if (this.moduleExports) {
            this.logger.debug('patching vue-server-renderer.prototype.createRenderer');
            shimmer.wrap(this.moduleExports, 'createRenderer', this.getPatchCreateRenderer(RendererType.NORMAL));
            this.logger.debug('patching vue-server-renderer.prototype.createBundleRenderer');
            shimmer.wrap(this.moduleExports, 'createBundleRenderer', this.getPatchCreateRenderer(RendererType.BUNDLE));
        }
        return this.moduleExports;
    }
    applyUnpatch() {
        shimmer.unwrap(this.moduleExports, 'createRenderer');
        shimmer.unwrap(this.moduleExports, 'createBundleRenderer');
        for (let result of this.rendererResults) {
            shimmer.unwrap(result, 'renderToString');
        }
    }
    getPatchCreateRenderer(type) {
        const plugin = this;
        return function createRendererWrap(original) {
            return function create_renderer_trace() {
                const result = original.apply(this, arguments);
                plugin.logger.debug(`patching ${type}.renderToString`);
                shimmer.wrap(result, 'renderToString', plugin.getPatchRenderToString(type));
                plugin.rendererResults.push(result);
                return result;
            };
        };
    }
    getPatchRenderToString(type) {
        const plugin = this;
        return function renderToStringWrap(original) {
            return function render_string_trace() {
                if (!plugin.tracer.currentRootSpan) {
                    return original.apply(this, arguments);
                }
                const span = plugin.tracer.startChildSpan(`vue-renderer`, core_1.SpanKind.CLIENT);
                if (span === null)
                    return original.apply(this, arguments);
                const promise = original.apply(this, arguments);
                if (promise instanceof Promise) {
                    promise.then(plugin.patchEnd(span)).catch(plugin.patchEnd(span));
                }
                return promise;
            };
        };
    }
    patchEnd(span) {
        const plugin = this;
        const patchedEnd = function (err) {
            if (plugin.options.detailedCommands === true && err instanceof Error) {
                span.addAttribute('error', err.message);
            }
            if (span.ended === false) {
                span.end();
            }
        };
        return this.tracer.wrap(patchedEnd);
    }
}
exports.VuePlugin = VuePlugin;
const plugin = new VuePlugin('vue-server-renderer');
exports.plugin = plugin;
//# sourceMappingURL=data:application/json;base64,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