"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OpenIdAuthentication = void 0;
var fs = _interopRequireWildcard(require("fs"));
var _wreck = _interopRequireDefault(require("@hapi/wreck"));
var _http = _interopRequireDefault(require("http"));
var _https = _interopRequireDefault(require("https"));
var _security_cookie = require("../../../session/security_cookie");
var _routes = require("./routes");
var _authentication_type = require("../authentication_type");
var _helper = require("./helper");
var _object_properties_defined = require("../../../utils/object_properties_defined");
var _common = require("../../../../common");
var _cookie_splitter = require("../../../session/cookie_splitter");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && Object.prototype.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
class OpenIdAuthentication extends _authentication_type.AuthenticationType {
  constructor(config, sessionStorageFactory, router, esClient, core, logger) {
    var _this$config$openid, _this$config$openid2;
    super(config, sessionStorageFactory, router, esClient, core, logger);
    _defineProperty(this, "type", _common.AuthType.OPEN_ID);
    _defineProperty(this, "openIdAuthConfig", void 0);
    _defineProperty(this, "authHeaderName", void 0);
    _defineProperty(this, "openIdConnectUrl", void 0);
    _defineProperty(this, "wreckClient", void 0);
    _defineProperty(this, "wreckHttpsOption", {});
    _defineProperty(this, "redirectOIDCCapture", (request, toolkit) => {
      const nextUrl = this.generateNextUrl(request);
      const clearOldVersionCookie = (0, _security_cookie.clearOldVersionCookieValue)(this.config);
      return toolkit.redirected({
        location: `${this.coreSetup.http.basePath.serverBasePath}/auth/openid/captureUrlFragment?nextUrl=${nextUrl}`,
        'set-cookie': clearOldVersionCookie
      });
    });
    this.wreckClient = this.createWreckClient();
    this.openIdAuthConfig = {};
    this.authHeaderName = ((_this$config$openid = this.config.openid) === null || _this$config$openid === void 0 ? void 0 : _this$config$openid.header) || '';
    this.openIdAuthConfig.authHeaderName = this.authHeaderName;
    this.openIdConnectUrl = ((_this$config$openid2 = this.config.openid) === null || _this$config$openid2 === void 0 ? void 0 : _this$config$openid2.connect_url) || '';
    let scope = this.config.openid.scope;
    if (scope.indexOf('openid') < 0) {
      scope = `openid ${scope}`;
    }
    this.openIdAuthConfig.scope = scope;
  }
  async init() {
    try {
      const response = await this.wreckClient.get(this.openIdConnectUrl);
      const payload = JSON.parse(response.payload);
      this.openIdAuthConfig.authorizationEndpoint = payload.authorization_endpoint;
      this.openIdAuthConfig.tokenEndpoint = payload.token_endpoint;
      this.openIdAuthConfig.endSessionEndpoint = payload.end_session_endpoint || undefined;
      this.createExtraStorage();
      const routes = new _routes.OpenIdAuthRoutes(this.router, this.config, this.sessionStorageFactory, this.openIdAuthConfig, this.securityClient, this.coreSetup, this.wreckClient);
      routes.setupRoutes();
    } catch (error) {
      this.logger.error(error); // TODO: log more info
      throw new Error('Failed when trying to obtain the endpoints from your IdP');
    }
  }
  generateNextUrl(request) {
    const path = this.coreSetup.http.basePath.serverBasePath + (request.url.pathname || '/app/opensearch-dashboards');
    return escape(path);
  }
  createWreckClient() {
    var _this$config$openid3, _this$config$openid4, _this$config$openid5, _this$config$openid6, _this$config$openid7, _this$config$openid9;
    if ((_this$config$openid3 = this.config.openid) !== null && _this$config$openid3 !== void 0 && _this$config$openid3.root_ca) {
      this.wreckHttpsOption.ca = [fs.readFileSync(this.config.openid.root_ca)];
      this.logger.debug(`Using CA Cert: ${this.config.openid.root_ca}`);
    }
    if ((_this$config$openid4 = this.config.openid) !== null && _this$config$openid4 !== void 0 && _this$config$openid4.pfx) {
      // Use PFX or PKCS12 if provided
      this.logger.debug(`Using PFX or PKCS12: ${this.config.openid.pfx}`);
      this.wreckHttpsOption.pfx = [fs.readFileSync(this.config.openid.pfx)];
    } else if ((_this$config$openid5 = this.config.openid) !== null && _this$config$openid5 !== void 0 && _this$config$openid5.certificate && (_this$config$openid6 = this.config.openid) !== null && _this$config$openid6 !== void 0 && _this$config$openid6.private_key) {
      // Use 'certificate' and 'private_key' if provided
      this.logger.debug(`Using Certificate: ${this.config.openid.certificate}`);
      this.logger.debug(`Using Private Key: ${this.config.openid.private_key}`);
      this.wreckHttpsOption.cert = [fs.readFileSync(this.config.openid.certificate)];
      this.wreckHttpsOption.key = [fs.readFileSync(this.config.openid.private_key)];
    } else {
      this.logger.debug(`Client certificates not provided. Mutual TLS will not be used to obtain endpoints.`);
    }
    // Check if passphrase is provided, use it for 'pfx' and 'key'
    if (((_this$config$openid7 = this.config.openid) === null || _this$config$openid7 === void 0 ? void 0 : _this$config$openid7.passphrase) !== '') {
      var _this$config$openid8;
      this.logger.debug(`Passphrase not provided for private key and/or pfx.`);
      this.wreckHttpsOption.passphrase = (_this$config$openid8 = this.config.openid) === null || _this$config$openid8 === void 0 ? void 0 : _this$config$openid8.passphrase;
    }
    if (((_this$config$openid9 = this.config.openid) === null || _this$config$openid9 === void 0 ? void 0 : _this$config$openid9.verify_hostnames) === false) {
      this.logger.debug(`openId auth 'verify_hostnames' option is off.`);
      this.wreckHttpsOption.checkServerIdentity = (host, cert) => {
        return undefined;
      };
    }
    this.logger.info((0, _object_properties_defined.getObjectProperties)(this.wreckHttpsOption, 'WreckHttpsOptions'));
    if (Object.keys(this.wreckHttpsOption).length > 0) {
      return _wreck.default.defaults({
        agents: {
          http: new _http.default.Agent(),
          https: new _https.default.Agent(this.wreckHttpsOption),
          httpsAllowUnauthorized: new _https.default.Agent({
            rejectUnauthorized: false
          })
        }
      });
    } else {
      return _wreck.default;
    }
  }
  getWreckHttpsOptions() {
    return this.wreckHttpsOption;
  }
  createExtraStorage() {
    // @ts-ignore
    const hapiServer = this.sessionStorageFactory.asScoped({}).server;
    const extraCookiePrefix = this.config.openid.extra_storage.cookie_prefix;
    const extraCookieSettings = {
      isSecure: this.config.cookie.secure,
      isSameSite: this.config.cookie.isSameSite,
      password: this.config.cookie.password,
      domain: this.config.cookie.domain,
      path: this.coreSetup.http.basePath.serverBasePath || '/',
      clearInvalid: false,
      isHttpOnly: true,
      ignoreErrors: true,
      encoding: 'iron' // Same as hapi auth cookie
    };

    for (let i = 1; i <= this.config.openid.extra_storage.additional_cookies; i++) {
      hapiServer.states.add(extraCookiePrefix + i, extraCookieSettings);
    }
  }
  getExtraAuthStorageOptions() {
    // If we're here, we will always have the openid configuration
    return {
      cookiePrefix: this.config.openid.extra_storage.cookie_prefix,
      additionalCookies: this.config.openid.extra_storage.additional_cookies,
      logger: this.logger
    };
  }
  requestIncludesAuthInfo(request) {
    return request.headers.authorization ? true : false;
  }
  async getAdditionalAuthHeader(request) {
    return {};
  }
  getCookie(request, authInfo) {
    (0, _cookie_splitter.setExtraAuthStorage)(request, request.headers.authorization, this.getExtraAuthStorageOptions());
    return {
      username: authInfo.user_name,
      credentials: {
        authHeaderValueExtra: true
      },
      authType: this.type,
      expiryTime: Date.now() + this.config.session.ttl
    };
  }
  getKeepAliveExpiry(cookie, request) {
    return Date.now() + this.config.session.ttl;
  }

  // TODO: Add token expiration check here
  async isValidCookie(cookie, request) {
    var _cookie$credentials;
    if (cookie.authType !== this.type || !cookie.username || !cookie.expiryTime || !((_cookie$credentials = cookie.credentials) !== null && _cookie$credentials !== void 0 && _cookie$credentials.authHeaderValue) && !this.getExtraAuthStorageValue(request, cookie)) {
      return false;
    }
    if (cookie.credentials.expiryTime > Date.now()) {
      return true;
    }

    // need to renew id token
    if (cookie.credentials.refresh_token) {
      try {
        var _this$config$openid10, _this$config$openid11;
        const query = {
          grant_type: 'refresh_token',
          client_id: (_this$config$openid10 = this.config.openid) === null || _this$config$openid10 === void 0 ? void 0 : _this$config$openid10.client_id,
          client_secret: (_this$config$openid11 = this.config.openid) === null || _this$config$openid11 === void 0 ? void 0 : _this$config$openid11.client_secret,
          refresh_token: cookie.credentials.refresh_token
        };
        const refreshTokenResponse = await (0, _helper.callTokenEndpoint)(this.openIdAuthConfig.tokenEndpoint, query, this.wreckClient);

        // if no id_token from refresh token call, maybe the Idp doesn't allow refresh id_token
        if (refreshTokenResponse.idToken) {
          cookie.credentials = {
            authHeaderValueExtra: true,
            refresh_token: refreshTokenResponse.refreshToken,
            expiryTime: (0, _helper.getExpirationDate)(refreshTokenResponse)
          };
          (0, _cookie_splitter.setExtraAuthStorage)(request, `Bearer ${refreshTokenResponse.idToken}`, this.getExtraAuthStorageOptions());
          return true;
        } else {
          return false;
        }
      } catch (error) {
        this.logger.error(error);
        return false;
      }
    } else {
      // no refresh token, and current token is expired
      return false;
    }
  }
  handleUnauthedRequest(request, response, toolkit) {
    if (this.isPageRequest(request)) {
      return this.redirectOIDCCapture(request, toolkit);
    } else {
      return response.unauthorized();
    }
  }
  getExtraAuthStorageValue(request, cookie) {
    var _cookie$credentials2;
    let extraValue = '';
    if (!((_cookie$credentials2 = cookie.credentials) !== null && _cookie$credentials2 !== void 0 && _cookie$credentials2.authHeaderValueExtra)) {
      return extraValue;
    }
    try {
      extraValue = (0, _cookie_splitter.getExtraAuthStorageValue)(request, this.getExtraAuthStorageOptions());
    } catch (error) {
      this.logger.info(error);
    }
    return extraValue;
  }
  buildAuthHeaderFromCookie(cookie, request) {
    var _cookie$credentials3;
    const header = {};
    if (cookie.credentials.authHeaderValueExtra) {
      try {
        const extraAuthStorageValue = this.getExtraAuthStorageValue(request, cookie);
        header.authorization = extraAuthStorageValue;
        return header;
      } catch (error) {
        this.logger.error(error);
        // TODO Re-throw?
        // throw error;
      }
    }

    const authHeaderValue = (_cookie$credentials3 = cookie.credentials) === null || _cookie$credentials3 === void 0 ? void 0 : _cookie$credentials3.authHeaderValue;
    if (authHeaderValue) {
      header.authorization = authHeaderValue;
    }
    return header;
  }
}
exports.OpenIdAuthentication = OpenIdAuthentication;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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