"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OpenIdAuthRoutes = void 0;
var _configSchema = require("@osd/config-schema");
var _cryptiles = require("@hapi/cryptiles");
var _querystring = require("querystring");
var _helper = require("./helper");
var _next_url = require("../../../utils/next_url");
var _common = require("../../../../common");
var _cookie_splitter = require("../../../session/cookie_splitter");
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
class OpenIdAuthRoutes {
  constructor(router, config, sessionStorageFactory, openIdAuthConfig, securityClient, core, wreckClient) {
    this.router = router;
    this.config = config;
    this.sessionStorageFactory = sessionStorageFactory;
    this.openIdAuthConfig = openIdAuthConfig;
    this.securityClient = securityClient;
    this.core = core;
    this.wreckClient = wreckClient;
  }
  redirectToLogin(request, response) {
    this.sessionStorageFactory.asScoped(request).clear();
    return response.redirected({
      headers: {
        location: `${this.core.http.basePath.serverBasePath}${_common.OPENID_AUTH_LOGIN}`
      }
    });
  }
  getExtraAuthStorageOptions(logger) {
    // If we're here, we will always have the openid configuration
    return {
      cookiePrefix: this.config.openid.extra_storage.cookie_prefix,
      additionalCookies: this.config.openid.extra_storage.additional_cookies,
      logger
    };
  }
  setupRoutes() {
    this.router.get({
      path: _common.OPENID_AUTH_LOGIN,
      validate: {
        query: _configSchema.schema.object({
          code: _configSchema.schema.maybe(_configSchema.schema.string()),
          nextUrl: _configSchema.schema.maybe(_configSchema.schema.string({
            validate: _next_url.validateNextUrl
          })),
          redirectHash: _configSchema.schema.maybe(_configSchema.schema.boolean()),
          state: _configSchema.schema.maybe(_configSchema.schema.string()),
          refresh: _configSchema.schema.maybe(_configSchema.schema.string())
        }, {
          unknowns: 'allow'
        })
      },
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      var _this$config$openid2, _this$config$openid3, _cookie$oidc2;
      // implementation refers to https://github.com/hapijs/bell/blob/master/lib/oauth.js
      // Sign-in initialization
      if (!request.query.code) {
        var _this$config$openid;
        const nonce = (0, _cryptiles.randomString)(OpenIdAuthRoutes.NONCE_LENGTH);
        const query = {
          client_id: (_this$config$openid = this.config.openid) === null || _this$config$openid === void 0 ? void 0 : _this$config$openid.client_id,
          response_type: _common.AUTH_RESPONSE_TYPE,
          redirect_uri: `${(0, _helper.getBaseRedirectUrl)(this.config, this.core, request)}${_common.OPENID_AUTH_LOGIN}`,
          state: nonce,
          scope: this.openIdAuthConfig.scope
        };
        (0, _helper.includeAdditionalParameters)(query, context, this.config);
        const queryString = (0, _querystring.stringify)(query);
        const location = `${this.openIdAuthConfig.authorizationEndpoint}?${queryString}`;
        const cookie = {
          oidc: {
            state: nonce,
            nextUrl: (0, _helper.getNextUrl)(this.config, this.core, request),
            redirectHash: request.query.redirectHash === 'true'
          },
          authType: _common.AuthType.OPEN_ID
        };
        this.sessionStorageFactory.asScoped(request).set(cookie);
        return response.redirected({
          headers: {
            location
          }
        });
      }

      // Authentication callback
      // validate state first
      let cookie;
      try {
        var _cookie$oidc;
        cookie = await this.sessionStorageFactory.asScoped(request).get();
        if (!cookie || !((_cookie$oidc = cookie.oidc) !== null && _cookie$oidc !== void 0 && _cookie$oidc.state) || cookie.oidc.state !== request.query.state) {
          return this.redirectToLogin(request, response);
        }
      } catch (error) {
        return this.redirectToLogin(request, response);
      }
      const nextUrl = cookie.oidc.nextUrl;
      const clientId = (_this$config$openid2 = this.config.openid) === null || _this$config$openid2 === void 0 ? void 0 : _this$config$openid2.client_id;
      const clientSecret = (_this$config$openid3 = this.config.openid) === null || _this$config$openid3 === void 0 ? void 0 : _this$config$openid3.client_secret;
      const redirectHash = ((_cookie$oidc2 = cookie.oidc) === null || _cookie$oidc2 === void 0 ? void 0 : _cookie$oidc2.redirectHash) || false;
      const query = {
        grant_type: _common.AUTH_GRANT_TYPE,
        code: request.query.code,
        redirect_uri: `${(0, _helper.getBaseRedirectUrl)(this.config, this.core, request)}${_common.OPENID_AUTH_LOGIN}`,
        client_id: clientId,
        client_secret: clientSecret
      };
      (0, _helper.includeAdditionalParameters)(query, context, this.config);
      try {
        var _this$config$openid4;
        const tokenResponse = await (0, _helper.callTokenEndpoint)(this.openIdAuthConfig.tokenEndpoint, query, this.wreckClient);
        const user = await this.securityClient.authenticateWithHeader(request, this.openIdAuthConfig.authHeaderName, `Bearer ${tokenResponse.idToken}`);

        // set to cookie
        const sessionStorage = {
          username: user.username,
          credentials: {
            authHeaderValueExtra: true,
            expiryTime: (0, _helper.getExpirationDate)(tokenResponse)
          },
          authType: _common.AuthType.OPEN_ID,
          expiryTime: Date.now() + this.config.session.ttl
        };
        if ((_this$config$openid4 = this.config.openid) !== null && _this$config$openid4 !== void 0 && _this$config$openid4.refresh_tokens && tokenResponse.refreshToken) {
          Object.assign(sessionStorage.credentials, {
            refresh_token: tokenResponse.refreshToken
          });
        }
        (0, _cookie_splitter.setExtraAuthStorage)(request, `Bearer ${tokenResponse.idToken}`, this.getExtraAuthStorageOptions(context.security_plugin.logger));
        this.sessionStorageFactory.asScoped(request).set(sessionStorage);
        if (redirectHash) {
          return response.redirected({
            headers: {
              location: `${this.core.http.basePath.serverBasePath}/auth/openid/redirectUrlFragment?nextUrl=${escape(nextUrl)}`
            }
          });
        } else {
          return response.redirected({
            headers: {
              location: nextUrl
            }
          });
        }
      } catch (error) {
        context.security_plugin.logger.error(`OpenId authentication failed: ${error}`);
        if (error.toString().toLowerCase().includes('authentication exception')) {
          return response.unauthorized();
        } else {
          return this.redirectToLogin(request, response);
        }
      }
    });
    this.router.get({
      path: _common.OPENID_AUTH_LOGOUT,
      validate: false
    }, async (context, request, response) => {
      var _cookie$credentials, _this$config$openid5;
      const cookie = await this.sessionStorageFactory.asScoped(request).get();
      let tokenFromExtraStorage = '';
      const extraAuthStorageOptions = this.getExtraAuthStorageOptions(context.security_plugin.logger);
      if (cookie !== null && cookie !== void 0 && (_cookie$credentials = cookie.credentials) !== null && _cookie$credentials !== void 0 && _cookie$credentials.authHeaderValueExtra) {
        tokenFromExtraStorage = (0, _cookie_splitter.getExtraAuthStorageValue)(request, extraAuthStorageOptions);
      }
      (0, _cookie_splitter.clearSplitCookies)(request, extraAuthStorageOptions);
      this.sessionStorageFactory.asScoped(request).clear();

      // authHeaderValue is the bearer header, e.g. "Bearer <auth_token>"
      const token = tokenFromExtraStorage.length ? tokenFromExtraStorage.split(' ')[1] : cookie === null || cookie === void 0 ? void 0 : cookie.credentials.authHeaderValue.split(' ')[1]; // get auth token
      const nextUrl = (0, _helper.getBaseRedirectUrl)(this.config, this.core, request);
      const logoutQueryParams = {
        post_logout_redirect_uri: `${nextUrl}`,
        id_token_hint: token
      };
      const endSessionUrl = (0, _helper.composeLogoutUrl)((_this$config$openid5 = this.config.openid) === null || _this$config$openid5 === void 0 ? void 0 : _this$config$openid5.logout_url, this.openIdAuthConfig.endSessionEndpoint, logoutQueryParams);
      return response.redirected({
        headers: {
          location: endSessionUrl
        }
      });
    });

    // captureUrlFragment is the first route that will be invoked in the SP initiated login.
    // This route will execute the captureUrlFragment.js script.
    this.core.http.resources.register({
      path: '/auth/openid/captureUrlFragment',
      validate: {
        query: _configSchema.schema.object({
          nextUrl: _configSchema.schema.maybe(_configSchema.schema.string({
            validate: _next_url.validateNextUrl
          }))
        })
      },
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      this.sessionStorageFactory.asScoped(request).clear();
      const serverBasePath = this.core.http.basePath.serverBasePath;
      return response.renderHtml({
        body: `
          <!DOCTYPE html>
          <title>OSD OIDC Capture</title>
          <link rel="icon" href="data:,">
          <script src="${serverBasePath}/auth/openid/captureUrlFragment.js"></script>
        `
      });
    });

    // This script will store the URL Hash in browser's local storage.
    this.core.http.resources.register({
      path: '/auth/openid/captureUrlFragment.js',
      validate: false,
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      this.sessionStorageFactory.asScoped(request).clear();
      return response.renderJs({
        body: `let oidcHash=window.location.hash.toString();
                let redirectHash = false;
                if (oidcHash !== "") {
                  window.localStorage.removeItem('oidcHash');
                  window.localStorage.setItem('oidcHash', oidcHash);
                    redirectHash = true;
                }
                let params = new URLSearchParams(window.location.search);
                let nextUrl = params.get("nextUrl");
                finalUrl = "login?redirectHash=" + encodeURIComponent(redirectHash);
                if (!!nextUrl) {
                  finalUrl += "&nextUrl=" + encodeURIComponent(nextUrl);
                }
                window.location.replace(finalUrl);
              `
      });
    });

    //  Once the User is authenticated the browser will be redirected to '/auth/openid/redirectUrlFragment'
    //  route, which will execute the redirectUrlFragment.js.
    this.core.http.resources.register({
      path: '/auth/openid/redirectUrlFragment',
      validate: {
        query: _configSchema.schema.object({
          nextUrl: _configSchema.schema.any()
        })
      },
      options: {
        authRequired: true
      }
    }, async (context, request, response) => {
      const serverBasePath = this.core.http.basePath.serverBasePath;
      return response.renderHtml({
        body: `
          <!DOCTYPE html>
          <title>OSD OpenID Success</title>
          <link rel="icon" href="data:,">
          <script src="${serverBasePath}/auth/openid/redirectUrlFragment.js"></script>
        `
      });
    });

    // This script will pop the Hash from local storage if it exists.
    // And forward the browser to the next url.
    this.core.http.resources.register({
      path: '/auth/openid/redirectUrlFragment.js',
      validate: false,
      options: {
        authRequired: true
      }
    }, async (context, request, response) => {
      return response.renderJs({
        body: `let oidcHash=window.localStorage.getItem('oidcHash');
                window.localStorage.removeItem('oidcHash');
                let params = new URLSearchParams(window.location.search);
                let nextUrl = params.get("nextUrl");
                finalUrl = nextUrl + oidcHash;
                window.location.replace(finalUrl);
              `
      });
    });
  }
}
exports.OpenIdAuthRoutes = OpenIdAuthRoutes;
_defineProperty(OpenIdAuthRoutes, "NONCE_LENGTH", 22);
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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